/*
 * #%L
 * Wikitty :: wikitty-solr-impl
 * 
 * $Id: WikittySearchEngineSolr.java 642 2010-12-22 03:40:56Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.1/wikitty-solr-impl/src/main/java/org/nuiton/wikitty/storage/solr/WikittySearchEngineSolr.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin POUSSIN
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty.storage.solr;

import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.embedded.EmbeddedSolrServer;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.core.CoreContainer;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.FacetTopic;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.FieldType.TYPE;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.storage.WikittyExtensionStorage;
import org.nuiton.wikitty.storage.WikittySearchEngine;
import org.nuiton.wikitty.services.WikittyTransaction;
import org.nuiton.wikitty.search.Search;

import java.io.File;
import java.util.Collections;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.wikitty.WikittyConfig;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.entities.WikittyTreeNodeHelper;

/**
 *
 * @author poussin
 * @version $Revision: 642 $
 *
 * Last update: $Date: 2010-12-22 04:40:56 +0100 (mer., 22 déc. 2010) $
 * by : $Author: bpoussin $
 */
public class WikittySearchEngineSolr implements WikittySearchEngine {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittySearchEngineSolr.class);

//    static final public String[] fieldNotToCopyPattern = {
//        Pattern.quote(TREENODE_PREFIX) + ".*"
//    };

    /** pattern to copy field from solr document to another solr document
     *  copy field s_c and s_t are not copied
     *  tree fields are not copied
     *  #all. fields are not copied
     */
    static final public String[] fieldToCopyPattern = {
         // match: id, extensions, not_null_fields
        SOLR_ID, SOLR_EXTENSIONS, SOLR_NOT_NULL_FIELDS,
        // match: ".*_bi" accepte ce qui fini par _bi
        ".*" + SUFFIX_BINARY,
        ".*" + SUFFIX_BOOLEAN,
        ".*" + SUFFIX_DATE,
        ".*" + SUFFIX_NUMERIC,
        ".*" + SUFFIX_STRING,
        ".*" + SUFFIX_WIKITTY
    };

    /**
     * NOTE: On ne pourra utiliser ces patterns pour la copie que lorsque la config
     * solr acceptera de creer des copyField avec des expressions regulieres
     * Ce qui permettra de genere les champs #all.* via la config solr et non
     * pas de devoir les ajouter via le code Java.
     * <copyField source="[^.]+\.(.*)"   dest="#all.#1"/>
     */
    static final public String[] fieldToCopyPatternWithExcludeAll = {
         // match: id, extensions, not_null_fields
        SOLR_ID, SOLR_EXTENSIONS, SOLR_NOT_NULL_FIELDS,
        // match: "(?!(all\.)).*_bi" accept ce qui fini par _bi sauf si ca commence par "all."
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_BINARY,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_BOOLEAN,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_DATE,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_NUMERIC,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_STRING,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_WIKITTY
    };

    /**
     * Helper to get information nodes and elements for reindexation.
     */
    static protected class ReindexChildTreeNode {

        protected SolrResource solrResource;
        protected SolrServer solrServer;
        
        protected Map<String, Collection<String>> includedNodeIds;
        protected Map<String, Collection<String>> excludedNodeIds;
        protected Map<String, String> parents;

        public ReindexChildTreeNode(SolrServer solrServer, SolrResource solrResource) {
            this.solrServer = solrServer;
            this.solrResource = solrResource;
            includedNodeIds = new HashMap<String, Collection<String>>();
            excludedNodeIds = new HashMap<String, Collection<String>>();
            parents = new HashMap<String, String>();
        }

        public void putIncludedAttachments(String nodeId, Collection<String> attchmentIds) {
            putAttachements(includedNodeIds, nodeId, attchmentIds);
        }

        public void putExcludedAttachments(String nodeId, Collection<String> attachmentIds) {
            putAttachements(excludedNodeIds, nodeId, attachmentIds);
        }

        public void putIncludedAttachment(String nodeId, String attachmentId) {
            putAttachment(includedNodeIds, nodeId, attachmentId);
        }

        public void putExcludedAttachment(String nodeId, String attachmentId) {
            putAttachment(excludedNodeIds, nodeId, attachmentId);
        }

        public Collection<String> getExcludedNodeIds(String attachmentId) {
            Collection<String> result = excludedNodeIds.get(attachmentId);
            if (result == null) {
                result = new HashSet<String>();
            }
            return result;
        }

        public Collection<String> getIncludedNodeIds(String attachmentId) {
            Collection<String> result = includedNodeIds.get(attachmentId);
            if (result == null) {
                result = new HashSet<String>();
            }
            return result;
        }

        protected void putAttachements(Map<String, Collection<String>> map, String nodeId, Collection<String> attachmentIds) {
            if (attachmentIds != null) {
                for (String attachmentId : attachmentIds) {
                    putAttachment(map, nodeId, attachmentId);
                }
            }
        }

        protected void putAttachment(Map<String, Collection<String>> map, String nodeId, String attachmentId) {
            Collection<String> values = map.get(attachmentId);
            if(values == null) {
                values = new HashSet<String>();
                map.put(attachmentId, values);
            }
            values.add(nodeId);
        }

        public void putParent(String nodeId, String parentId) {
            parents.put(nodeId, parentId);
        }

        public String getParent(String nodeId) {
            String parentId = parents.get(nodeId);

            // If not found in map, search in index
            if(parentId == null) {
                SolrDocument doc = SolrUtil.findById(solrServer, nodeId);
                if(doc == null) {
                    // is root
                    return null;
                }
                parentId = (String) doc.getFieldValue(WikittyTreeNode
                        .FQ_FIELD_WIKITTYTREENODE_PARENT + SUFFIX_WIKITTY);
                putParent(nodeId, parentId);
            }

            Collection<String> deletedDocIds = solrResource.getDeletedDocs();
            if(deletedDocIds.contains(parentId)) {
                return null;
            }
            return parentId;
        }

        public Collection<String> getReindexIds() {
            Collection<String> result = new HashSet<String>();
            result.addAll(includedNodeIds.keySet());
            result.addAll(excludedNodeIds.keySet());
            result.addAll(solrResource.getAddedDocIds());
            return result;
        }

        /**
         * Add in doc fields on association between nodes.
         *
         * For example if you have a element in node with parent, like this
         * A -> B -> C => element, the method add field in document solr :
         * TreeNode.root : A
         * TreeNode.A : B
         * TreeNode.B : C
         * TreeNode.C : TreeNode.empty
         * 
         * @throws SolrServerException 
         */
        public void reindex() throws SolrServerException {
            for (String id : getReindexIds()) {

                // Get documents
                SolrInputDocument doc = solrResource.getAddedDoc(id);
                if(doc == null) {
                    // Copy old field value
                    SolrDocument found = SolrUtil.findById(solrServer, id);
                    if (found != null) {
                        doc = SolrUtil.copySolrDocument(found, true, fieldToCopyPattern);
    	                solrResource.addDoc(id, doc);
    	            } else {

                        if (log.isWarnEnabled()) {
                            log.warn(String.format("Can't find wikitty id '%s'"
                                    + " in index. Skip this wikitty.", id));
                        }

                        // If not found just pass
                        continue;
                    }
                }

                // Add tree node fields
                Collection<String> includedChildNodeIds = getIncludedNodeIds(id);
                Collection<String> excludedChildNodeIds = getExcludedNodeIds(id);

                // Find all node contain is as attachment
                SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER
                        + WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_ATTACHMENT
                        + SUFFIX_WIKITTY + ":" + id);
                QueryResponse response = solrServer.query(query);
                SolrDocumentList updateDocs = response.getResults();

                for (Iterator<SolrDocument> iterator = updateDocs.iterator();
                        iterator.hasNext();) {
                    SolrDocument solrDocument = iterator.next();

                    String nodeId = (String) solrDocument.getFieldValue(SOLR_ID);
                    includedChildNodeIds.add(nodeId);
                }

                // Excluded nodes
                includedChildNodeIds.removeAll(excludedChildNodeIds);
                includedChildNodeIds.removeAll(solrResource.getDeletedDocs());

                // Add paths in doc
                Map<String, String> paths = new HashMap<String, String>();
                for (String nodeId : includedChildNodeIds) {
                    doc.addField(TREENODE_PREFIX + nodeId, TREENODE_EMPTY);

                    // Add path
                    String childParent = nodeId;
                    String parent = getParent(childParent);
                    while (parent != null) {
                        String parentPath = paths.get(childParent);
                        if(parentPath == null) {
                            doc.addField(TREENODE_PREFIX + parent, childParent);
                            paths.put(childParent, parent);
                        }

                        childParent = parent;
                        parent = getParent(childParent);
                    }

                    String parentPath = paths.get(childParent);
                    if(parentPath == null) {
                        doc.addField(TREENODE_ROOT, childParent);
                        paths.put(childParent, TREENODE_ROOT);
                    }
                }
            }
        }
    }

    /** solr server */
    protected SolrServer solrServer;

    /** Field modifier use to transform to solr format */
    protected TypeFieldModifier fieldModifier;

    /** JTA resource */
    protected SolrResource solrResource;

    /**
     * Init wikitty search engine on solr embedded server.
     * 
     * @param extensionStorage extension storage
     * @param properties properties (can be null)
     */
    public WikittySearchEngineSolr(
            ApplicationConfig config, WikittyExtensionStorage extensionStorage) {

        // init system env solr.data.dir
        if (config != null) {
            // choix du storage (file or Ram)
            String solrDirFactoryKey =
                    WikittyConfig.WikittyOption.WIKITTY_SEARCHENGINE_SOLR_DIRECTORY_FACTORY.getKey();
            String solrDirFactory = config.getOption(solrDirFactoryKey);
            if (solrDirFactory != null) {
                System.setProperty(solrDirFactoryKey, solrDirFactory);
            }

            // on utilise le directory que si on est pas en Ram
            if (solrDirFactory != null && !solrDirFactory.contains("RAMDirectoryFactory")) {
                String solrDataDirKey =
                        WikittyConfig.WikittyOption.WIKITTY_SEARCHENGINE_SOLR_DIRECTORY_DATA.getKey();
                String solrDataDir = config.getOption(solrDataDirKey);
                // make sure that dir exists
                if (solrDataDir != null) {
                    File file = new File(solrDataDir);
                    file.mkdirs();
                    System.setProperty(solrDataDirKey, solrDataDir);
                }
            }
        }

        try {
            CoreContainer.Initializer initializer = new CoreContainer.Initializer();
            CoreContainer coreContainer = initializer.initialize();
            solrServer = new EmbeddedSolrServer(coreContainer, "");

            fieldModifier = new TypeFieldModifier(extensionStorage);
            solrResource = new SolrResource(solrServer);
            
        } catch (Exception eee) {
            throw new WikittyException("SolR initialization error", eee);
        }
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        try {
            // FIXME poussin 20100618 pourquoi n'est pas fait dans la transaction ?
            solrResource.init();
            solrServer.deleteByQuery("*:*");
        } catch (Exception eee) {
            throw new WikittyException("Error during clearing SolR data", eee);
        }
    }

    @Override
    public void store(WikittyTransaction transaction, Collection<Wikitty> wikitties) {
        try {
            solrResource.init();
            ReindexChildTreeNode reindexChildTreeNode =
                new ReindexChildTreeNode(solrServer, solrResource);
            for (Wikitty w : wikitties) {
                String id = w.getId();

                if (w.hasExtension(WikittyTreeNode.EXT_WIKITTYTREENODE)) {

                    Set<String> attachments = WikittyTreeNodeHelper.getAttachment(w);
                    if (attachments == null) {
                        attachments = Collections.emptySet();
                    }

                    //
                    // recherche des anciennes attachments
                    //
                    Collection oldAttachments = null;
                    Set<String> oldAttachmentsCopy;

                    // Search deleted attachments
                    SolrDocument treeNodeDoc = SolrUtil.findById(solrServer, id);
                    if (treeNodeDoc != null) {
                        oldAttachments = treeNodeDoc.getFieldValues(
                                WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_ATTACHMENT + SUFFIX_WIKITTY);
                    }
                    // make copy to modify it
                    if (oldAttachments == null) {
                        oldAttachments = Collections.emptySet();
                    }
                    oldAttachmentsCopy = new HashSet<String>(oldAttachments);



                    // les attachments seulement dans old doivent etre exclus
                    oldAttachmentsCopy.removeAll(attachments);
                    reindexChildTreeNode.putExcludedAttachments(id, oldAttachmentsCopy);

                    if (w.getDirty().contains(WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT)
                            || null == WikittyTreeNodeHelper.getParent(w)
                            || WikittyUtil.versionGreaterThan("1", w.getVersion())) {
                        // si le pere a change
                        // ou qu'il est null (creation d'un nouvel arbre)
                        // ou que l'objet n'a jamais ete sauve 1 > version
                        // il faut indexer les attachments

                        // il faut reindexer tous les attachments
                        reindexChildTreeNode.putIncludedAttachments(id, attachments);

                        // Get new parent id (may be the same old parent)
                        String parentId = WikittyTreeNodeHelper.getParent(w);
                        reindexChildTreeNode.putParent(id, parentId);
                    } else {
                        // les attachments seulement dans les nouveaux doivent etre inclus
                        Set<String> attachmentsCopy = new HashSet<String>(attachments);
                        attachmentsCopy.removeAll(oldAttachments);
                        reindexChildTreeNode.putIncludedAttachments(id, attachmentsCopy);
                    }


                }

                // Index
                SolrInputDocument doc = createIndexDocument(w);
                solrResource.addDoc(id, doc);
            }

            // Reindex child in tree node
            reindexChildTreeNode.reindex();            
        } catch (Exception eee) {
            throw new WikittyException("Can't store wikitty", eee);
        }
    }

    @Override
    public void delete(WikittyTransaction transaction, Collection<String> ids) throws WikittyException {
        try {
            solrResource.init();
            ReindexChildTreeNode reindexChildTreeNode =
                new ReindexChildTreeNode(solrServer, solrResource);
            for (String id : ids) {

                // Find child in node id
                SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + TREENODE_PREFIX + id + ":*");
                QueryResponse response = solrServer.query(query);
                SolrDocumentList updateDocs = response.getResults();

                for (Iterator<SolrDocument> iterator = updateDocs.iterator(); iterator.hasNext();) {
                    SolrDocument solrDocument = iterator.next();
                    String childId = (String) solrDocument.getFieldValue(SOLR_ID);
                    reindexChildTreeNode.putExcludedAttachment(id, childId);
                }

                solrResource.deleteDoc(id);
            }

            // Reindex child in tree node
            reindexChildTreeNode.reindex();
        } catch (Exception eee) {
            throw new WikittyException("Can't delete wikitty in index", eee);
        }
    }

//    @Override
//    public void delete(Collection<String> idList) throws WikittyException {
//        try {
//            for (String id : idList) {
//                solrServer.deleteById(id);
//            }
//            solrServer.commit();
//        } catch (Exception eee) {
//            throw new WikittyException("Can't delete wikitty in index", eee);
//        }
//    }
    
    @Override
    public PagedResult<String> findAllByCriteria(WikittyTransaction transaction, Criteria criteria) {
        try {
            // Create query with restriction
            Restriction2Solr restriction2Solr = new Restriction2Solr(transaction, fieldModifier);
            String queryString = restriction2Solr.toSolr(criteria.getRestriction(), solrServer);
            SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + queryString);

            // Add paged
            int firstIndex = criteria.getFirstIndex();
            int endIndex = criteria.getEndIndex();

            query.setStart(firstIndex);
            int nbRows;
            if (endIndex == -1) {
                // WARNING It is necessary to substract 'start' otherwise,
                // there is a capacity overlow in solR
                nbRows = Integer.MAX_VALUE - firstIndex;
            } else {
                nbRows = endIndex - firstIndex + 1;
            }
            query.setRows(nbRows);

            // Add sorting
            List<String> sortAscending = criteria.getSortAscending();
            if(sortAscending != null) {
                for (String sort : sortAscending) {
                    String tranform = fieldModifier.convertToSolr(transaction, sort);
                    query.addSortField(tranform, SolrQuery.ORDER.asc);
                }
            }
            
            List<String> sortDescending = criteria.getSortDescending();
            if(sortDescending != null) {
                for (String sort : sortDescending) {
                    String tranform = fieldModifier.convertToSolr(transaction, sort);
                    query.addSortField(tranform, SolrQuery.ORDER.desc);
                }
            }

            // Add faceting
            List<String> facetField = criteria.getFacetField();
            log.debug("facetField : " + facetField);
            List<Criteria> facetCriteria = criteria.getFacetCriteria();

            // use to map query string to criteria facet name
            Map<String, String> facetQueryToName = new HashMap<String, String>();

            if ((facetField != null && !facetField.isEmpty())
                    || (facetCriteria != null && !facetCriteria.isEmpty())) {
                query.setFacet(true);
                query.setFacetMinCount(1);
                // query.setFacetLimit(8); // no limit actualy

                // field facetisation
                if (facetField != null) {
                    for (String fqfieldName : facetField) {
                        String tranform = fieldModifier.convertToSolr(transaction, fqfieldName);
                        query.addFacetField(tranform);
                    }
                }

                // query facetisation
                if (facetCriteria != null) {
                    for (Criteria facet : facetCriteria) {
                        String queryFacet =
                                restriction2Solr.toSolr(facet.getRestriction());
                        facetQueryToName.put(queryFacet, facet.getName());
                        query.addFacetQuery(queryFacet);
                    }
                }
            }

            QueryResponse resp = solrServer.query(query);
            SolrDocumentList solrResults = resp.getResults();

            Map<String, List<FacetTopic>> facets = new HashMap<String, List<FacetTopic>>();
            if (facetField != null && !facetField.isEmpty()) {
                for (FacetField facet : resp.getFacetFields()) {
                    String facetName = fieldModifier.convertToField(transaction, facet.getName());
                    List<FacetTopic> topics = new ArrayList<FacetTopic>();
                    if (facet.getValues() != null) {
                        for (FacetField.Count value : facet.getValues()) {
                            String topicName = value.getName();
                            if(!topicName.endsWith(TREENODE_EMPTY)) {
                                int topicCount = (int) value.getCount();
                                FacetTopic topic = new FacetTopic(facetName, topicName, topicCount);
                                topics.add(topic);
                            }
                        }
                    }
                    facets.put(facetName, topics);
                }
            }
            if (facetCriteria != null && !facetCriteria.isEmpty()) {
                for (Map.Entry<String, Integer> facet : resp.getFacetQuery().entrySet()) {
                    String facetName = facet.getKey();
                    // don't use contains because, map can have key with null value
                    if (null != facetQueryToName.get(facetName)) {
                        facetName = facetQueryToName.get(facetName);
                    }
                    Integer count = facet.getValue();
                    List<FacetTopic> topics = new ArrayList<FacetTopic>();
                    FacetTopic topic = new FacetTopic(facetName, facetName, count);
                    topics.add(topic);
                    facets.put(facetName, topics);
                }
            }

            List<String> ids = new ArrayList<String>(solrResults.size());
            for (SolrDocument doc : solrResults) {
                String id = (String) doc.getFieldValue(SOLR_ID);
                ids.add(id);
            }

            int numFound = (int)resp.getResults().getNumFound();
            PagedResult<String> result = new PagedResult<String>(
                    firstIndex, numFound, queryString, facets, ids);

            return result;
        } catch (SolrServerException eee) {
            throw new WikittyException("Error during find", eee);
        }
    }

    @Override
    public Integer findNodeCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
        String wikittyId = w.getId();

        String parent = WikittyTreeNodeHelper.getParent(w);
        if(parent == null) {
            parent = TREENODE_ROOT;
        } else {
            parent = TREENODE_PREFIX + parent;
        }

        Criteria criteria = Search.query(filter)
                .eq(parent, wikittyId).criteria()
                .setFirstIndex(0).setEndIndex(0);
        PagedResult<String> search = findAllByCriteria(transaction, criteria);

        int numFound = search.getNumFound();
        return numFound;
    }


    @Override
    public Map<String, Integer> findAllChildrenCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
        String wikittyId = w.getId();
        
        String parent = WikittyTreeNodeHelper.getParent(w);
        if(parent == null) {
            parent = TREENODE_ROOT;
        } else {
            parent = TREENODE_PREFIX + parent;
        }
        
        // Find count with facet, if the node not contain recurcively content,
        // the node not found with facet
        Criteria criteria = Search.query(filter).eq(parent, wikittyId).criteria()
                .setFirstIndex(0).setEndIndex(0)
                .addFacetField(TREENODE_PREFIX + wikittyId);
        PagedResult<String> search = findAllByCriteria(transaction, criteria);

        Map<String, Integer> counts = new HashMap<String, Integer>();
        List<FacetTopic> topics = search.getTopic(TREENODE_PREFIX + wikittyId);
        if(topics != null) {
            for (FacetTopic topic : topics) {
                String topicName = topic.getTopicName();
                int topicCount = topic.getCount();
                counts.put(topicName, topicCount);
            }
        }

        log.debug("Facet result " + counts);

        // Find all children, add the other node not found with facet
        criteria = Search.query().eq(WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT, wikittyId).criteria()
                .setFirstIndex(0).setEndIndex(Criteria.ALL_ELEMENTS);
        search = findAllByCriteria(transaction, criteria);

        List<String> children = search.getAll();
        for (String child : children) {
            if(!counts.containsKey(child)) {
                counts.put(child, 0);
            }
        }

        return counts;
    }

    /**
     * Create all index document to used to modify indexation.
     * this method don't modify index.
     * 
     * The document looks like :
     * SolrId : wikittyId
     * extensions : extensionNames
     * fieldName : fieldValue
     *
     * @param w all wikitties object to index
     * @return solrInputDocument used to modify index
     */
    protected SolrInputDocument createIndexDocument(Wikitty w) {
        if (log.isDebugEnabled()) {
            log.debug("index wikitty " + w.getId());
        }

        SolrInputDocument doc = new SolrInputDocument();
        String id = w.getId();
        doc.addField(SOLR_ID, id);

        for (String name : w.getExtensionNames()) {
            doc.addField(SOLR_EXTENSIONS, name);
        }

        for (String fqfieldName : w.fieldNames()) {
            FieldType fieldType = w.getFieldType(fqfieldName);
            TYPE type = fieldType.getType();
            String solrFqFieldName = SolrUtil.getSolrFieldName(fqfieldName, type);
            
            String solrAllFieldName = SOLR_ALL_EXTENSIONS
                    + WikittyUtil.FQ_FIELD_NAME_SEPARATOR
                    + WikittyUtil.getFieldNameFromFQFieldName(solrFqFieldName);
            
            Object objectValue = w.getFqField(fqfieldName);
            if(objectValue != null) {
                if (fieldType.isCollection()) {
                    Collection collectionValue = (Collection) objectValue;
                    for (Object itemValue : collectionValue) {
                        if (itemValue != null) {
                            doc.addField(solrFqFieldName, itemValue);
                            doc.addField(solrAllFieldName, itemValue);
//
//                            // Store string field in differents styles
//                            if(type == TYPE.STRING) {
//                                doc.addField(solrFqFieldName + SUFFIX_STRING_FULLTEXT, itemValue);
//                                doc.addField(solrAllFieldName + SUFFIX_STRING_FULLTEXT, itemValue);
//                                String itemValueLowerCase = itemValue.toString().toLowerCase();
//                                doc.addField(solrFqFieldName + SUFFIX_STRING_LOWERCASE, itemValueLowerCase);
//                                doc.addField(solrAllFieldName + SUFFIX_STRING_LOWERCASE, itemValueLowerCase);
//                            }
//
                            doc.addField(SOLR_NOT_NULL_FIELDS, fqfieldName);
                            if (log.isDebugEnabled()) {
                                log.debug("index field " + solrFqFieldName +
                                        " with value '" + itemValue + "'");
                            }
                        }
                    }
                } else {
                    doc.addField(solrFqFieldName, objectValue);
                    doc.addField(solrAllFieldName, objectValue);
//
//                    // Store string field in differents styles
//                    if(type == TYPE.STRING) {
//                        doc.addField(solrFqFieldName + SUFFIX_STRING_FULLTEXT, objectValue);
//                        doc.addField(solrAllFieldName + SUFFIX_STRING_FULLTEXT, objectValue);
//                        String objectValueLowerCase = objectValue.toString().toLowerCase();
//                        doc.addField(solrFqFieldName + SUFFIX_STRING_LOWERCASE, objectValueLowerCase);
//                        doc.addField(solrAllFieldName + SUFFIX_STRING_LOWERCASE, objectValueLowerCase);
//                    }
//
                    doc.addField(SOLR_NOT_NULL_FIELDS, fqfieldName);
                    if (log.isDebugEnabled()) {
                        log.debug("index field " + solrFqFieldName +
                                " with value '" + objectValue + "'");
                    }
                }
            }
        }
        return doc;
    }

}
