/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittySearchEngineInMemory.java 619 2010-12-17 16:33:52Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0/wikitty-api/src/main/java/org/nuiton/wikitty/storage/WikittySearchEngineInMemory.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.storage;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.FieldType.TYPE;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.operators.And;
import org.nuiton.wikitty.search.operators.BinaryOperator;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.search.operators.Restriction;
import org.nuiton.wikitty.services.WikittyTransaction;

public class WikittySearchEngineInMemory implements WikittySearchEngine {

    WikittyStorageInMemory wikittyStorage;

    public WikittySearchEngineInMemory(WikittyStorageInMemory wikittyStorage) {
        this.wikittyStorage = wikittyStorage;
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        // do nothing
    }

    @Override
    public void store(WikittyTransaction transaction, Collection<Wikitty> wikitties) {
    }

    @Override
    public void delete(WikittyTransaction transaction, Collection<String> idList) throws WikittyException {
    }

//    @Override
//    public void delete(Collection<String> idList) throws WikittyException {
//    }

    public boolean checkRestriction(Restriction restriction, Wikitty w) {
        if (restriction instanceof BinaryOperator) {
            BinaryOperator binOp = (BinaryOperator) restriction;
            if (binOp.getElement().getName().equals(Element.ELT_EXTENSION)) {
                return w.hasExtension(binOp.getValue());
            }
            String fqfieldName = binOp.getElement().getName();
            if (Element.ELT_EXTENSION.equals(fqfieldName)) {
                return true;
            } else if (Element.ELT_ID.equals(fqfieldName)) {
                return w.getId().equals(binOp.getValue());
            } // si les wikitty n'ont meme pas l'extension concerné
            // Le check restriction, ne doit pas tester les champs
            // si les wikitty n'ont meme pas l'extension concerné
            String[] extName = fqfieldName.split("\\.");
            if (!w.hasField(extName[0], extName[1])) {
                return false;
            }
            // recupere la valeur dans le wikitty
            Object o = w.getFqField(fqfieldName);
            // recupere le type de la valeur
            FieldType t = w.getFieldType(fqfieldName);
            // convertie la valeur a verifier dans le meme type que la valeur
            // du wikitty
            Object value = binOp.getValue();
            if ( !(value instanceof Collection) && t.isCollection()) {
                // on doit encapsuler dans une collection, car la creation
                // de la requete ajoute autant de v == o && ... que de valeurs
                // dans la collection (champs multi-value solr). Mais
                // dans le inmemory on doit retrouve des collections et non pas
                // des objets seuls :(
                value = Collections.singleton(value);
            }
            value = t.getValidValue(value);
            boolean checked = false;
            switch (restriction.getName()) {
                case EQUALS:
                    checked = value.equals(o);
                    break;
                case LESS:
                    checked = ((Comparable) o).compareTo(value) < 0;
                    break;
                case LESS_OR_EQUAL:
                    checked = ((Comparable) o).compareTo(value) <= 0;
                    break;
                case GREATER:
                    checked = ((Comparable) o).compareTo(value) > 0;
                    break;
                case GREATER_OR_EQUAL:
                    checked = ((Comparable) o).compareTo(value) >= 0;
                    break;
                case NOT_EQUALS:
                    checked = !value.equals(o);
                    break;
                case ENDS_WITH:
                    if (t.getType() != TYPE.STRING) {
                        throw new WikittyException("Can't search for contents that 'ends with' on attribute type different of String. " + "Attribute " + fqfieldName + " is " + t.getType().name());
                    }
                    checked = ((String) o).endsWith((String) value);
                    break;
                case STARTS_WITH:
                    if (t.getType() != TYPE.STRING) {
                        throw new WikittyException("Can't search for contents that 'starts with' on attribute type different of String. " + "Attribute " + fqfieldName + " is " + t.getType().name());
                    }
                    checked = ((String) o).startsWith((String) value);
                    break;
            }
            return checked;
        } else if (restriction instanceof And) {
            And and = (And) restriction;
            for (Restriction sub : and.getRestrictions()) {
                if (!checkRestriction(sub, w)) {
                    return false;
                }
            }
            return true;
        } else {
            throw new UnsupportedOperationException(restriction.getName() + " Search Not yet implemented");
        }
    }

    @Override
    public PagedResult<String> findAllByCriteria(WikittyTransaction transaction, Criteria criteria) {
        // throw new UnsupportedOperationException("Not supported yet.");
        int firstIndex = criteria.getFirstIndex();
        int endIndex = criteria.getEndIndex();
        List<String> ids = new LinkedList<String>();
        int currentIndex = 0;
        for (Entry<String, Wikitty> entry : wikittyStorage.getWikitties().entrySet()) {
            Wikitty w = entry.getValue();
            String id = entry.getKey();
            Restriction dto = criteria.getRestriction();
            if (!w.isDeleted() && checkRestriction(dto, w)) {
                currentIndex++;
                if (currentIndex > firstIndex) {
                    ids.add(id);
                }
                if (endIndex >= 0 && currentIndex >= endIndex) {
                    break;
                }
            }
        }
        return new PagedResult<String>(firstIndex, ids.size(), criteria.getRestriction().toString(), null, ids);
    }

    @Override
    public Integer findNodeCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public Map<String, Integer> findAllChildrenCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
        throw new UnsupportedOperationException("Not supported yet.");
    }
}
