/* *##%
 * Copyright (c) 2010 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Override some method of WikittyService to use cache
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyServiceCached implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceCached.class);

    /**
     * Utiliser pour mettre à jour le cache en ecoutant les evenements
     * sur le service.
     * 
     * Le service {@link #ws} doit supporter l'enregistrement de listener.
     */
    static public final String WIKITTY_CACHE_LISTENEVENTS_OPTION = "wikitty.service.cache.listenevents";

    /** Cache. */
    protected WikittyCache cache = null;

    /** Delegated wikitty service implementation. */
    protected WikittyService ws;

    /**
     * Default constructor.
     * 
     * @param ws delegate service
     */
    public WikittyServiceCached(WikittyService ws) {
        this(ws, null);
    }

    /**
     * Constructor with configuration.
     * 
     * @param ws delegate service
     * @param props properties (can be null)
     */
    public WikittyServiceCached(WikittyService ws, Properties props) {
        this.ws = ws;
        cache = new WikittyCache();
        registerWikittyServiceListener(props);
    }

    /**
     * Add cache as service listener if configuration request it.
     * 
     * @param props properties (can be null)
     */
    protected void registerWikittyServiceListener(Properties props) {

        if (props != null) {
            // add notifier as listener
            String listenEvents = props.getProperty(WIKITTY_CACHE_LISTENEVENTS_OPTION, "false");
            if ("true".equalsIgnoreCase(listenEvents)) {
                // add service listener for synchronisation
                // listener des remote event
                addWikittyServiceListener(cache, ServiceListenerType.REMOTE);
                
                if (log.isDebugEnabled()) {
                    log.debug("Listen remote event on service");
                }
            }
        }
    }

    @Override
    public String login(String login, String password) {
        return ws.login(login, password);
    }

    @Override
    public void logout(String securityToken) {
        ws.logout(securityToken);
    }

    @Override
    public boolean canWrite(String securityToken, Wikitty wikitty) {
        return ws.canWrite(securityToken, wikitty);
    }

    @Override
    public boolean canDelete(String securityToken, String wikittyId) {
        return ws.canDelete(securityToken, wikittyId);
    }

    @Override
    public boolean canRead(String securityToken, String wikittyId) {
        return ws.canRead(securityToken, wikittyId);
    }

    @Override
    public void clear(String securityToken) {
        ws.clear(securityToken);
        cache.clearWikitty();
    }

    /**
     * Override to refresh wikitty after put label
     * @param wikittyId
     * @param label
     */
    @Override
    public void addLabel(String securityToken, String wikittyId, String label) {
        ws.addLabel(securityToken, wikittyId, label);
        restore(securityToken, wikittyId);
    }

    /**
     * delete object in cache
     * @param id
     */
    @Override
    public void delete(String securityToken, String id) {
        ws.delete(securityToken, id);
        cache.removeWikitty(id);
    }

    /**
     * delete objets in cache
     * @param ids
     */
    @Override
    public void delete(String securityToken, Collection<String> ids) {
        ws.delete(securityToken, ids);
        cache.removeAllWikitty(ids);
    }

    /**
     * just wrap service method
     * 
     * @param wikittyId
     * @return
     */
    @Override
    public Set<String> findAllAppliedLabels(String securityToken, String wikittyId) {
        // if we want to add cache for this method, we must clear cache when
        // addLabel is called
        return ws.findAllAppliedLabels(securityToken, wikittyId);
    }

    /**
     * just wrap service method
     * 
     * @param criteria
     * @return
     */
    @Override
    public PagedResult<String> findAllByCriteria(
            String securityToken, Criteria criteria) {
        // if we want to add cache for this method, we must clear cache when
        // addLabel, store, storeExtension are called
        return ws.findAllByCriteria(securityToken, criteria);
    }

    /**
     * just wrap service method
     *
     * @param criteria
     * @return
     */
    @Override
    public PagedResult<String> findAllByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        // if we want to add cache for this method, we must use
        // transaction.getCache(), and we must clear cache when
        // addLabel, store, storeExtension are called
        return ws.findAllByCriteria(securityToken, transaction, criteria);
    }


    /**
     * just wrap service method
     *
     * @param label
     * @param firstIndex
     * @param endIndex
     * @return
     */
    @Override
    public PagedResult<String> findAllByLabel(String securityToken,
            String label, int firstIndex, int endIndex) {
        // if we want to add cache for this method, we must clear cache when
        // addLabel is called
        return ws.findAllByLabel(securityToken, label, firstIndex, endIndex);
    }

    /**
     * just wrap service method
     *
     * @param criteria
     * @return
     */
    @Override
    public Wikitty findByCriteria(String securityToken, Criteria criteria) {
        // if we want to add cache for this method, we must clear cache when
        // addLabel, store, storeExtension are called
        return ws.findByCriteria(securityToken, criteria);
    }

    /**
     * just wrap service method
     *
     * @param label
     * @return
     */
    @Override
    public Wikitty findByLabel(String securityToken, String label) {
        // if we want to add cache for this method, we must clear cache when
        // addLabel is called
        return ws.findByLabel(securityToken, label);
    }

    /**
     * just wrap service method
     *
     * @return
     */
    @Override
    public List<String> getAllExtensionIds(String securityToken) {
        // TODO poussin 20100412: perhaps use cache for extension ?
        return ws.getAllExtensionIds(securityToken);
    }

    /**
     * just wrap service method
     *
     * @param extensionName
     * @return
     */
    @Override
    public List<String> getAllExtensionsRequires(
            String securityToken, String extensionName) {
        // TODO poussin 20100412: perhaps use cache for extension ?
        return ws.getAllExtensionsRequires(securityToken, extensionName);
    }

    /**
     * just wrap service method
     *
     * @param id
     * @return
     */
    @Override
    public Wikitty restore(String securityToken, String id) {
        Wikitty result = cache.getWikitty(id);
        if (result == null) {
            if (log.isTraceEnabled()) {
                log.trace("Wikitty " + id + " not found in cache");
            }
            result = ws.restore(securityToken, id);
            cache.putWikitty(result);
        }
        else {
            if (log.isTraceEnabled()) {
                log.trace("Use cached wikitty " + id);
            }
        }
        return result;
    }

    /**
     * Overriden to put all restored object from server in cache.
     *
     * @param ids id to restore
     * @return restored wikitties
     */
    @Override
    public List<Wikitty> restore(String securityToken, List<String> ids) {
        ArrayList<String> notInCache = new ArrayList<String>();
        // linked to maintains the ordre
        LinkedHashMap<String, Wikitty> fromCache =
                new  LinkedHashMap<String, Wikitty>();
        for (String id : ids) {
            Wikitty w = cache.getWikitty(id);
            fromCache.put(id, w); // put all to maintains order
            if (w == null) { // if not found on cache, ask the server
                notInCache.add(id);
            }
        }

        // retrieve missing object
        List<Wikitty> missingInCache = ws.restore(securityToken, notInCache);

        cache.putAllWikitty(missingInCache);

        for (Wikitty w : missingInCache) {
            // add missing object
            fromCache.put(w.getId(), w);
        }

        Collection<Wikitty> tmp = fromCache.values();
        ArrayList<Wikitty> result = new ArrayList<Wikitty>(tmp);
        
        return result;
    }
    
    /**
     * Overriden to put all restored object from server in cache
     *
     * @param securityToken security token
     * @param transaction transaction to use
     * @param ids wikitty ids to restore
     * @return wikitty list
     */
    @Override
    public List<Wikitty> restore(String securityToken, WikittyTransaction transaction, List<String> ids) {
        ArrayList<String> notInCache = new ArrayList<String>();
        // linked to maintains the ordre
        LinkedHashMap<String, Wikitty> fromCache =
                new  LinkedHashMap<String, Wikitty>();
        for (String id : ids) {
            Wikitty w = cache.getWikitty(id);
            fromCache.put(id, w); // put all to maintains order
            if (w == null) { // if not found on cache, ask the server
                notInCache.add(id);
            }
        }

        // retrieve missing object
        List<Wikitty> missingInCache = ws.restore(securityToken, transaction, notInCache);

        cache.putAllWikitty(missingInCache);

        for (Wikitty w : missingInCache) {
            // add missing object
            fromCache.put(w.getId(), w);
        }

        Collection<Wikitty> tmp = fromCache.values();
        ArrayList<Wikitty> result = new ArrayList<Wikitty>(tmp);
        
        return result;
    }

    /**
     * just wrap service method
     *
     * @param wikittyId
     * @param filter
     * @return
     */
    @Override
    public Map<TreeNode, Integer> restoreChildren(
            String securityToken, String wikittyId, Criteria filter) {
        // FIXME lookup in cache, and put in cache
        return ws.restoreChildren(securityToken, wikittyId, filter);
    }

    /**
     * just wrap service method
     *
     * @param id
     * @return
     */
    @Override
    public WikittyExtension restoreExtension(String securityToken, String id) {
        // TODO poussin 20100412: perhaps use cache for extension ?
        return ws.restoreExtension(securityToken, id);
    }

    /**
     * just wrap service method
     *
     * @param name
     * @return
     */
    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name) {
        // TODO poussin 20100412: perhaps use cache for extension ?
        return ws.restoreExtensionLastVersion(securityToken, name);
    }
    
    /**
     * just wrap service method
     *
     * @param name
     * @return
     */
    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, WikittyTransaction transaction, String name) {
        // TODO poussin 20100412: perhaps use cache for extension ?
        // if use cache use transaction.getCache()
        return ws.restoreExtensionLastVersion(securityToken, transaction, name);
    }

    /**
     * just wrap service method
     *
     * @param wikittyId
     * @param filter
     * @return
     */
    @Override
    public Entry<TreeNode, Integer> restoreNode(
            String securityToken, String wikittyId, Criteria filter) {
        // FIXME lookup in cache, and put in cache
        return ws.restoreNode(securityToken, wikittyId, filter);
    }

    /**
     * just wrap service method
     *
     * @param wikittyId
     * @return
     */
    @Override
    public Tree restoreTree(String securityToken, String wikittyId) {
        // FIXME lookup in cache, and put in cache
        return ws.restoreTree(securityToken, wikittyId);
    }

    /**
     * Overriden to update wikitty and put it in cache
     * @param wikitty
     * @return
     */
    @Override
    public UpdateResponse store(String securityToken, Wikitty wikitty) {
        UpdateResponse result = ws.store(securityToken, wikitty);
        
        result.update(wikitty);
        cache.putWikitty(wikitty);
        
        return result;
    }

    /**
     * Overriden to put wikitty in cache
     *
     * @param wikitties
     * @return
     */
    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties) {
        UpdateResponse result = ws.store(securityToken, wikitties);

        for (Wikitty w : wikitties) {
            result.update(w);
        }
        cache.putAllWikitty(wikitties);

        return result;
    }

    /**
     * Overriden to put wikitty in cache
     *
     * @param wikitties
     * @param disableAutoVersionIncrement
     * @return
     */
    @Override
    public UpdateResponse store(String securityToken,
            Collection<Wikitty> wikitties, boolean disableAutoVersionIncrement) {
        UpdateResponse result = ws.store(securityToken, wikitties, disableAutoVersionIncrement);

        for (Wikitty w : wikitties) {
            result.update(w);
        }
        cache.putAllWikitty(wikitties);

        return result;
    }

    /**
     * Overriden to put wikitty in cache
     *
     * @param wikitties
     * @param disableAutoVersionIncrement
     * @return
     */
    @Override
    public UpdateResponse store(String securityToken, WikittyTransaction transaction,
            Collection<Wikitty> wikitties, boolean disableAutoVersionIncrement) {
        UpdateResponse result = ws.store(securityToken, transaction,
                wikitties, disableAutoVersionIncrement);

        for (Wikitty w : wikitties) {
            result.update(w);
        }
        transaction.getCache().putAllWikitty(wikitties);

        return result;
    }


    /**
     * just wrap service method
     *
     * @param ext
     * @return
     */
    @Override
    public UpdateResponse storeExtension(String securityToken, WikittyExtension ext) {
        // TODO poussin 20100412: perhaps use cache for extension ?
        return ws.storeExtension(securityToken, ext);
    }

    /**
     * just wrap service method
     *
     * @param exts
     * @return
     */
    @Override
    public UpdateResponse storeExtension(String securityToken, Collection<WikittyExtension> exts) {
        // TODO poussin 20100412: perhaps use cache for extension ?
        return ws.storeExtension(securityToken, exts);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken,
            WikittyTransaction transaction, Collection<WikittyExtension> exts) {
        // si on implante le cache pour cette methode, utiliser transaction.getCache()
        //throw new UnsupportedOperationException("Not supported yet.");
        return ws.storeExtension(securityToken, transaction, exts);
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken,
            WikittyTransaction transaction, String id) {
        // si on implante le cache pour cette methode, utiliser transaction.getCache()
        //throw new UnsupportedOperationException("Not supported yet.");
        return ws.restoreExtension(securityToken, transaction, id);
    }

    @Override
    public Wikitty restoreVersion(
            String securityToken, String wikittyId, String version) {
        // not put it in cache
        return ws.restoreVersion(securityToken, wikittyId, version);
    }

    //
    // Just delegate method
    //

    @Override
    public void addWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        ws.addWikittyServiceListener(listener, type);
    }

    @Override
    public void removeWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        ws.removeWikittyServiceListener(listener, type);
    }

    @Override
    public UpdateResponse syncEngin(String securityToken) {
        return ws.syncEngin(securityToken);
    }

}
