/* *##%
 * Copyright (c) 2009 Sharengo, Guillaume Dufrene, Benjamin POUSSIN.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

/**
 *
 * @author poussin
 * @version $Revision: 144 $
 *
 * Last update: $Date: 2010-06-22 19:03:49 +0200 (mar., 22 juin 2010) $
 * by : $Author: bpoussin $
 */
public class FieldType implements Serializable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -4375308750387837026L;

    /** tag/value use for unique */
    static public String UNIQUE = "unique";
    /** tag/value use for not null */
    static public String NOT_NULL = "notNull";
    
    static public enum TYPE {
        BOOLEAN, DATE, NUMERIC, STRING, WIKITTY;

        /**
         * convert string to TYPE, this method accept not trimed and not well
         * cased string (difference with valueOf)
         * @param name
         * @return TYPE else exception is throw
         */
        public static TYPE parse(String name) {
            TYPE result = valueOf(name.trim().toUpperCase());
            return result;
        }
    }

    public static final int NOLIMIT = Integer.MAX_VALUE;

    protected TYPE type;
    protected int lowerBound;
    protected int upperBound;

    /** used to store tag/value used by client side ex: editor=xhtml */
    Map<String, String> tagValues = new HashMap<String, String>();

    public FieldType() {
    }

    public FieldType(TYPE type, int lowerBound, int upperBound) {
        this.type = type;
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
    }

    public void addTagValue(String tag, String value) {
        tagValues.put(tag, value);
    }

    public String getTagValue(String tag) {
        String result = tagValues.get(tag);
        return result;
    }

    public Set<String> getTagNames() {
        return tagValues.keySet();
    }

    public Map<String, String> getTagValues() {
        return tagValues;
    }

    public void setTagValues(Map<String, String> tagValues) {
        this.tagValues = tagValues;
    }

    /**
     * Return true if this field have upperBound > 1.
     * 
     * @return {@code true} is field is collection
     */
    public boolean isCollection() {
        return upperBound > 1;
    }

    /**
     * Return string definition for this field.
     * 
     * @param name field name used for definition
     * @return field definition
     */
    public String toDefinition(String name) {
        String result = type + " " + name;
        if (lowerBound != 0 || upperBound != 0) {
            if (upperBound != NOLIMIT) {
                result += "[" + lowerBound + "-" + upperBound + "]";
            } else {
                result += "[" + lowerBound + "-*]";
            }
        }
        result += WikittyUtil.tagValuesToString(tagValues);
        return result;
    }

    /**
     * Convert value in argument in right type for this FieldType. Don't support
     * collection.
     *
     * @param value value to convert
     * @return object in type of this FieldType
     */
    protected Object getContainedValidObject( Object value ) {
        Object result = null;
        switch (type) {
        case DATE:
            result = WikittyUtil.toDate(value); break;
        case NUMERIC:
            result = WikittyUtil.toBigDecimal(value); break;
        case BOOLEAN:
            result = WikittyUtil.toBoolean(value); break;
        case STRING:
            result = WikittyUtil.toString(value); break;
        default:
            // if type is not found then type is business type
            // and is wikity object
            result = WikittyUtil.toWikitty(value); break;
        }
        return result;
    }

    /**
     * Return a valid value for this field.
     * 
     * @param value is casted if possible to an actual correct value.
     * @return value validity
     * @throws WikittyException if value can't be obtained
     */
    public Object getValidValue(Object value) throws WikittyException {
        if (value == null && isNotNull()) {
            throw new WikittyException("Value can't be null for this field");
        }

        Object result;
        if (value == null) {
            result = null;
        } else if (isCollection()) {
            if ( !(value instanceof Collection) ) {
                throw new WikittyException( "A collection is expected for type "
                        + type.name() + "[" + lowerBound + " - " + upperBound + "]" );
            }
            Collection<Object> col;
            if (isUnique()) {
                col = new LinkedHashSet<Object>();
            } else {
                col = new ArrayList<Object>();
            }

            // copy all value in new collections
            for ( Object o : (Collection<?>) value ) {
                col.add( getContainedValidObject(o) );
            }
            result = col;
        } else {
            result = getContainedValidObject(value);
        }
        return result;
    }

    /**
     * Test if value in argument is valid for this field type.
     * 
     * @param value to test
     * @return true if value is valid
     */
    public boolean isValidValue(Object value) {
        return getValidValue(value) != null;
    }

    public TYPE getType() {
        return type;
    }

    public int getLowerBound() {
        return lowerBound;
    }

    public int getUpperBound() {
        return upperBound;
    }

    public boolean isUnique() {
        return "true".equalsIgnoreCase(getTagValue(UNIQUE));
    }

    public boolean isNotNull() {
        return "true".equalsIgnoreCase(getTagValue(NOT_NULL));
    }

}
