/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags.swing;

import jaxx.CompilerException;
import jaxx.UnsupportedAttributeException;
import jaxx.compiler.JAXXCompiler;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.runtime.swing.Item;
import jaxx.tags.TagHandler;
import jaxx.types.TypeManager;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import java.io.IOException;
import java.util.List;

public class ItemHandler implements TagHandler {
    private String DATA_BINDING = "<data binding has not been processed yet>";

    public void compileFirstPass(Element tag, JAXXCompiler  compiler) throws CompilerException, IOException {
        compileChildrenFirstPass(tag, compiler);
    }

    public void compileSecondPass(Element tag, JAXXCompiler  compiler) throws CompilerException, IOException {
        String id = tag.getAttribute("id");
        if (id == null || id.length() == 0)
            id = compiler.getAutoId(ClassDescriptorLoader.getClassDescriptor(Item.class));
        String label = null;
        String value = null;
        boolean selected = false;
        NamedNodeMap children = tag.getAttributes();

        for (int i = 0; i < children.getLength(); i++) {
            Attr attribute = (Attr) children.item(i);
            String name = attribute.getName();
            String attrValue = attribute.getValue();
            if (name.equals("id")) {
                // already handled
                continue;
            }
            if (name.equals(Item.LABEL_PROPERTY)) {
                String labelBinding = compiler.processDataBindings(attrValue, ClassDescriptorLoader.getClassDescriptor(String.class));
                if (labelBinding != null)
                    compiler.registerDataBinding(labelBinding, id + ".label", id + ".setLabel(" + labelBinding + ");");
                else
                    label = attrValue;
                continue;
            }
            if (name.equals(Item.VALUE_PROPERTY)) {
                String valueBinding = compiler.processDataBindings(attrValue, ClassDescriptorLoader.getClassDescriptor(Object.class));
                if (valueBinding != null) {
                    value = DATA_BINDING;
                    compiler.registerDataBinding(valueBinding, id + ".value", id + ".setValue(" + valueBinding + ");");
                } else
                    value = attrValue;
                continue;
            }
            if (name.equals(Item.SELECTED_PROPERTY)) {
                String selectedBinding = compiler.processDataBindings(attrValue, ClassDescriptorLoader.getClassDescriptor(Boolean.class));
                if (selectedBinding != null)
                    compiler.registerDataBinding(selectedBinding, id + ".selected", id + ".setSelected(" + selectedBinding + ");");
                else
                    selected = (Boolean) TypeManager.convertFromString(attrValue, Boolean.class);
                continue;
            }

            if (!name.startsWith("xmlns") && !JAXXCompiler.JAXX_INTERNAL_NAMESPACE.equals(attribute.getNamespaceURI())) {
                throw new UnsupportedAttributeException(name);
            }
        }

        Item item = new Item(id, label, value, selected);
        CompiledItemContainer list = (CompiledItemContainer) compiler.getOpenComponent();
        if (value == null)
            compiler.reportError("<item> tag is missing required 'value' attribute");
        else {
            if (!value.equals(DATA_BINDING)) {
                List<Item> items = list.getItems();
                for (Item item1 : items) {
                    if (item1.getValue().equals(value)) {
                        compiler.reportError("This container already has an <item> tag with the value '" + value + "'");
                        break;
                    }
                }
            }
            list.openItem(item);
            compileChildrenSecondPass(tag, compiler);
            list.closeItem(item);
        }
    }


    protected void compileChildrenFirstPass(Element tag, JAXXCompiler  compiler) throws CompilerException, IOException {
        NodeList children = tag.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node node = children.item(i);
            int nodeType = node.getNodeType();
            if (nodeType == Node.ELEMENT_NODE) {
                Element child = (Element) node;
                compileChildTagFirstPass(child, compiler);
            } else
            if (nodeType == Node.TEXT_NODE || nodeType == Node.CDATA_SECTION_NODE) {
                String text = ((Text) node).getData().trim();
                if (text.length() > 0)
                    compiler.reportError("tag '" + tag.getLocalName() + "' may not contain text ('" + ((Text) node).getData().trim() + "')");
            }
        }
    }


    protected void compileChildrenSecondPass(Element tag, JAXXCompiler  compiler) throws CompilerException, IOException {
        NodeList children = tag.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node node = children.item(i);
            int nodeType = node.getNodeType();
            if (nodeType == Node.ELEMENT_NODE) {
                Element child = (Element) node;
                compileChildTagSecondPass(child, compiler);
            } else
            if (nodeType == Node.TEXT_NODE || nodeType == Node.CDATA_SECTION_NODE) {
                String text = ((Text) node).getData().trim();
                if (text.length() > 0)
                    compiler.reportError("tag '" + tag.getLocalName() + "' may not contain text ('" + ((Text) node).getData().trim() + "')");
            }
        }
    }


    protected void compileChildTagFirstPass(Element tag, JAXXCompiler  compiler) throws CompilerException, IOException {
        compiler.compileFirstPass(tag);
    }


    protected void compileChildTagSecondPass(Element tag, JAXXCompiler  compiler) throws CompilerException, IOException {
        compiler.compileSecondPass(tag);
    }
}