/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.components;

import java.awt.Color;
import java.awt.Font;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.MarkupWriter;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.dom.Element;
import org.apache.tapestry5.internal.services.ArrayEventContext;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.TypeCoercer;
import org.apache.tapestry5.services.Response;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.labels.StandardPieSectionLabelGenerator;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.plot.PiePlot3D;
import org.jfree.chart.plot.RingPlot;
import org.jfree.chart.title.TextTitle;
import org.jfree.data.DefaultKeyedValues;
import org.jfree.data.general.DefaultPieDataset;
import org.jfree.data.general.PieDataset;
import org.jfree.util.SortOrder;
import org.nuiton.util.ArrayUtil;

/**
 * Composant qui affiche un graphe de type PieChart.
 *
 * <pre>
 * Utilisation :
 * &lt;t:chart width=&quot;210&quot; height=&quot;190&quot; values=&quot;choice&quot; title=&quot;title&quot; type=&quot;1&quot;/&gt;
 * - choice est une liste de chaînes représentant les labels et leur valeurs respectives. ({&quot;val1&quot;, &quot;20&quot;, &quot;val2&quot;, &quot;10&quot;}...)
 * - type est le type de représentation du diagramme (1:PIE, 2:PIE3D...)
 * Pour généraliser à d'autres types de graphes il faudrait passer un JFreeChart :
 * Persist Parameter(required=true)
 * private JFreeChart _chart;
 * </pre>
 *
 * @author rannou
 * @version $Id: Chart.java 2836 2010-01-11 12:46:59Z schorlet $
 */
public class Chart {

    /** diagramme de type PIE */
    public static final int PIE = 1;
    /** diagramme de type PIE3D */
    public static final int PIE3D = 2;
    /** diagramme de type RING */
    public static final int RING = 3;

    /** titre du diagramme */
    @Parameter
    private String _title = "";

    /** type de représentation du diagramme */
    @Parameter(required = true)
    private int _type;

    @Parameter(required = true)
    private int _width;

    @Parameter(required = true)
    private int _height;

    /**
     * list(array) of paired values(label,value):
     * [String,Number,String,Number,...]
     */
    @Parameter(required = true)
    private List<Object> _values;

    @Parameter(required = false)
    private List<Object> _subtitles;

    @Inject
    private ComponentResources _resources;
    
    @Inject
    private TypeCoercer typeCoercer;
    
    void beginRender(MarkupWriter writer) {

        // event link params
        Object[] params = new Object[] { _title, _type, _width, _height };
        
        if (_subtitles != null) {
            Object[] size = new Object[] { _subtitles.size() };
            params = ArrayUtil.concat(params, size);
            params = ArrayUtil.concat(params, _subtitles.toArray());
        } else {
            Object[] size = new Object[] { 0 };
            params = ArrayUtil.concat(params, size);
        }
        params = ArrayUtil.concat(params, _values.toArray());
        
        // generate event link
        Link link = _resources.createEventLink("chart", params);
        Element img = writer.element("img", "src", link);

        _resources.renderInformalParameters(writer);
    }

    void afterRender(MarkupWriter writer) {
        writer.end();
    }

    public StreamResponse onChart(final String title, final int type, final int width,
            final int height, final int nbSubtitles, final Object... objects) {
        
        ArrayEventContext ec = new ArrayEventContext(typeCoercer, objects);
        List<Object> subtitles = new ArrayList<Object>(nbSubtitles);
        for (int i = 5; i < 5 + nbSubtitles; i += 2) {
            String key = ec.get(String.class, i);
            String value = ec.get(String.class, i + 1);
            subtitles.add(key);
            subtitles.add(value);
        }
        
        DefaultKeyedValues values = new DefaultKeyedValues();
        for (int i = 5 + nbSubtitles; i < ec.getCount(); i += 2) {
            String key = ec.get(String.class, i);
            Double value = ec.get(Double.class, i + 1);
            values.addValue(key, value);
        }
        
        values.sortByValues(SortOrder.DESCENDING);
        PieDataset data = new DefaultPieDataset(values);

        // create the chart look according to the type
        PiePlot plot = null;
        switch (type) {
        case PIE:
            plot = new PiePlot(data);
            break;
        case PIE3D:
            plot = new PiePlot3D(data);
            break;
        case RING:
            plot = new RingPlot(data);
            break;
        default:
            plot = new PiePlot(data);
            break;
        }

        //plot.setForegroundAlpha(0.7f);
        plot.setCircular(true);
        plot.setLabelGenerator(new StandardPieSectionLabelGenerator("{0} {2}"));
        plot.setLegendLabelGenerator(new StandardPieSectionLabelGenerator(
                "{0} = {1}"));
        plot.setNoDataMessage("No Data");

        final JFreeChart chart = new JFreeChart(plot);
        chart.setTitle(new TextTitle(title, new Font(Font.SANS_SERIF,
                Font.BOLD, 12)));
        chart.setBackgroundPaint(new Color(255, 255, 255));

        if (subtitles != null) {
            Font subtitlesFont = new Font(Font.SANS_SERIF, Font.ITALIC, 11);
            for (int i = 0; i < subtitles.size(); i += 2) {
                String subtitle = subtitles.get(i) + "=" + subtitles.get(i + 1);
                TextTitle textTitle = new TextTitle(subtitle, subtitlesFont);
                chart.addSubtitle(textTitle);
            }
        }
        
        // return the image
        return new StreamResponse() {
            public String getContentType() {
                return "image/png";
            }

            public InputStream getStream() throws IOException {
                BufferedImage image = chart.createBufferedImage(width, height);
                ByteArrayOutputStream byteArray = new ByteArrayOutputStream();
                ChartUtilities.writeBufferedImageAsPNG(byteArray, image);
                return new ByteArrayInputStream(byteArray.toByteArray());
            }

            public void prepareResponse(Response response) {
            }
        };
    }
}