/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.components;

import org.chorem.pollen.ui.base.ContextLink;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import javax.activation.MimetypesFileTypeMap;

import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.MarkupWriter;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.corelib.base.AbstractLink;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.Response;
import org.slf4j.Logger;

/**
 * Composant qui affiche un lien vers un fichier dynamique.
 *
 * Ce composant permet d'afficher un lien vers un fichier. Le fichier est traité
 * comme un flux affiché par le navigateur lorqu'on clique sur le lien.
 *
 * <pre>
 * Utilisation :
 * &lt;t:filelink src=&quot;/feeds/feed.xml&quot; type=&quot;literal:application/atom+xml&quot; /&gt;
 * </pre>
 *
 * @author rannou
 * @version $Id: FileLink.java 2791 2009-11-05 13:58:23Z echatellier $
 */
public class FileLink extends AbstractLink {

    /** Chemin du fichier */
    @Parameter(required = true)
    private String _filename;

    /** Type MIME du fichier */
    @Parameter
    private String _type;

    @Inject
    private Logger logger;

    @Inject
    private ComponentResources _resources;

    /**
     * Context to get the file
     */
    @Parameter(required=true)
    private ContextLink _context;

    @Log
    void beginRender(MarkupWriter writer) {
        Link link = _resources.createEventLink("createLink", _filename, _type);
        writeLink(writer, link);
    }

    void afterRender(MarkupWriter writer) {
        writer.end();
    }

    /** Affichage du fichier */
    public StreamResponse onCreateLink(final String filename, String type) {

        // security : src containing .. are filtered, to not access to full system
        if (StringUtils.isEmpty(filename) || filename.contains("..")) {
            return null;
        }

        //File file = new File(/*_context.getContextPath() + */filename/* + _context.getExtension()*/);
        File file = _context.getFile(filename);
        FileInputStream stream = null;
        try {
            stream = new FileInputStream(file);
        } catch (FileNotFoundException e) {
            logger.error("Fichier inexistant : " + file.getAbsolutePath(), e);
        }

        if (!_resources.isBound("type")) {
            MimetypesFileTypeMap mimes = new MimetypesFileTypeMap();
            type = mimes.getContentType(file);
        }

        return new FileStreamResponse(stream, type);
    }

    /** Création du flux sur le fichier */
    class FileStreamResponse implements StreamResponse {
        protected String contentType;
        protected InputStream stream;

        public FileStreamResponse(InputStream stream, String contentType) {
            this.stream = stream;
            this.contentType = contentType;
        }

        public String getContentType() {
            return contentType;
        }

        public InputStream getStream() throws IOException {
            return stream;
        }

        public void prepareResponse(Response arg0) {

        }
    }
}