/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.poll;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.fileupload.FileUploadException;
import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.ValidationException;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.json.JSONObject;
import org.chenillekit.tapestry.core.components.DateTimeField;
import org.chorem.pollen.business.ServiceList;
import org.chorem.pollen.business.ServicePoll;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PersonListDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.utils.MD5;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.ui.data.GenericSelectModel;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.data.PollStep;
import org.chorem.pollen.ui.data.uio.DateChoiceUIO;
import org.chorem.pollen.ui.data.uio.ImageChoiceUIO;
import org.chorem.pollen.ui.services.Configuration;
import org.chorem.pollen.ui.utils.ImageUtil;
import org.slf4j.Logger;

/**
 * Classe de la page de création d'un sondage.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: PollCreation.java 2630 2009-07-08 15:23:00Z nrannou $
 */
@IncludeStylesheet("context:css/pollcreation.css")
public class PollCreation {

    /** Étape courante du formulaire */
    @Persist
    private PollStep step;

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SessionState
    private UserDTO user;
    @Property
    private boolean userExists;

    /**
     * Objet de session représentant l'url du site.
     */
    @SessionState
    @Property
    private String siteURL;

    /** Composants DateTimeField pour le début des ajouts de choix */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField beginChoiceDate;

    /** Composants DateTimeField pour le début et la fin du sondage */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField beginDate;
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField endDate;

    /** Composant DateTimeField pour les choix du sondage */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField dateDTF;

    /** Date de début du sondage (utilisée pour la validation). */
    private Date beginDateValidation;

    @InjectComponent
    private Zone pollCreationZone;

    /**
     * PollDTO créé par le formulaire
     */
    @Property
    @Persist
    private PollDTO poll;

    /** variable utilisée pour déterminer si le bouton addPerson a été cliqué */
    private boolean addPersonSelected;

    /** variable utilisée pour déterminer si le bouton addPerson a été cliqué */
    @Persist
    private boolean addChoiceSelected;

    /** variable utilisée pour l'affichage de la choiceNbCheckBox */
    @Property
    @Persist
    private boolean choiceNbCheckBox;

    /** variable utilisée pour l'affichage de la notificationCheckBox */
    @Property
    @Persist
    private boolean notificationCheckBox;

    /** variable utilisée pour l'affichage de la reminderCheckBox */
    @Property
    @Persist
    private boolean reminderCheckBox;

    /** Règle de notification de vote */
    @Property
    @Persist
    private PreventRuleDTO notificationPreventRule;

    /** Règle de notification pour le rappel des votants */
    @Property
    @Persist
    private PreventRuleDTO reminderPreventRule;

    @SuppressWarnings("unused")
    @Property
    @Persist
    private PollAccountDTO pollAccount;

    /**
     * Nombre de votants affichés initialement
     */
    @Property
    private int nbVotingListPersons = 5;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des votants
     */
    @SuppressWarnings("unused")
    @Property
    private PollAccountDTO votingListPerson;

    /**
     * Liste des votants à créer
     */
    @Property
    @Persist
    private List<PollAccountDTO> votingListPersons;

    /**
     * Listes de favoris de l'utilisateur.
     */
    @Property
    @Persist
    private GenericSelectModel<PersonListDTO> personLists;

    /**
     * Liste de favoris sélectionnée.
     */
    @Property
    @Persist
    private PersonListDTO personList;

    /**
     * Nombre de choix affichés initialement
     */
    @Property
    private int nbChoices = 5;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des choix
     */
    @SuppressWarnings("unused")
    @Property
    private ChoiceDTO choice;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des choix
     */
    @SuppressWarnings("unused")
    @Property
    private DateChoiceUIO dateTypeChoice;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des choix
     */
    @SuppressWarnings("unused")
    @Property
    private ImageChoiceUIO imgTypeChoice;

    /**
     * Liste des choix à créer
     */
    @Property
    @Persist
    private List<ChoiceDTO> choices;

    /**
     * Liste des choix à créer
     */
    @Property
    @Persist
    private List<DateChoiceUIO> dateTypeChoices;

    /**
     * Liste des choix à créer
     */
    @Property
    @Persist
    private List<ImageChoiceUIO> imgTypeChoices;

    /**
     * Formulaire de création de sondage
     */
    @Component(id = "pollCreationForm")
    private Form pollCreationForm;
    @Component(id = "choicesCreationForm")
    private Form choicesCreationForm;

    @InjectPage
    private CreationValidation creationValidation;

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    @Inject
    private PropertyAccess _propertyAccess;

    @Inject
    private ComponentResources resources;

    @Inject
    private Messages messages;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    @Inject
    private Logger logger;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServiceList serviceList;

    /**
     * Méthode appelée lorsqu'on souhaite accéder à l'étape suivante de la
     * création de sondage.
     */
    Object onSuccessFromPollCreationForm() {
        switch (step) {
        case POLL:
            step = PollStep.OPTIONS;
            break;
        case OPTIONS:
            if (poll.isAnonymous()) {
                poll.setAnonymousVoteAllowed(true);
            }
            if (poll.isContinuousResults()) {
                poll.setPublicResults(true);
            }
            
            if (isFreePoll()) {
                step = PollStep.CHOICES;
            } else {
                step = PollStep.LISTS;
            }
            break;
        case LISTS:
            if (!addPersonSelected) {
                step = PollStep.CHOICES;
            }
            break;
        case CHOICES:
            break;
        default:
            step = PollStep.POLL;
            break;
        }
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée que le formulaire soit valide ou non. Il est nécessaire
     * de la redéfinir pour qu'en cas d'erreur de validation, la zone soit tout
     * de même mise à jour pour afficher l'erreur.
     */
    Object onSubmitFromPollCreationForm() {
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée lorsqu'on souhaite valider la création du sondage.
     */
    Object onSuccessFromChoicesCreationForm() {
        if (!addChoiceSelected) {
            
            // Ajout de la liste restreinte au sondage
            if (!isFreePoll()) {
                VotingListDTO restrictedList = new VotingListDTO();
                for (PollAccountDTO person : votingListPersons) {
                    if (person.getVotingId() != null) {
                        restrictedList.getPollAccountDTOs().add(person);
                    }
                }
                poll.getVotingListDTOs().add(restrictedList);
            }

            // Ajout des choix au sondage
            if (isTextChoices()) {
                for (ChoiceDTO choice : choices)
                    if (choice.getName() != null) {
                        choice.setValidate(true);
                        poll.getChoiceDTOs().add(choice);
                    }
            } else if (isDateChoices()) {
                for (DateChoiceUIO choice : dateTypeChoices)
                    if (choice.getDate() != null) {
                        choice.setValidate(true);
                        choice.setName(String.valueOf(choice.getDate()
                                .getTime()));
                        poll.getChoiceDTOs().add(choice);
                    }
            } else if (isImgChoices()) {
                for (ImageChoiceUIO imgChoice : imgTypeChoices) {
                    if (imgChoice.getImg() != null) {
                        System.out.println(imgChoice.getImg().getFileName());
                        if (imgChoice.getImg().getContentType().contains(
                                "image")
                                || imgChoice.getImg().getContentType()
                                        .contains("IMAGE")) {
                            imgChoice.setValidate(true);
                            imgChoice.setName(imgChoice.getImg().getFileName()
                                    .replace(' ', '_'));
                            poll.getChoiceDTOs().add(imgChoice);
                        } else {
                            return this;
                        }
                    }
                }
            }

            // Mise à jour du sondage
            updatePoll();

            creationValidation.setPoll(poll);
            return creationValidation;
        }

        return this;
    }

    /**
     * Méthode appelée lorsqu'on souhaite accéder à l'étape précédente de la
     * création de sondage.
     */
    Object onPrevious() {
        switch (step) {
        case OPTIONS:
            step = PollStep.POLL;
            break;
        case LISTS:
            step = PollStep.OPTIONS;
            break;
        case CHOICES:
            if (isFreePoll()) {
                step = PollStep.OPTIONS;
            } else {
                step = PollStep.LISTS;
            }
            break;
        default:
            step = PollStep.POLL;
            break;
        }
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * beginDate.
     *
     * @throws ValidationException
     */
    void onValidateFromBeginDate(Date value) throws ValidationException {
        beginDateValidation = value;
        if (value != null && value.before(new Date())) {
            throw new ValidationException(messages.get("beginDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * endDate.
     *
     * @throws ValidationException
     */
    void onValidateFromEndDate(Date value) throws ValidationException {
        if (beginDateValidation == null) {
            beginDateValidation = new Date();
        }

        if (value != null && value.before(beginDateValidation)) {
            throw new ValidationException(messages.get("endDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * beginChoiceDate.
     *
     * @throws ValidationException
     */
    void onValidateFromBeginChoiceDate(Date value) throws ValidationException {
        if (beginDateValidation == null) {
            beginDateValidation = new Date();
        }

        if (value != null && value.after(beginDateValidation)) {
            throw new ValidationException(messages
                    .get("beginChoiceDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation des
     * listes de votants.
     *
     * @throws ValidationException
     */
    void onValidateFormFromPollCreationForm() throws ValidationException {
        int nbEqual = 0;
        int nbNotNull = 0;

        // Validation des votants
        if (step == PollStep.LISTS) {

            // Repérage des doublons
            for (PollAccountDTO account1 : votingListPersons) {
                if (account1.getVotingId() != null) {
                    nbNotNull++;
                    for (PollAccountDTO account2 : votingListPersons) {
                        if (account1.getVotingId().equals(
                                account2.getVotingId())) {
                            nbEqual++;
                        }
                    }
                }
            }

            if (nbEqual > nbNotNull) {
                throw new ValidationException(messages.get("list-validate"));
            }
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation de la
     * liste de choix.
     *
     * @throws ValidationException
     */
    void onValidateFormFromChoicesCreationForm() throws ValidationException {
        int nbEqual = 0;
        int nbNotNull = 0;

        // Validation des choix
        if (step == PollStep.CHOICES) {

            // Repérage des doublons (type Texte)
            if (isTextChoices()) {
                for (ChoiceDTO choice1 : choices) {
                    if (choice1.getName() != null) {
                        nbNotNull++;
                        for (ChoiceDTO choice2 : choices) {
                            if (choice2.getName() != null) {
                                if (choice1.getName().equals(choice2.getName())) {
                                    nbEqual++;
                                }
                            }
                        }
                    }
                }
                // Repérage des doublons (type Date)
            } else if (isDateChoices()) {
                for (DateChoiceUIO choice1 : dateTypeChoices) {
                    if (choice1.getDate() != null) {
                        nbNotNull++;
                        for (DateChoiceUIO choice2 : dateTypeChoices) {
                            if (choice2.getDate() != null) {
                                if (choice1.getDate().equals(choice2.getDate())) {
                                    nbEqual++;
                                }
                            }
                        }
                    }
                }
                // Repérage des doublons (type Image)
            } else if (isImgChoices()) {
                for (ImageChoiceUIO choice1 : imgTypeChoices) {
                    if (choice1.getImg() != null) {
                        nbNotNull++;
                        for (ImageChoiceUIO choice2 : imgTypeChoices) {
                            if (choice2.getImg() != null) {
                                if (choice1.getImg().getFileName().equals(
                                        choice2.getImg().getFileName())) {
                                    nbEqual++;
                                }
                            }
                        }
                    }
                }
            }

            if (nbEqual > nbNotNull) {
                throw new ValidationException(messages.get("choice-validate"));
            }
        }
    }

    /** Echec lors du téléchargement */
    Object onUploadException(FileUploadException e) {
        pollCreationForm.recordError("Upload exception: " + e.getMessage());
        return this;
    }

    /**
     * Méthode de mise à jour du sondage.
     *
     * @return vrai si le sondage a bien été mise à jour.
     */
    private boolean updatePoll() {
        if (!poll.getChoiceDTOs().isEmpty()) {
            if (userExists) {
                if (logger.isDebugEnabled()) {
                    logger.debug("user.getId() : " + user.getId());
                }
                poll.setUserId(user.getId());
            }
            for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                choice.setValidate(true);
            }

            // Règles de notification
            if (notificationCheckBox) {
                poll.getPreventRuleDTOs().add(notificationPreventRule);
            }
            if (reminderCheckBox) {
                poll.getPreventRuleDTOs().add(reminderPreventRule);
            }

            // Date de début du sondage
            if (poll.getBeginDate() == null) {
                poll.setBeginDate(new Date());
            }

            poll.setPollId(UUID.randomUUID().toString().replaceAll("-", ""));
            poll.setId(servicePoll.createPoll(poll));
            if (poll.isAnonymous()) {
                poll.setAnonymousVoteAllowed(true);
            }
            if (poll.isContinuousResults()) {
                poll.setPublicResults(true);
            }
            if (poll.getMaxChoiceNb() < 1
                    || poll.getMaxChoiceNb() > poll.getChoiceDTOs().size()) {
                poll.setMaxChoiceNb(poll.getChoiceDTOs().size());
            }
            if (servicePoll.updatePoll(poll)) {
                if (poll.getChoiceType() == ChoiceType.IMAGE) {
                    ImageUtil.saveImages(imgTypeChoices, new File(conf
                            .getProperty("upImgDir"), poll.getPollId()));
                }
                poll = servicePoll.findPollById(poll.getId());

                sendMailNotification();
                return true;
            } else
                return false;
        } else
            return false;
    }

    /** Envoi du mail de notification */
    private void sendMailNotification() {
        String voteURL = siteURL + "poll/VoteFor/" + poll.getPollId();
        String modifURL = siteURL + "poll/Modification/" + poll.getPollId()
                + ":" + MD5.encode(poll.getCreatorId());
        Map<String, String> data = new HashMap<String, String>();
        data.put("host", conf.getProperty("email_host"));
        data.put("port", conf.getProperty("email_port"));
        data.put("from", conf.getProperty("email_from"));

        // Mail au créateur
        if (poll.getCreatorEmail() != null) {
            data.put("to", poll.getCreatorEmail());
            data.put("title", messages.format("creatorEmail_subject", poll
                    .getTitle()));
            data.put("msg", messages.format("creatorEmail_msg",
                    poll.getTitle(), voteURL, modifURL));

            PreventRuleManager.emailAction(data);
        }

        // Mails aux votants
        for (VotingListDTO list : poll.getVotingListDTOs()) {
            for (PollAccountDTO account : list.getPollAccountDTOs()) {
                if (account.getEmail() != null) {
                    data.put("to", account.getEmail());
                    data.put("title", messages.format("votingEmail_subject",
                            poll.getTitle()));
                    data.put("msg", messages.format("votingEmail_msg", poll
                            .getTitle(), account.getVotingId(), voteURL));

                    PreventRuleManager.emailAction(data);
                }
            }
        }
    }

    public String getChoiceDateDisplay() {
        return poll.isChoiceAddAllowed() ? "display: block;" : "display: none;";
    }

    public String getChoiceNbDisplay() {
        return choiceNbCheckBox ? "display: block;" : "display: none;";
    }

    public String getNotificationDisplay() {
        return notificationCheckBox ? "display: block;" : "display: none;";
    }

    public String getReminderDisplay() {
        return reminderCheckBox ? "display: block;" : "display: none;";
    }

    /** Retourne la chaîne correspondant à l'étape courante */
    public String getStepLegend() {
        Integer index = step.getIndex();

        // corrections selon le type de sondage
        if (isFreePoll()) {
            if (step == PollStep.CHOICES) {
                index--;
            }
        }

        // mise en forme du message
        switch (step) {
        case POLL:
            return messages.format("pollLegend", index);
        case OPTIONS:
            return messages.format("optionsLegend", index);
        case LISTS:
            return messages.format("listsLegend", index);
        case CHOICES:
            return messages.format("choicesLegend", index);
        default:
            return "";
        }
    }

    public boolean isInPoll() {
        return step == PollStep.POLL;
    }

    public boolean isInLists() {
        return step == PollStep.LISTS;
    }

    public boolean isInChoices() {
        return step == PollStep.CHOICES;
    }

    public boolean isInOptions() {
        return step == PollStep.OPTIONS;
    }

    public boolean isNormalVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.NORMAL;
    }

    public boolean isPercentageVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.PERCENTAGE;
    }

    public boolean isCondorcetVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.CONDORCET;
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextChoices() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateChoices() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImgChoices() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    /**
     * Méthode appelée à la sélection d'une liste de votants. Le mixin
     * ck/OnEvent ne permet pas de retourner le contenu d'une zone. Il faut donc
     * passer par une fonction JavaScript pour activer un event link. Un lien
     * est créé. Il sera retourné à la fonction JavaScript onCompleteCallback
     * pour mettre à jour la zone.
     *
     * @return un JSONObject contenant l'url de l'évènement mettant à jour la
     *         zone.
     */
    public JSONObject onChangeFromListSelect(String value) {
        if (!"".equals(value)) {
            personList = serviceList.findPersonListById(value);
            
            // Copie des personnes de la liste de favoris dans la liste de votants
            for (PollAccountDTO account : personList.getPollAccountDTOs()) {
                account.setId("");
                account.setPersonListId("");
            }
            votingListPersons = personList.getPollAccountDTOs();
        }
        return createParamsForCallback();
    }

    /**
     * Méthode appelée à la sélection d'un type de choix. Le mixin ck/OnEvent ne
     * permet pas de retourner le contenu d'une zone. Il faut donc passer par
     * une fonction JavaScript pour activer un event link. Un lien est créé. Il
     * sera retourné à la fonction JavaScript onCompleteCallback pour mettre à
     * jour la zone.
     *
     * @return un JSONObject contenant l'url de l'évènement mettant à jour la
     *         zone.
     */
    public JSONObject onChangeFromChoiceType(String value) {
        poll.setChoiceType(ChoiceType.valueOf(value));
        return createParamsForCallback();
    }

    /**
     * Création d'un JSONObject contenant un identifiant de zone et une url pour
     * un évènement.
     *
     * @return un JSONObject contenant un zoneId et une url.
     */
    private JSONObject createParamsForCallback() {
        JSONObject json = new JSONObject();
        Link link = resources.createEventLink("updatePollCreationZone");
        json.put("link", link.toAbsoluteURI());
        json.put("zoneId", "pollCreationZone");
        return json;
    }

    /**
     * Méthode appelée par le callback JavaScript pour mettre à jour la zone.
     *
     * @return le contenu mis à jour de la zone.
     */
    public Object onUpdatePollCreationZone() {
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée pour l'ajout d'une ligne supplémentaire dans le
     * formulaire des listes de votants.
     */
    void onSelectedFromAddPerson() {
        votingListPersons.add(new PollAccountDTO());
        addPersonSelected = true;
    }

    /**
     * Méthode appelée pour l'ajout d'une ligne supplémentaire dans le
     * formulaire des choix.
     */
    void onSelectedFromAddChoice() {
        if (poll.getChoiceType() == ChoiceType.DATE) {
            dateTypeChoices.add(new DateChoiceUIO());
        } else if (poll.getChoiceType() == ChoiceType.IMAGE) {
            imgTypeChoices.add(new ImageChoiceUIO());
        } else {
            choices.add(new ChoiceDTO());
        }
        addChoiceSelected = true;
    }

    /** Retourne vrai si des listes de favoris existent */
    public boolean isPersonListsExists() {
        return personLists != null && !personLists.getList().isEmpty();
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };

        // on réinitialise le sondage sauf pour l'ajout de choix
        if (!addChoiceSelected && !choicesCreationForm.getHasErrors()) {
            initPoll();
        }
        addChoiceSelected = false;
    }

    /**
     * Initialisation du sondage.
     */
    private void initPoll() {
        step = PollStep.POLL;

        // Initialisation du sondage
        poll = new PollDTO();
        pollAccount = new PollAccountDTO();
        if (userExists) {
            poll.setCreatorId(user.getLogin());
            poll.setCreatorEmail(user.getEmail());
        }

        // Initialisation des règles de notification
        notificationPreventRule = new PreventRuleDTO("vote", 0, true,
                PreventRuleManager.EMAIL_ACTION);
        reminderPreventRule = new PreventRuleDTO("rappel", 0, false,
                PreventRuleManager.EMAIL_ACTION);

        notificationCheckBox = false;
        reminderCheckBox = false;
        choiceNbCheckBox = false;

        // Initialisation des choix
        choices = new ArrayList<ChoiceDTO>();
        dateTypeChoices = new ArrayList<DateChoiceUIO>();
        imgTypeChoices = new ArrayList<ImageChoiceUIO>();
        for (int i = 0; i < nbChoices; i++) {
            choices.add(new ChoiceDTO());
            dateTypeChoices.add(new DateChoiceUIO());
            imgTypeChoices.add(new ImageChoiceUIO());
        }

        // Initialisation de la liste de votants
        addPersonSelected = false;
        votingListPersons = new ArrayList<PollAccountDTO>();
        for (int i = 0; i < nbVotingListPersons; i++) {
            votingListPersons.add(new PollAccountDTO());
        }

        // initialisation de la liste de favoris
        personList = null;
        if (userExists) {
            List<PersonListDTO> _personLists = serviceList
                    .findPersonListByUser(user.getId());
            personLists = new GenericSelectModel<PersonListDTO>(_personLists,
                    PersonListDTO.class, "name", "id", _propertyAccess);
        }
    }
}
