/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.security;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.shiro.util.StringUtils;
import org.apache.shiro.web.filter.authz.AuthorizationFilter;
import org.apache.shiro.web.util.WebUtils;
import org.chorem.pollen.PollenApplicationContext;
import org.chorem.pollen.PollenUserSecurityContext;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.services.DefaultPollenServiceContext;
import org.chorem.pollen.services.PollenServiceContext;
import org.chorem.pollen.services.PollenServiceFactory;
import org.chorem.pollen.services.impl.PollService;
import org.chorem.pollen.services.impl.SecurityService;
import org.chorem.pollen.ui.PollenSession;
import org.chorem.pollen.ui.PollenUIUtils;
import org.nuiton.topia.TopiaContext;
import org.nuiton.web.filter.TopiaTransactionFilter;

import javax.servlet.ServletContext;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static org.nuiton.i18n.I18n.n_;

public abstract class AbstractPollenAuthorization extends AuthorizationFilter {

    /**
     * Pattern to detects uri from a pollen url :
     * group 1 is action namespace
     * group 2 is action name
     * group 3 is pollUri
     * group 4 is poll page (optional)
     * group 5 is method (optional)
     */
    public static final Pattern URI_PATTERN =
            Pattern.compile("/(.[^/]+)/(.[^/]+)/(.[^!/]+)(/.[^!/]*){0,1}(!.*){0,1}");

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractPollenAuthorization.class);

    private static final String ERROR_MESSAGE_PARAMETER =
            "securityPollenErrorMessage";

    protected AbstractPollenAuthorization() {
        if (log.isInfoEnabled()) {
            log.info("Init - " + this);
        }
    }

    @Override
    public ServletContext getServletContext() {
        ServletContext servletContext = super.getServletContext();
        if (servletContext == null) {
            servletContext = PollenUIUtils.getServletContext();
            Preconditions.checkNotNull(servletContext);
            // store it back
            setServletContext(servletContext);
        }
        return servletContext;
    }

    protected PollenUserSecurityContext createSecurityContext(ServletRequest request) {

        PollenSession pollenSession = PollenSession.get(request);
        UserAccount userAccount = pollenSession.getUserAccount();

        String servletPath = ((HttpServletRequest) request).getServletPath();
        Matcher m = URI_PATTERN.matcher(servletPath);

        PollUri pollUri = null;
        if (m.find()) {

            String uriId = m.group(3);
            pollUri = PollUri.newPollUri(uriId);
        }

        PollenUserSecurityContext securityContext =
                PollenUserSecurityContext.newContext(userAccount, pollUri);

        SecurityService securityService = getSecurityService(request);

        Poll poll = null;
        String errorMessage = null;

        if (pollUri == null) {

            // no pollUri in url
            errorMessage = n_("pollen.security.error.no.pollId");

        } else {

            if (!pollUri.isPollIdNotBlank()) {

                // no pollId in uri
                errorMessage = n_("pollen.security.error.no.pollId");
            } else {

                // there is a pollId check that it exists

                PollService pollService =
                        securityService.newService(PollService.class);

                poll = pollService.getPollByPollId(pollUri.getPollId());

                if (poll == null) {

                    // poll not found
                    errorMessage = n_("pollen.security.error.poll.not.found");
                }

                if (log.isDebugEnabled()) {
                    log.debug("Can access to this poll " + pollUri.getPollId());
                }
            }
        }

        if (errorMessage != null) {

            // something wrong happens
            poll = null;
            registerError(request, errorMessage);
        }

        if (poll != null) {

            securityContext.setPoll(poll);

            if (securityContext.isWithAccountId()) {

                boolean accountExist = securityService.isAccountExist(
                        securityContext.getAccountId());

                if (!accountExist) {

                    // bad accountId
                    errorMessage = n_("pollen.security.error.bad.accountId");
                    registerError(request, errorMessage);
                }

            }

            if (errorMessage == null) {

                // load user roles
                securityContext.loadUserRoles(securityService);
            }
        }

        if (log.isInfoEnabled()) {
            log.info(String.format("request [%s], pollUri %s, user %s%s - roles %s",
                                   ((HttpServletRequest) request).getServletPath(),
                                   securityContext.getPollUri(),
                                   securityContext.getUserAccount() == null ? "no connected" : securityContext.getUserAccount().getEmail(),
                                   securityContext.isAdmin() ? " Pollen Admin" : "",
                                   securityContext.getUserRoles())
            );
        }
        return securityContext;
    }

    @Override
    protected final boolean onAccessDenied(ServletRequest request,
                                           ServletResponse response) throws IOException {
        String unauthorizedUrl = getUnauthorizedUrl();
        //SHIRO-142 - ensure that redirect _or_ error code occurs - both cannot happen due to response commit:
        if (StringUtils.hasText(unauthorizedUrl)) {
            String attribute = (String) request.getAttribute(ERROR_MESSAGE_PARAMETER);
            if (StringUtils.hasText(attribute)) {

                // add a params
                Map<String, String> params = Maps.newHashMap();
                params.put("errorMessage", attribute);
                WebUtils.issueRedirect(request, response, unauthorizedUrl, params);

            } else {

                WebUtils.issueRedirect(request, response, unauthorizedUrl);
            }
        } else {
            WebUtils.toHttp(response).sendError(HttpServletResponse.SC_UNAUTHORIZED);
        }
        return false;
    }

    protected SecurityService getSecurityService(ServletRequest request) {
        PollenApplicationContext applicationContext =
                PollenApplicationContext.get(getServletContext());

        PollenServiceFactory serviceFactory = new PollenServiceFactory();

        TopiaContext transaction =
                TopiaTransactionFilter.getTransaction(request);

        Locale locale = Locale.getDefault();
        PollenServiceContext serviceContext = DefaultPollenServiceContext.newContext(
                locale,
                transaction,
                applicationContext.getConfiguration(),
                serviceFactory,
                applicationContext.getVoteCountingFactory()
        );

        return serviceContext.newService(SecurityService.class);
    }

    protected void registerError(ServletRequest request, String errorMessage) {
        request.setAttribute(ERROR_MESSAGE_PARAMETER, errorMessage);
    }

}
