/*
 * #%L
 * Pollen :: Services
 * $Id: PollVoteCountingService.java 3716 2012-10-01 05:11:24Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollVoteCountingService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceType;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.Polls;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.votecounting.VoteCounting;
import org.chorem.pollen.votecounting.VoteCountingStrategy;
import org.chorem.pollen.votecounting.model.ChoiceScore;
import org.chorem.pollen.votecounting.model.GroupOfVoter;
import org.chorem.pollen.votecounting.model.GroupVoteCountingResult;
import org.chorem.pollen.votecounting.model.VoteCountingResult;
import org.nuiton.topia.persistence.TopiaId;

import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

/**
 * New Poll vote counting service.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public class PollVoteCountingService extends PollenServiceSupport {

    /** log. */
    private static final Log log =
            LogFactory.getLog(PollVoteCountingService.class);

    public VoteCountingResult getMainResult(Poll poll) {
        VoteCountingResult mainResult;
        if (poll.isPollGroup()) {
            GroupVoteCountingResult groupResult = getGroupResult(poll);
            mainResult = groupResult.getMainResult();
        } else {
            mainResult = getSimpleResult(poll);
        }
        return mainResult;
    }

    /**
     * Execute a simple vote counting for a given poll (if the poll
     * is a group poll, none of his group will be used).
     *
     * @param poll poll on wich vote count
     * @return vote counting result
     */
    public VoteCountingResult getSimpleResult(Poll poll) {

        Preconditions.checkNotNull(poll);
        VoteCounting voteCounting = Polls.getVoteCounting(
                serviceContext.getVoteCountingFactory(), poll);
        Preconditions.checkNotNull(voteCounting);
        VoteCountingStrategy strategy = voteCounting.newStrategy();

        // create vote counting model
        GroupOfVoter group = Polls.toSimpleVotersGroup(poll);

        // compute result by the strategy
        GroupVoteCountingResult result = strategy.votecount(group);
        return result.getMainResult();
    }

    public GroupVoteCountingResult getGroupResult(Poll poll) {

        Preconditions.checkNotNull(poll);
        Preconditions.checkNotNull(poll);
        VoteCounting voteCounting = Polls.getVoteCounting(
                serviceContext.getVoteCountingFactory(), poll);
        Preconditions.checkNotNull(voteCounting);
        VoteCountingStrategy strategy = voteCounting.newStrategy();

        // Create a groupVoter including of the root voters
        GroupOfVoter group = Polls.toGroupOfVoters(poll);

        GroupVoteCountingResult result = strategy.votecount(group);
        return result;
    }

    public String getResultValue(Choice choice,
                                 List<ChoiceScore> results) {

        String val = "";
        for (ChoiceScore result : results) {
            if (result.getChoiceId().equals(choice.getTopiaId())) {
                val = getResultValue(result);
                break;
            }
        }
        return val;
    }

    public String getResultValue(ChoiceScore result) {

        BigDecimal scoreValue = result.getScoreValue();
        String val = removeTrailing0(scoreValue == null ? "" :
                                     String.valueOf(scoreValue));
        return val;
    }

    /**
     * Retourne une chaîne contenant les résultats du sondage.
     *
     * @param poll le sondage
     * @return les résultats sous forme de chaine de caractères
     */
    public String getResultsAsString(Poll poll) {

        VoteCountingResult mainResult = getMainResult(poll);
        List<ChoiceScore> results = mainResult.getTopRanking();

        ImmutableMap<String, Choice> choicesById = Maps.uniqueIndex(
                poll.getChoice(), TopiaId.GET_TOPIA_ID);

        DateFormat dateFormat = new SimpleDateFormat(getDateTimePattern());
        List<String> resultsAsString = Lists.newArrayList();
        for (ChoiceScore result : results) {
            String choiceResult;
            Choice choice = choicesById.get(result.getChoiceId());
            if (poll.getChoiceType() == ChoiceType.DATE) {
                Date date = new Date(Long.parseLong(choice.getName()));
                choiceResult = dateFormat.format(date);
            } else {
                choiceResult = choice.getName();
            }
            choiceResult += "=" + removeTrailing0(getResultValue(result));
            resultsAsString.add(choiceResult);
        }
        String result = Joiner.on(",").join(resultsAsString);
        return result;
    }

    /**
     * Supprime le 0 final d'un nombre à virgule. Le résultat peut-être un
     * double : 1,0 -> 1 et 1,2 -> 1,2.
     *
     * @param val le nombre
     * @return le nombre sans 0 final
     */
    protected String removeTrailing0(String val) {
        if (val.endsWith(".0")) {
            val = val.substring(0, val.indexOf('.'));
        }
        return val;
    }


}
