/*
 * #%L
 * Pollen :: Services
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.PollenUserSecurityContext;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollVoteVisibility;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.services.PollenServiceSupport;
import org.nuiton.topia.TopiaException;

import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import static org.chorem.pollen.PollenUserSecurityContext.PollenUserSecurityRole;
import static org.nuiton.i18n.I18n.n_;

/**
 * Service to manager security.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4
 */
public class SecurityService extends PollenServiceSupport {

    /**
     * Obtain all user roles for the given {@code poll} using the optional
     * {@code accountId} if one is given by url or optional {@code userAccount}
     * if user is connected.
     *
     * @param poll        the poll on which finding roles
     * @param accountId   the optional accountId to get back user
     * @param userAccount the optional user account if user is connected
     * @since 1.4.5
     */
    public Set<PollenUserSecurityContext.PollenUserSecurityRole> getUserRoles(Poll poll,
                                                                              String accountId,
                                                                              UserAccount userAccount) {

        Preconditions.checkNotNull(poll);

        EnumSet<PollenUserSecurityContext.PollenUserSecurityRole> result =
                EnumSet.noneOf(PollenUserSecurityRole.class);

        if (StringUtils.isNotBlank(accountId) || userAccount != null) {

            // there is an accountId or a connected user, can find so user roles

            // try to find an admin/creator of this poll
            boolean creator = isPollCreator(poll, accountId, userAccount);

            if (creator) {

                // user is adminstrator of Pollen, so can acts as creator of poll
                // or user is creator of poll linked by his accountId or user
                // account email
                result.add(PollenUserSecurityRole.CREATOR);
            }

            if (poll.isPollFree()) {

                // try to match a voter
                boolean found = isVoterAccountId(poll, accountId, userAccount);

                if (found) {

                    // found a voter match
                    result.add(PollenUserSecurityRole.VOTER);
                }
            } else {

                // try to match a restricted voter
                boolean found = isRestrictAccountId(poll, accountId, userAccount);

                if (found) {

                    // find a restricted voter match
                    result.add(PollenUserSecurityRole.RESTRICTED_VOTER);
                }
            }
        }

        return result;
    }

    public boolean isCanAccessResult(PollenUserSecurityContext securityContext) {

        String errorMessage = checkAccessResult(securityContext);
        return errorMessage == null;
    }

    public String checkAccessResult(PollenUserSecurityContext securityContext) {

        Poll poll = securityContext.getPoll();

        // check now poll results can be displayed

        boolean publicResults = poll.isPublicResults();
        boolean continuousResults = poll.isContinuousResults();

        if (!continuousResults && !poll.isClosed()) {

            // results are not continuous and poll is not closed
            return n_("pollen.security.error.poll.not.closed.and.results.not.continuous");
        }

        if (!publicResults) {

            // poll results are private, only poll admin can see results

            if (!securityContext.isCreator() && !securityContext.isAdmin()) {

                // not a poll admin, can not access result
                return n_("pollen.security.error.poll.result.private.and.access.not.granted");
            }
        }

        return null;
    }

    public boolean isCanAccessVote(PollenUserSecurityContext securityContext) {

        return checkCanAccessVote(securityContext) == null;
    }

    public String checkCanAccessVote(PollenUserSecurityContext securityContext) {

        Poll poll = securityContext.getPoll();

        if (securityContext.isCreator()) {

            // poll admin can always access vote page
            return null;
        }

        if (securityContext.isAdmin()) {

            // pollen admin can always access vote page
            return null;
        }

        if (poll.isPublicResults()) {

            // with public results, everybody can access to vote page (but
            // can not vote for a non free poll)
            return null;
        }

        if (!poll.isPollFree() && !securityContext.isRestrictedVoter()) {

            // on none free poll, only restricted user can vote
            return n_("pollen.security.error.poll.not.free.and.access.not.granted");
        }
        return null;
    }

    public boolean isCanVote(PollenUserSecurityContext securityContext) {

        Poll poll = securityContext.getPoll();

        Date now = serviceContext.getCurrentTime();

        if (!poll.isRunning(now)) {

            // poll is not running, can not vote
            return false;
        }

        if (!poll.isPollFree() && !securityContext.isRestrictedVoter()) {

            // on none free poll, only restricted user can vote
            return false;
        }

        // ok can vote
        return true;
    }

    public boolean isCanModifyVote(PollenUserSecurityContext securityContext,
                                   String voteId) {

        Poll poll = securityContext.getPoll();

        String accountId = securityContext.getAccountId();
        UserAccount userConnected = securityContext.getUserAccount();

        Date now = serviceContext.getCurrentTime();

        if (!poll.isRunning(now)) {

            // poll is not running can not modify anything
            return false;
        }

        Vote vote = poll.getVoteByTopiaId(voteId);

        if (vote == null) {

            // vote not found, can not modify it
            return false;
        }

        PollAccount votePollAccount = vote.getPollAccount();

        if (votePollAccount.getAccountId().equals(accountId)) {

            // accountId is voteAccountId, can modifiy the vote
            return true;
        }

        if (userConnected != null) {

            if (userConnected.equals(votePollAccount.getUserAccount())) {

                // user conntected is the voter
                return true;
            }

            if (userConnected.getEmail().equals(votePollAccount.getEmail())) {

                //FIXME-tchemit-2012-08-28 : voir http://chorem.org/issues/796
                // owner linked by email
                return true;
            }
        }

        // can not modify vote in other cases
        return false;
    }

    public boolean isCanDeleteComment(PollenUserSecurityContext securityContext,
                                      Comment comment) {

        String accountId = securityContext.getAccountId();
        UserAccount userAccount = securityContext.getUserAccount();


        if (securityContext.isCreator()) {

            // poll admin can always delete comments
            return true;
        }

        PollAccount commentAccount = comment.getPollAccount();

        if (commentAccount.getAccountId().equals(accountId)) {

            // owner of comment (linked by accountId) can delete his comment
            return true;
        }

        if (userAccount != null) {

            if (userAccount.equals(commentAccount.getUserAccount())) {
                // owner of comment (linked by userAccount) can delete his comment
                return true;
            }
        }
        return false;
    }

    public boolean isCanDeleteVote(PollenUserSecurityContext securityContext,
                                   String voteId) {

        Poll poll = securityContext.getPoll();

        String accountId = securityContext.getAccountId();
        UserAccount userConnected = securityContext.getUserAccount();

        Date now = serviceContext.getCurrentTime();

        if (!poll.isRunning(now)) {

            // poll is not running can not remove anything
            return false;
        }

        Vote vote = poll.getVoteByTopiaId(voteId);

        if (vote == null) {

            // vote not found, can not delete it
            return false;
        }

        if (securityContext.isCreator()) {

            // poll admin can delete any vote
            return true;
        }

        PollAccount votePollAccount = vote.getPollAccount();

        if (votePollAccount.getAccountId().equals(accountId)) {

            // owner of vote (linked by accountId) can delete his own vote
            return true;
        }

        if (userConnected != null) {

            if (userConnected.equals(votePollAccount.getUserAccount())) {

                // owner of vote (linked by userAccount) can delete his own vote
                return true;
            }

            if (userConnected.getEmail().equals(votePollAccount.getEmail())) {

                //FIXME-tchemit-2012-08-28 : voir http://chorem.org/issues/796
                // owner linked by email
                return true;
            }
        }

        // can not modify vote in other cases
        return false;
    }

    public boolean isCanClosePoll(PollenUserSecurityContext securityContext) {

        Poll poll = securityContext.getPoll();

        boolean result = !poll.isClosed();

        if (result) {

            // poll can be closed, check user can do action
            result = securityContext.isCreator();
        }

        return result;
    }

    public List<Vote> filterVotes(Poll poll,
                                  List<Vote> allVotes,
                                  PollenUserSecurityContext userSecurityContext) {
        List<Vote> result = null;

        PollVoteVisibility voteVisibility = poll.getPollVoteVisibility();
        switch (voteVisibility) {

            case NOBODY:

                // no votes visible to anybody
                break;
            case CREATOR_ONLY:

                if (userSecurityContext.isAdmin() ||
                    userSecurityContext.isCreator()) {
                    // only admin or creator can see votes
                    result = allVotes;
                }
                break;
            case PARTICIPANT_ONLY:

                if (userSecurityContext.isAdmin() ||
                    userSecurityContext.isCreator() ||
                    userSecurityContext.isRestrictedVoter() ||
                    userSecurityContext.isVoter()) {
                    // only participant or voter can see votes
                    result = allVotes;
                }
                break;
            case EVERYBODY:

                // no restriction, everybody can sse votes
                result = allVotes;
                break;
        }

        if (result == null) {

            // no votes authorized

            result = Lists.newArrayList();

            if (userSecurityContext.isVoter() ||
                userSecurityContext.isRestrictedVoter()) {

                // but can still see his own vote

                if (userSecurityContext.isWithAccountId()) {

                    // find back his vote by accountId
                    String accountId = userSecurityContext.getAccountId();
                    for (Vote vote : allVotes) {
                        if (accountId.equals(vote.getPollAccount().getAccountId())) {
                            result.add(vote);
                            break;
                        }
                    }
                } else {

                    // user is connected, find back his vote by userAccount
                    UserAccount userAccount = userSecurityContext.getUserAccount();
                    for (Vote vote : allVotes) {
                        if (userAccount.equals(vote.getPollAccount().getUserAccount())) {
                            result.add(vote);
                            break;
                        }
                    }

                    //FIXME-tchemit-2012-08-28 : voir http://chorem.org/issues/796
                    // owner linked by email

                    if (result.isEmpty()) {
                        String userAccountEmail = userAccount.getEmail();
                        for (Vote vote : allVotes) {

                            if (userAccountEmail.equals(vote.getPollAccount().getEmail())) {
                                result.add(vote);
                                break;
                            }
                        }
                    }
                }

            }
        }
        return result;
    }

    protected boolean isPollCreator(Poll poll,
                                    String accountId,
                                    UserAccount userAccount) {

        Preconditions.checkNotNull(poll);
        Preconditions.checkState(userAccount != null ||
                                 StringUtils.isNotBlank(accountId));

        boolean result;

        if (ObjectUtils.equals(poll.getCreator().getAccountId(), accountId)) {

            // creator found by account id
            result = true;

        } else if (userAccount != null) {

            if (userAccount.isAdministrator()) {

                // use is admin of Pollen, so is also poll creator
                result = true;
            } else {

                //FIXME-tchemit-2012-08-28 : voir http://chorem.org/issues/796
                // try to link bo user account email
                result = ObjectUtils.equals(poll.getCreator().getEmail(),
                                            userAccount.getEmail());
            }
        } else {

            // no accountId, nor user connected, can not be a creator
            result = false;
        }
        return result;
    }

    protected boolean isVoterAccountId(Poll poll,
                                       String accountId,
                                       UserAccount userAccount) {

        String pollId = poll.getPollId();

        try {

            PollAccountDAO dao = getDAO(PollAccount.class);
            PollAccount pollAccount =
                    dao.findVoterPollAccount(pollId,
                                             accountId,
                                             userAccount);
            return pollAccount != null;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not check voter pollAccount existence from poll '" +
                    pollId + "'", e);
        }
    }

    protected boolean isRestrictAccountId(Poll poll,
                                          String accountId,
                                          UserAccount userAccount) {

        String pollId = poll.getPollId();
        try {

            PollAccountDAO dao = getDAO(PollAccount.class);

            PollAccount result =
                    dao.findRestrictedPollAccount(pollId,
                                                  accountId,
                                                  userAccount);

            return result != null;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not check restricted voter pollAccount existence " +
                    "from poll '" + pollId + "'", e);
        }
    }

    public boolean isAccountExist(String accountId) {

        try {

            PollAccountDAO dao = getDAO(PollAccount.class);

            PollAccount result = dao.findByAccountId(accountId);

            return result != null;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not check pollAccount existence " +
                    "from account '" + accountId + "'", e);
        }
    }

    public boolean isCanShowFeed(PollenUserSecurityContext userSecurityContext) {
        Poll poll = userSecurityContext.getPoll();
        boolean result = newService(PollFeedService.class).isFeedExists(poll);
        if (result) {
            PollVoteVisibility voteVisibility = poll.getPollVoteVisibility();
            switch (voteVisibility) {
                case NOBODY:
                case CREATOR_ONLY:
                    result = userSecurityContext.isAdmin() || userSecurityContext.isCreator();
                    break;
                case PARTICIPANT_ONLY:
                    result = userSecurityContext.isVoter() || userSecurityContext.isRestrictedVoter();
                    break;
                case EVERYBODY:
                    result = true;
                    break;
            }
        }
        return result;
    }
}
