/*
 * #%L
 * Pollen :: Services
 * 
 * $Id: PollService.java 3387 2012-05-25 07:56:56Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1.1/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenConfiguration;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollDateChoice;
import org.chorem.pollen.bean.PollImageChoice;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceDAO;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.PersonToListDAO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.business.persistence.PreventRuleDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.persistence.VotingListDAO;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.entities.PollenBinderHelper;
import org.chorem.pollen.services.PollenServiceFunctions;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.PollAccountNotFound;
import org.chorem.pollen.services.exceptions.PollChoiceNotFoundException;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.chorem.pollen.services.exceptions.UnauthorizedPollAccessException;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaFilterPagerUtil;
import org.nuiton.util.beans.Binder;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import static org.nuiton.i18n.I18n.l_;

public class PollService extends PollenServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(PollService.class);

    /**
     * Build a new Poll instance with given {@code user} as creator
     *
     * @param user Build a new Poll instance with given {@code user} as creator
     * @return Build a new Poll instance with given {@code user} as creator
     */
    public Poll getNewPoll(UserAccount user) {

        PollDAO pollDAO = getDAO(Poll.class);
        Poll result = newInstance(pollDAO);

        PollenConfiguration configuration = getConfiguration();

        // default values from configuration
        result.setChoiceType(configuration.getDefaultChoiceType());
        result.setVoteCountingType(configuration.getDefaultVoteCountingType());
        result.setPollType(configuration.getDefaultPollType());

        // Initialize creator of the poll
        PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
        PollAccount creator = newInstance(pollAccountDAO);
        creator.setAccountId(serviceContext.createPollenUrlId());

        if (user != null) {

            // Link the creator with the user
            creator.setVotingId(user.getDisplayName());
            creator.setEmail(user.getEmail());
            creator.setUserAccount(user);
        }
        result.setCreator(creator);

        // Generate Uid to have a unique poll. Will avoid multi-submit, existing
        // poll will be check by its uid.
        result.setPollId(serviceContext.createPollenUrlId());
        return result;
    }

    /**
     * Retrieve a Poll for edition. It is loaded from database if {@code pollUid}
     * is defined, otherwise a new instance is returned. The {@code clone}
     * parameter will return the {@code poll} without any technical properties (topiaId)
     * and ensure business ids (accountUid, pollUid) with new values. If {@code userId}
     * matches an existing {@link UserAccount}, the resulting poll creator will
     * be attached to it.
     *
     * @param pollUid     Uid of the poll to edit, if not defined, a new poll is instanciated
     * @param userAccount User account to attach to the creator
     * @param clone       Flag to copy or not ids in case of poll cloning
     * @return the Poll ready for edition (no longer linked to the topia context)
     * @see #getNewPoll(UserAccount)
     * @see #getNewPollCopy(Poll, UserAccount, boolean)
     */
    public Poll getPollEditable(String pollUid, UserAccount userAccount, boolean clone) {

        Poll result;
        if (pollUid == null) {
            result = getNewPoll(userAccount);

        } else {

            Poll poll = getPollByPollId(pollUid);

            if (poll == null) {
                result = getNewPoll(userAccount);

            } else {
                result = getNewPollCopy(poll, userAccount, clone);
            }
        }
        return result;
    }

    protected Poll getNewPollCopy(Poll source, UserAccount userAccount, boolean clone) {

        Preconditions.checkNotNull(source);

        PollDAO pollDAO = getDAO(Poll.class);
        Poll result = newInstance(pollDAO);

        PollenBinderHelper.simpleCopy(source, result, !clone);
        if (clone) {
            result.setPollId(serviceContext.createPollenUrlId());
        }

        // -- Creator -- //
        PollAccount creatorLoaded = source.getCreator();
        PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
        PollAccount creatorEditable = newInstance(pollAccountDAO);
        result.setCreator(creatorEditable);

        PollenBinderHelper.simpleCopy(
                creatorLoaded, creatorEditable, !clone);
        if (clone) {
            creatorEditable.setAccountId(serviceContext.createPollenUrlId());
        }

        if (creatorLoaded.getUserAccount() != null) {
            creatorEditable.setUserAccount(creatorLoaded.getUserAccount());

            // Set userAccount from arguments
        } else if (userAccount != null) {
            creatorEditable.setUserAccount(userAccount);
        }

        // -- Choice -- //
        Function<Choice, Choice> choiceCreator =
                PollenServiceFunctions.newChoiceCreator(source.getChoiceType());
        Iterable<Choice> choices =
                Iterables.transform(source.getChoice(), choiceCreator);

        for (Choice choiceLoaded : choices) {
            // Reset topiaId for clone case
            if (clone) {
                choiceLoaded.setTopiaId(null);
            }
            result.addChoice(choiceLoaded);
        }

        // -- VotingList -- //
        VotingListDAO votingListDAO = getDAO(VotingList.class);
        PersonToListDAO personToListDAO = getDAO(PersonToList.class);
        for (VotingList votingListLoaded : source.getVotingList()) {
            VotingList votingListEditable = newInstance(votingListDAO);
            result.addVotingList(votingListEditable);
            // Do not keep votingLists topiaId, to simplify the update will delete old votingLists and create new ones
            PollenBinderHelper.simpleCopy(
                    votingListLoaded, votingListEditable, false);

            for (PersonToList personToListLoaded : votingListLoaded.getPollAccountPersonToList()) {
                PersonToList personToListEditable = newInstance(personToListDAO);
                votingListEditable.addPollAccountPersonToList(personToListEditable);
                // Do not keep personToLists topiaId, to simplify the update will delete old personToLists and create new ones
                PollenBinderHelper.simpleCopy(
                        personToListLoaded, personToListEditable, false);

                PollAccount personLoaded = personToListLoaded.getPollAccount();
                PollAccount personEditable = newInstance(pollAccountDAO);
                personToListEditable.setPollAccount(personEditable);
                // copy the person, keeping topiaId is useless because we have the link with PersonToList
                PollenBinderHelper.simpleCopy(
                        personLoaded, personEditable, false);
                if (clone) {
                    personEditable.setAccountId(serviceContext.createPollenUrlId());
                }
            }
        }

        // -- PreventRule -- //
        PreventRuleDAO preventRuleDAO = getDAO(PreventRule.class);
        for (PreventRule preventRuleLoaded : source.getPreventRule()) {
            PreventRule preventRuleEditable = newInstance(preventRuleDAO);
            PollenBinderHelper.simpleCopy(
                    preventRuleLoaded, preventRuleEditable, !clone);
            result.addPreventRule(preventRuleEditable);
        }

        // Load votes to have the correct size used to check if vote is started
        result.setVote(source.getVote());

        return result;
    }

    public Map<String, Object> pollToMap(Poll poll, Binder<Poll, Poll> binder) {

        Map<String, Object> map = binder.obtainProperties(
                poll,
                Poll.PROPERTY_TITLE,
                Poll.PROPERTY_POLL_ID,
                Poll.PROPERTY_DESCRIPTION
        );
        Locale l = serviceContext.getLocale();
        if (poll.getBeginDate() == null) {
            map.put(Poll.PROPERTY_BEGIN_DATE, l_(l, "pollen.common.undefined"));
        } else {
            map.put(Poll.PROPERTY_BEGIN_DATE, decorateDate(poll.getBeginDate()));
        }
        if (poll.getEndDate() == null) {
            map.put(Poll.PROPERTY_END_DATE, l_(l, "pollen.common.undefined"));
        } else {
            map.put(Poll.PROPERTY_END_DATE, decorateDate(poll.getEndDate()));
        }

        String addingchoiceText;
        if (poll.isChoiceAddAllowed()) {

            Date beginDate = poll.getBeginChoiceDate();
            Date endDate = poll.getEndChoiceDate();

            if (beginDate == null && endDate == null) {

                // can always add choices
                addingchoiceText =
                        l_(l, "pollen.common.addingChoicesAlways");
            } else if (beginDate == null) {
                // until enddate
                addingchoiceText =
                        l_(l, "pollen.common.addingChoicesTo",
                           decorateDate(endDate));
            } else if (endDate == null) {

                // from begin date
                addingchoiceText =
                        l_(l, "pollen.common.addingChoicesFrom",
                           decorateDate(beginDate));
            } else {
                addingchoiceText =
                        l_(l, "pollen.common.addingChoicesContent",
                           decorateDate(beginDate),
                           decorateDate(endDate));
            }
        } else {
            addingchoiceText = l_(l, "pollen.common.unauthorized");
        }
        map.put("addingChoices", addingchoiceText);
        map.put("id", poll.getTopiaId());
        return map;
    }

    public Poll createPoll(Poll poll) {

        PollDAO pollDAO = getDAO(Poll.class);
        Poll result = create(pollDAO);

        // check max number choice authorized (can not be more than
        // XXX-fdesbois-2012-04-11 : don't know why the maxNbChoice must be
        // forced to the choice size, it seems useless. The case maxNbChoice = 0
        // must be available (equivalent to undefined)
//        if (poll.getMaxChoiceNb() < 1
//            || poll.getMaxChoiceNb() > poll.sizeChoice()) {
//            poll.setMaxChoiceNb(poll.sizeChoice());
//        }

        result.setAnonymous(poll.isAnonymous());
        result.setAnonymousVoteAllowed(poll.isAnonymousVoteAllowed());
        result.setChoiceAddAllowed(poll.isChoiceAddAllowed());
        result.setBeginChoiceDate(poll.getBeginChoiceDate());
        result.setBeginDate(poll.getBeginDate());
        result.setChoiceType(poll.getChoiceType());
        result.setContinuousResults(poll.isContinuousResults());
        result.setDescription(poll.getDescription());
        result.setEndDate(poll.getEndDate());
        result.setEndChoiceDate(poll.getEndChoiceDate());
        result.setPollId(poll.getPollId());
        result.setPollType(poll.getPollType());
        result.setMaxChoiceNb(poll.getMaxChoiceNb());
        result.setPublicResults(poll.isPublicResults());
        result.setTitle(poll.getTitle());
        result.setVoteCountingType(poll.getVoteCountingType());

        // create creator
        PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
        PollAccount creatorCreated = create(pollAccountDAO);
        PollAccount creator = poll.getCreator();
        creatorCreated.setVotingId(creator.getVotingId());
        creatorCreated.setEmail(creator.getEmail());
        creatorCreated.setUserAccount(creator.getUserAccount());
        creatorCreated.setAccountId(creator.getAccountId());

        result.setCreator(creatorCreated);

        // -- Choice -- //
        for (Choice choice : poll.getChoice()) {
            saveChoice(result, choice);
        }

        // -- PreventRule -- //
        for (PreventRule choice : poll.getPreventRule()) {
            savePreventRule(result, choice);
        }

        // -- VotingList -- //
        try {
            for (VotingList votingList : poll.getVotingList()) {
                // Do not set emailService, all emails will be send during 
                // onPollCreated
                saveVotingList(result, votingList, null);
            }
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Can't save votingLists", e);
        }

        commitTransaction("Could not create poll " + poll.getTitle());

        PollUrl pollVoteUrl = getPollVoteUrl(poll, false);
        PollUrl pollModerateUrl = getPollVoteUrl(poll, true);
        PollUrl pollEditUrl = getPollEditUrl(poll);

        // email notification
        EmailService emailService = newService(EmailService.class);
        emailService.onPollCreated(result, pollVoteUrl, pollModerateUrl, pollEditUrl);

        // feed notification
        // XXX-fdesbois-2012-04-12 : if the feed is the same for everybody maybe
        // not publish restricted poll that can't be accessed by everybody
        PollFeedService pollFeedService = newService(PollFeedService.class);
        pollFeedService.onPollCreated(result, pollVoteUrl);

        return result;
    }

    public void updatePoll(Poll poll) throws PollNotFoundException {

        Poll pollToUpdate = getEntityById(Poll.class, poll.getTopiaId());

        if (pollToUpdate == null) {
            throw new PollNotFoundException();
        }

        boolean voteStarted = pollToUpdate.sizeVote() > 0;

        PollAccount creatorToUpdate = pollToUpdate.getCreator();

        // FIXME-fdesbois-2012-04-05 : improve binder to avoid creator copy
        PollenBinderHelper.simpleCopy(poll, pollToUpdate, false);

        // -- Creator -- //
        PollAccount creator = poll.getCreator();
        creatorToUpdate.setUserAccount(creator.getUserAccount());
        creatorToUpdate.setVotingId(creator.getVotingId());
        creatorToUpdate.setEmail(creator.getEmail());

        // -- Choices -- //
        if (!voteStarted) {

            // Save all choices from source poll
            List<Choice> choicesAdded = Lists.newLinkedList();
            for (Choice choice : poll.getChoice()) {
                Choice choiceLoaded = saveChoice(pollToUpdate, choice);
                choicesAdded.add(choiceLoaded);
            }

            // removed choice will be deleted from db by composition relation
            pollToUpdate.clearChoice();

            // re-add choices in incoming order.
            pollToUpdate.addAllChoice(choicesAdded);
        }

        // -- PreventRules -- //
        for (PreventRule preventRule : poll.getPreventRule()) {
            savePreventRule(pollToUpdate, preventRule);
        }

        // -- VotingLists -- //
        if (!voteStarted) {

            EmailService emailService = newService(EmailService.class);
            pollToUpdate.clearVotingList();
            try {
                for (VotingList votingList : poll.getVotingList()) {
                    saveVotingList(pollToUpdate, votingList, emailService);
                }
            } catch (TopiaException e) {
                throw new PollenTechnicalException("Can't save votingLists", e);
            }
        }

        commitTransaction("Could not update poll [" + poll.getTopiaId() + "]");
    }

    public PollUrl getPollVoteUrl(Poll poll, boolean moderate) {
        URL applicationUrl = serviceContext.getApplicationURL();
        String baseUrl = applicationUrl + "/poll/votefor/";
        PollUri pollUri = PollUri.newPollUri(poll.getPollId());

        if (moderate) {
            String creatorId = poll.getCreator().getAccountId();
            pollUri.setAccountId(creatorId);
        }

        PollUrl result = PollUrl.newPollUrl(baseUrl, pollUri);
        return result;
    }

    public PollUrl getPollEditUrl(Poll poll) {
        URL applicationUrl = serviceContext.getApplicationURL();
        String baseUrl = applicationUrl + "/poll/modification/";

        String creatorId = poll.getCreator().getAccountId();
        PollUri pollUri = PollUri.newPollUri(poll.getPollId(), creatorId);

        PollUrl result = PollUrl.newPollUrl(baseUrl, pollUri);
        return result;
    }

    public PollUrl getPollResultUrl(Poll poll) {
        URL applicationUrl = serviceContext.getApplicationURL();
        String baseUrl = applicationUrl + "/poll/results/";
        PollUri pollUri = PollUri.newPollUri(poll.getPollId());

        if (!poll.isPublicResults()) {
            String creatorId = poll.getCreator().getAccountId();
            pollUri.setAccountId(creatorId);
        }

        PollUrl result = PollUrl.newPollUrl(baseUrl, pollUri);
        return result;
    }

    public List<Poll> getPolls(TopiaFilterPagerUtil.FilterPagerBean pager) {

        Preconditions.checkNotNull(pager);
        try {
            PollDAO dao = getDAO(Poll.class);
            List<Poll> result = dao.getPolls(pager);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain polls", e);
        }
    }

    public List<Poll> getCreatedPolls(TopiaFilterPagerUtil.FilterPagerBean pager,
                                      UserAccount user) {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);
        try {
            PollDAO dao = getDAO(Poll.class);
            List<Poll> result = dao.getCreatedPolls(pager, user);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain created polls", e);
        }
    }

    public List<Pair<Poll, PollAccount>> getInvitedPolls(
            TopiaFilterPagerUtil.FilterPagerBean pager,
            UserAccount user) {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);

        UserService userService = newService(UserService.class);
        UserAccount userToUse = userService.getEntityById(UserAccount.class,
                                                          user.getTopiaId());

        Preconditions.checkNotNull(userToUse);
        try {
            PollDAO pollDao = getDAO(Poll.class);
            List<Pair<Poll, PollAccount>> result =
                    pollDao.getInvitedPolls(pager, userToUse);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain invited polls", e);
        }
    }

    public List<Pair<Poll, PollAccount>> getParticipatedPolls(
            TopiaFilterPagerUtil.FilterPagerBean pager,
            UserAccount user) {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);

        UserService userService = newService(UserService.class);
        UserAccount userToUse =
                userService.getEntityById(UserAccount.class, user.getTopiaId());

        Preconditions.checkNotNull(userToUse);

        try {
            PollDAO pollDao = getDAO(Poll.class);
            List<Pair<Poll, PollAccount>> result =
                    pollDao.getParticipatedPolls(pager, userToUse);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain running polls", e);
        }
    }

    public List<Poll> getRunningPolls(boolean withEndDate) {

        try {
            PollDAO dao = getDAO(Poll.class);
            List<Poll> results = dao.getRunningPolls(withEndDate);
            return results;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain running polls", e);
        }
    }

    public Poll getPollByPollId(String pollId) {

        Preconditions.checkNotNull(pollId);
        try {
            PollDAO dao = getDAO(Poll.class);
            Poll result = dao.findByPollId(pollId);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not find poll with pollId '" + pollId + "'", e);
        }
    }

    /**
     * Retrieve a pollAccount with {@code accountId}. This pollAccount will be
     * attached to the given {@code userAccount} (only if not already attached).
     * If the {@code accountId} is undefined, a new instance of PollAccount will
     * be retrieved, otherwise a copy of the existing PollAccount is used. No
     * create or update is done here. The {@code poll} is used to retrieve a potential
     * existing vote with pollAccount linked to {@code userAccount}.
     *
     * @param accountId   Id of the existing account (optional)
     * @param userAccount UserAccount where account will be attached (optional)
     * @param poll        Poll where pollAccount will be added
     * @return the existing PollAccount or a new instance
     * @throws PollAccountNotFound if accountId is defined and doesn't match any
     *                             existing PollAccount
     */
    public PollAccount getPollAccountEditable(String accountId, UserAccount userAccount, Poll poll) throws PollAccountNotFound {
        PollAccount result = null;
        if (StringUtils.isNotEmpty(accountId)) {
            PollAccount pollAccountLoaded = getPollAccountByAccountId(accountId);

            result = copyPollAccount(pollAccountLoaded);

            // Don't remove or update userAccount link if already set
            if (userAccount != null && result.getUserAccount() == null) {
                result.setUserAccount(userAccount);

                if (log.isDebugEnabled()) {
                    log.debug(String.format("Attach User '%s' [%s] to the existing Account [%s]",
                                            userAccount.getDisplayName(),
                                            userAccount.getTopiaId(),
                                            accountId
                    ));
                }
            }

        } else {

            // Retrieve existing pollAccount from user
            if (userAccount != null) {
                PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
                PollAccount pollAccountLoaded;
                try {
                    pollAccountLoaded = pollAccountDAO.findByPollVoteUser(poll, userAccount);
                } catch (TopiaException e) {
                    throw new PollenTechnicalException(e);
                }

                if (pollAccountLoaded != null) {
                    result = copyPollAccount(pollAccountLoaded);
                }

                if (log.isDebugEnabled()) {
                    String account = result == null
                                     ? "null"
                                     : result.getVotingId() + " [" + result.getAccountId() + "]";

                    log.debug(String.format(
                            "PollAccount found from user '%s' = %s",
                            userAccount.getDisplayName(), account
                    ));
                }
            }

            if (result == null) {
                result = getNewPollAccount(userAccount);
            }
        }
        return result;
    }

    protected PollAccount copyPollAccount(PollAccount source) {
        PollAccountDAO dao = getDAO(PollAccount.class);
        PollAccount result = newInstance(dao);
        PollenBinderHelper.copy("", source, result, true);
        result.setUserAccount(source.getUserAccount());
        return result;
    }

    public PollAccount getNewPollAccount(UserAccount userAccount) {
        PollAccountDAO dao = getDAO(PollAccount.class);
        PollAccount result = newInstance(dao);
        result.setAccountId(serviceContext.createPollenUrlId());
        String votingId = userAccount != null ? userAccount.getDisplayName() : "";
        result.setVotingId(votingId);
        result.setUserAccount(userAccount);
        return result;
    }

    public PollAccount getPollAccountByAccountId(String accountId) throws PollAccountNotFound {
        try {
            PollAccountDAO dao = getDAO(PollAccount.class);
            PollAccount result = dao.findByAccountId(accountId);

            if (result == null) {
                throw new PollAccountNotFound();
            }

            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not botain account with this id", e);
        }
    }

    public void deletePoll(String pollId,
                           UserAccount userAccount,
                           String accountId) throws PollNotFoundException, PollAccountNotFound, UnauthorizedPollAccessException {

        Preconditions.checkNotNull(pollId);
        Preconditions.checkNotNull(accountId);

        Poll poll = getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }

        if (userAccount == null || userAccount.isAdministrator()) {

            // must check that accountId matches the poll creator id

            PollAccount account = getPollAccountByAccountId(accountId);

            if (!account.getAccountId().equals(poll.getCreator().getAccountId())) {
                throw new UnauthorizedPollAccessException();
            }
        }

        PollDAO dao = getDAO(Poll.class);
        delete(dao, poll);

        commitTransaction("Could not delete poll" + poll.getTitle());
    }

    public void closePoll(String pollId,
                          UserAccount userAccount,
                          String accountId) throws PollNotFoundException, PollAccountNotFound, UnauthorizedPollAccessException {

        Preconditions.checkNotNull(pollId);
        Preconditions.checkNotNull(accountId);

        Poll poll = getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }

        if (userAccount == null || userAccount.isAdministrator()) {

            // must check that accountId matches the poll creator id

            PollAccount account = getPollAccountByAccountId(accountId);

            if (!account.getAccountId().equals(poll.getCreator().getAccountId())) {
                throw new UnauthorizedPollAccessException();
            }
        }

        if (!poll.isChoiceEmpty()) {
            for (Choice choice : poll.getChoice()) {
                choice.setValidate(true);
            }
        }
        Date now = new Date();
        if (poll.getEndDate() == null || now.before(poll.getEndDate())) {
            poll.setEndDate(now);
        }
        poll.setClosed(true);

        commitTransaction("Could not close poll " + poll.getTitle());
    }

    public Choice getNewChoice(ChoiceType choiceType) {
        Choice result;
        switch (choiceType) {

            case DATE:
                result = new PollDateChoice();
                break;

            case IMAGE:
                result = new PollImageChoice();
                break;

            case TEXT:
            default:
                ChoiceDAO dao = getDAO(Choice.class);
                result = newInstance(dao);
        }
        return result;
    }

    public void addChoice(String pollId, Choice choice) throws PollNotFoundException {

        Preconditions.checkNotNull(pollId);
        Preconditions.checkNotNull(choice);

        Poll poll = getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }

        saveChoice(poll, choice);

        commitTransaction("Can't create new choice [" +
                          poll.getChoiceType() + "] for poll '" + pollId + "'");
    }

    protected Choice saveChoice(Poll poll, Choice choice) {

        ChoiceType choiceType = poll.getChoiceType();
        ChoiceDAO dao = getDAO(Choice.class);
        Choice choiceLoaded;
        if (choice.getTopiaId() == null) {
            choiceLoaded = create(dao);
            poll.addChoice(choiceLoaded);

        } else {
            choiceLoaded = poll.getChoiceByTopiaId(choice.getTopiaId());
        }

        if (choiceType == ChoiceType.IMAGE) {


            PollImageChoice imageChoice = (PollImageChoice) choice;
            imageChoice.toChoice(choiceLoaded);
            if (choice.getTopiaId() == null) {
                // copy image where it belong and generate the thumb
                // only if choice is to create
                try {
                    saveImages(poll, imageChoice);
                } catch (IOException e) {
                    throw new PollenTechnicalException(
                            "Could not create image choice", e);
                }
            }

        } else if (choiceType == ChoiceType.DATE) {

            // date choice

            PollDateChoice dateChoice = (PollDateChoice) choice;
            dateChoice.toChoice(choiceLoaded);

        } else {

            // text choice
            choiceLoaded.setDescription(choice.getDescription());
            choiceLoaded.setValidate(choice.isValidate());
            choiceLoaded.setName(choice.getName());
        }

        return choiceLoaded;
    }

    public void deleteChoice(String pollId, String choiceId)
            throws PollNotFoundException, PollChoiceNotFoundException {

        Poll poll = getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }
        Choice choice = poll.getChoiceByTopiaId(choiceId);

        if (choice == null) {
            throw new PollChoiceNotFoundException();
        }

        ChoiceDAO dao = getDAO(Choice.class);

        poll.removeChoice(choice);
        delete(dao, choice);

        commitTransaction("Could not delete choice " + choice.getName());
    }

    public void checkPoll(PollUri uri) throws PollNotFoundException {

        String pollId = uri.getPollId();

        Poll poll = getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }
    }

    public void checkPollResult(PollUri uri) throws PollNotFoundException, UnauthorizedPollAccessException {

        String pollId = uri.getPollId();

        Poll poll = getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }

        if (!poll.isPublicResults()) {
            throw new UnauthorizedPollAccessException();
        }
    }

    public void checkPollAccount(PollUri uri) throws PollNotFoundException, UnauthorizedPollAccessException {

        String pollId = uri.getPollId();
        String accountId = uri.getAccountId();

        Poll poll = getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }

        PollAccountDAO dao = getDAO(PollAccount.class);

        if (poll.getPollType() != PollType.FREE) {

            // check that uri contains a correct poll account for this

            // Use PersonToList association entity to find coherence between
            // the poll and votingId

            try {
                PollAccount result =
                        dao.getRestrictedPollAccount(pollId, accountId);

                if (result == null) {
                    throw new UnauthorizedPollAccessException();
                }
            } catch (TopiaException e) {
                throw new PollenTechnicalException(
                        "Could not obtain restricted pollAccount", e);
            }

        } else if (accountId != null) {

            try {

                if (!dao.existsByPollVoteAccountId(pollId, accountId)) {
                    throw new UnauthorizedPollAccessException();
                }
            } catch (TopiaException e) {
                throw new PollenTechnicalException(
                        "Could not check pollAccount existence from poll '" +
                        pollId + "' and account '" + accountId + "'", e);
            }
        }
    }

    public void checkPollCreator(PollUri uri) throws PollNotFoundException, UnauthorizedPollAccessException {

        String pollId = uri.getPollId();
        String accountId = uri.getAccountId();

        Poll poll = getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }

        String creatorId = poll.getCreator().getAccountId();

        if (!creatorId.equals(accountId)) {
            throw new UnauthorizedPollAccessException();
        }
    }

    public File getPollChoiceImageFile(String pollId,
                                       String choiceId,
                                       boolean thumb) {
        File imageDirectory = getConfiguration().getImageDirectory();
        File pollDirectory = new File(imageDirectory, pollId);

        if (thumb) {
            choiceId = THUMB_PREFIX + choiceId;
        }
        File result = new File(pollDirectory, choiceId);

        return result;
    }


    public static final String THUMB_PREFIX = "thumb_";

    /**
     * Sauvegarde des images d'un choix de type image.
     *
     * @param poll   le sondage contenant le choix
     * @param choice le choix à sauvegarder.
     * @throws IOException si un problème IO lors de la copie ou génération de la miniature
     */
    public void saveImages(Poll poll, PollImageChoice choice) throws IOException {

        String location = choice.getLocation();

        String pollId = poll.getPollId();
        String name = choice.getName();

        File pollChoiceImage = getPollChoiceImageFile(pollId, name, false);
        File pollChoiceImageThumb = getPollChoiceImageFile(pollId, name, true);

        FileUtils.copyFile(new File(location), pollChoiceImage);
        createThumbnail(pollChoiceImage, pollChoiceImageThumb, 100);
    }

    public void addVoteToPoll(Poll poll, Vote vote) {

        String pollId = poll.getTopiaId();
        String voteId = vote.getTopiaId();

        Poll pollToUpdate = getEntityById(Poll.class, pollId);
        Vote voteToAdd = getEntityById(Vote.class, voteId);

        pollToUpdate.addVote(voteToAdd);

        // Update hasVoted flag for RESTRICTED/GROUP poll
        if (PollType.FREE != poll.getPollType()) {

            PersonToList personToList = pollToUpdate.getPersonToListByVote(voteToAdd);
            personToList.setHasVoted(true);
        }

        commitTransaction("Can't add the vote [" + voteId + "] to the poll [" + pollId + "]");

        // Send notification if necessary
        PreventRuleService preventRuleService = newService(PreventRuleService.class);
        PollUrl pollVoteUrl = getPollVoteUrl(pollToUpdate, false);
        PollUrl modifUrl = getPollEditUrl(pollToUpdate);
        preventRuleService.onVoteAdded(poll, pollVoteUrl, modifUrl);
    }

    public PersonToList getNewPersonToList(PollAccount pollAccount) {
        PersonToList result = newInstance(getDAO(PersonToList.class));
        result.setWeight(1);
        PollAccount pollAccount2 = newInstance(getDAO(PollAccount.class));
        String accountId = serviceContext.createPollenUrlId();
        pollAccount2.setAccountId(accountId);
        result.setPollAccount(pollAccount2);
        if (pollAccount != null) {
            pollAccount2.setComment(pollAccount.getComment());
            pollAccount2.setEmail(pollAccount.getEmail());
            pollAccount2.setUserAccount(pollAccount.getUserAccount());
            pollAccount2.setVotingId(pollAccount.getVotingId());
        }
        return result;
    }

    protected void saveVotingList(Poll poll, VotingList votingList, EmailService emailService) throws TopiaException {

        VotingListDAO votingListDAO = getDAO(VotingList.class);
        PersonToListDAO personToListDAO = getDAO(PersonToList.class);
        PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);

        // Prepare the VotingList and add it to the poll
        VotingList result;
        if (votingList.getTopiaId() == null) {
            result = create(votingListDAO);
            poll.addVotingList(result);

        } else {
            result = poll.getVotingListByTopiaId(votingList.getTopiaId());

            // FIXME-fdesbois-2012-04-05 : why the poll association is not keeped ?!? because of the clear
            result.setPoll(poll);
        }

        result.setName(votingList.getName());
        result.setWeight(votingList.getWeight());

        PollUrl voteURL = getPollVoteUrl(poll, false);

        // Merge PersonToList
        List<PersonToList> personToListsUpdated = Lists.newArrayList();
        for (PersonToList personToList : votingList.getPollAccountPersonToList()) {

            PollAccount pollAccount = personToList.getPollAccount();

            PersonToList personToListLoaded;
            PollAccount pollAccountLoaded;
            if (personToList.getTopiaId() == null) {
                personToListLoaded = create(personToListDAO);

                // The model doesn't have any composition for this relation,
                // the link must be set in both objects
                personToListLoaded.setVotingList(result);

                // FIXME-fdesbois-2012-04-12 : find a better way to ensure accountId
                String accountId = pollAccount.getAccountId();
                if (accountId == null) {
                    accountId = serviceContext.createPollenUrlId();
                }

                pollAccountLoaded = pollAccountDAO.findByAccountId(accountId);
                if (pollAccountLoaded == null) {
                    pollAccountLoaded = create(pollAccountDAO);
                    pollAccountLoaded.setAccountId(accountId);
                }

            } else {
                personToListLoaded = getEntityById(PersonToList.class, personToList.getTopiaId());
                pollAccountLoaded = personToListLoaded.getPollAccount();
            }
            personToListsUpdated.add(personToListLoaded);

            personToListLoaded.setWeight(personToList.getWeight());
            pollAccountLoaded.setVotingId(pollAccount.getVotingId());

            boolean emailChanged = !Objects.equal(pollAccountLoaded.getEmail(),
                                                  pollAccount.getEmail());

            pollAccountLoaded.setEmail(pollAccount.getEmail());

            // Send email to the person if service is defined and email has changed
            if (emailService != null && emailChanged) {
                emailService.onRestrictedPersonAdded(poll, pollAccountLoaded, voteURL);
            }

            // Update the account (otherwise a saving error will occurs if
            // already exists on delete cascade)
            PollAccount pollAccountUpdated = update(pollAccountDAO, pollAccountLoaded);
            personToListLoaded.setPollAccount(pollAccountUpdated);
        }
        result.setPollAccountPersonToList(personToListsUpdated);
    }

    protected void savePreventRule(Poll poll, PreventRule preventRule) {

        PreventRuleDAO preventRuleDAO = getDAO(PreventRule.class);

        // For the moment only one rule by scope
        PreventRule preventRuleLoaded = poll.getPreventRuleByScope(preventRule.getScope());

        if (preventRuleLoaded == null) {
            preventRuleLoaded = create(preventRuleDAO);
            poll.addPreventRule(preventRuleLoaded);
            preventRuleLoaded.setScope(preventRule.getScope());
            preventRuleLoaded.setActive(true);
        }

        preventRuleLoaded.setMethod(preventRule.getMethod());
        preventRuleLoaded.setOneTime(preventRule.isOneTime());
        preventRuleLoaded.setRepeated(preventRule.isRepeated());
        preventRuleLoaded.setSensibility(preventRule.getSensibility());
    }

    /**
     * Création d'une miniature.
     *
     * @param img         L'image à miniaturiser.
     * @param thumbCopied le chemin complet vers la miniature.
     * @param width       La largeur de la miniature.
     * @throws IOException if could not create thumb image
     */
    protected static void createThumbnail(File img,
                                          File thumbCopied,
                                          int width) throws IOException {

        ImageIcon ii = new ImageIcon(img.getAbsolutePath());
        Image image = ii.getImage();
        double imageRatio = (double) image.getHeight(null)
                            / (double) image.getWidth(null);
        int height = (int) (width * imageRatio);

        BufferedImage thumbImage = new BufferedImage(width, height,
                                                     BufferedImage.TYPE_INT_RGB);
        Graphics2D graphics2D = thumbImage.createGraphics();
        graphics2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
                                    RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        graphics2D.drawImage(image, 0, 0, width, height, null);

        ImageIO.write(thumbImage, "jpg", thumbCopied);

        if (log.isDebugEnabled()) {
            log.debug("Thumbnail created: " + thumbCopied.getName() + " (size="
                      + thumbCopied.length() + ")");
        }
    }
}
