/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.services;

import java.util.List;
import java.util.Map;

import org.chorem.pollen.business.dto.PersonListDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.persistence.VotingList;
import org.nuiton.topia.TopiaContext;

/**
 * Interface du service de gestion des listes de votants et des listes de
 * favoris.
 *
 * @author enema
 * @version $Id: ServiceList.java 2672 2009-07-24 12:58:07Z nrannou $
 */
public interface ServiceList {

    /**
     * Création d'une liste de votant
     *
     * @param votingList la liste
     * @return identifiant de la liste créée
     */
    public String createVotingList(VotingListDTO votingList);

    /**
     * Création des groupes de votant du sondage
     *
     * @param votingLists les dtos listes
     * @param pollId l'identifiant du sondage
     * @param transaction la transaction du sondage
     * @return les groupes créés
     */
    public List<VotingList> createVotingLists(List<VotingListDTO> votingLists,
            String pollId, TopiaContext transaction);

    /**
     * Mise à jour d'une liste de votant
     *
     * @param votingList la liste
     * @return true si la liste a été mise à jours
     */
    public boolean updateVotingList(VotingListDTO votingList);

    /**
     * Suppression d'une liste de votant
     *
     * @param votingListId l'identifiant de la liste
     * @return true si la liste a été mise à jours
     */
    public boolean deleteVotingList(String votingListId);

    /**
     * Recherche d'une liste par son identifiant
     *
     * @param votingListId l'identifiant de la liste
     * @return la liste
     */
    public VotingListDTO findVotingListById(String votingListId);

    /**
     * Recherche d une liste de votant à partir du nom
     *
     * @param name le nom de la liste
     * @return les listes
     */
    public List<VotingListDTO> findVotingListByName(String name);

    /**
     * Recherche des listes à partir d'un filtre
     *
     * @param properties filtre sur les champs de la table VotingList
     * @return les listes de votants
     */
    public List<VotingListDTO> selectVotingList(Map<String, Object> properties);

    /**
     * Création d'une liste de favoris
     *
     * @param personList la liste
     * @return identifiant de la liste créée
     */
    public String createPersonList(PersonListDTO personList);

    /**
     * Création d'une liste de favoris pour un user
     *
     * @param lists les dtos personList
     * @param userId l'identifiant de l'utilisateur
     * @param transaction la transaction du sondage
     * @return true si toutes les listes ont été créés
     */
    public boolean createPersonLists(List<PersonListDTO> lists, String userId,
            TopiaContext transaction);

    /**
     * Mise à jour d'une liste de favoris
     *
     * @param personlistDTO la liste
     * @return true si la liste a été mise à jours
     */
    public boolean updatePersonList(PersonListDTO personlistDTO);

    /**
     * Suppression d'une liste de favoris
     *
     * @param personListId l'identifiant de la liste
     * @return true si la liste a été supprimée
     */
    public boolean deletePersonList(String personListId);

    /**
     * Suppression des listes de favoris d'un utilisateur
     *
     * @param userId l'identifiant de l'utilisateur
     * @return true si les listes ont été supprimées
     */
    public boolean deletePersonLists(String userId);

    /**
     * Recherche d'une liste par son identifiant
     *
     * @param personListId l'identifiant de la liste
     * @return la liste
     */
    public PersonListDTO findPersonListById(String personListId);

    /**
     * Recherche des listes de favoris à partir du nom
     *
     * @param name le nom de la liste
     * @return les listes
     */
    public List<PersonListDTO> findPersonListByName(String name);

    /**
     * Recherche des listes de favoris d'un utilisateur.
     *
     * @param userId Le créateur de la liste
     * @return les listes
     */
    public List<PersonListDTO> findPersonListByUser(String userId);

    /**
     * Recherche des listes à partir d'un filtre
     *
     * @param properties filtre sur les champs de la table personList
     * @return les listes de votants
     */
    public List<PersonListDTO> selectPersonList(Map<String, Object> properties);
}
