/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseApplicationListener.java 644 2012-09-11 12:48:41Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-2.0/echobase-ui/src/main/java/fr/ifremer/echobase/ui/EchoBaseApplicationListener.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui;

import com.google.common.base.Preconditions;
import fr.ifremer.echobase.EchoBaseConfiguration;
import fr.ifremer.echobase.entities.DriverType;
import fr.ifremer.echobase.entities.EchoBaseUser;
import fr.ifremer.echobase.persistence.EchoBaseDbMeta;
import fr.ifremer.echobase.persistence.EchoBaseEntityHelper;
import fr.ifremer.echobase.persistence.EchobaseTopiaContexts;
import fr.ifremer.echobase.services.DefaultEchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseServiceContext;
import fr.ifremer.echobase.services.UserService;
import fr.ifremer.echobase.services.embeddedapplication.EmbeddedApplicationService;
import fr.ifremer.echobase.services.workingDb.WorkingDbConfigurationService;
import fr.ird.converter.FloatConverter;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaRuntimeException;
import org.nuiton.util.converter.ConverterUtil;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * To listen start or end of the application.
 * <p/>
 * On start we will load the configuration and check connection to internal
 * database, creates schema and create an admin user in none found in database.
 * <p/>
 * On stop, just release the application configuration.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseApplicationListener implements ServletContextListener {

    /** Logger. */
    protected static final Log log =
            LogFactory.getLog(EchoBaseApplicationListener.class);

    static {
        try {
            Class.forName("org.postgresql.Driver");
        } catch (ClassNotFoundException e) {
            if (log.isErrorEnabled()) {
                log.error("Could not find pg driver", e);
            }
        }
        try {
            Class.forName("org.h2.Driver");
        } catch (ClassNotFoundException e) {
            if (log.isErrorEnabled()) {
                log.error("Could not find h2 driver", e);
            }
        }
    }

    @Override
    public void contextInitialized(ServletContextEvent sce) {

        if (log.isInfoEnabled()) {
            log.info("Application starting at " + new Date() + "...");
        }

        // init I18n
        DefaultI18nInitializer i18nInitializer =
                new DefaultI18nInitializer("echobase-i18n");
        i18nInitializer.setMissingKeyReturnNull(true);
        I18n.init(i18nInitializer, Locale.getDefault());

        // init converters
        Converter converter = ConverterUtil.getConverter(Float.class);
        if (converter != null) {
            ConvertUtils.deregister(Float.class);
        }
        ConvertUtils.register(new FloatConverter(), Float.class);

        // initialize configuration
        EchoBaseConfiguration configuration = new EchoBaseConfiguration();

        // initialize internal root context
        TopiaContext internalRootContext =
                EchobaseTopiaContexts.newInternalDb(
                        configuration.getInternalDbDirectory());

        // initialize application context
        EchoBaseApplicationContext applicationContext =
                new EchoBaseApplicationContext();
        EchoBaseApplicationContext.setApplicationContext(
                sce.getServletContext(), applicationContext);

        applicationContext.setConfiguration(configuration);
        applicationContext.setDbMeta(EchoBaseDbMeta.newDbMeta());
        applicationContext.setInternalRootContext(internalRootContext);

        EchoBaseServiceContext serviceContext =
                DefaultEchoBaseServiceContext.newContext(
                        Locale.getDefault(),
                        null,
                        null,
                        configuration,
                        applicationContext.getDbMeta()
                );

        // init database (and create minimal admin user if required)
        try {
            initInternalDatabase(applicationContext, serviceContext);
        } catch (TopiaException e) {
            throw new TopiaRuntimeException("Could not init internal db", e);
        }

        // copy drivers if required
        try {
            copyDriverFiles(serviceContext);
        } catch (IOException e) {
            throw new TopiaRuntimeException("Could not install drivers", e);
        }
    }

    @Override
    public void contextDestroyed(ServletContextEvent sce) {
        if (log.isInfoEnabled()) {
            log.info("Application is ending at " + new Date() + "...");
        }

        ServletContext servletContext = sce.getServletContext();

        // get application context
        EchoBaseApplicationContext applicationContext =
                EchoBaseApplicationContext.getApplicationContext(servletContext);

        // remove application context from container
        EchoBaseApplicationContext.removeApplicationContext(servletContext);

        try {
            // release internal db
            TopiaContext rootContext =
                    applicationContext.getInternalRootContext();
            EchoBaseEntityHelper.releaseRootContext(rootContext);
        } finally {

            // release all user sessions
            Set<EchoBaseSession> sessions = applicationContext.getEchoBaseSessions();
            if (CollectionUtils.isNotEmpty(sessions)) {
                for (EchoBaseSession session : sessions) {
                    applicationContext.destroyEchoBaseSession(session);
                }
            }
        }
    }

    /**
     * Init the internal database, says :
     * <ul>
     * <li>If no schema found or if asked to updateSchema using the
     * {@code updateSchema} configuration option is on.</li>
     * <li>If no user found is db, create a administrator user
     * {@code admin/admin}</li>
     * </ul>
     *
     * @param context application context where to store global internal db root context
     */
    protected void initInternalDatabase(EchoBaseApplicationContext context,
                                        EchoBaseServiceContext serviceContext) throws TopiaException {

        Preconditions.checkNotNull(context);

        EchoBaseConfiguration configuration = context.getConfiguration();
        Preconditions.checkNotNull(configuration);

        EchoBaseDbMeta dbMeta = context.getDbMeta();
        Preconditions.checkNotNull(dbMeta);

        TopiaContext rootContext = context.getInternalRootContext();
        Preconditions.checkNotNull(rootContext);

        if (configuration.isUpdateSchema()) {
            if (log.isInfoEnabled()) {
                log.info("Will update schema...");
            }
            rootContext.updateSchema();
        }

        TopiaContext tx = rootContext.beginTransaction();
        try {
            serviceContext.setInternalTransaction(tx);

            UserService service = serviceContext.getService(UserService.class);

            List<EchoBaseUser> users = service.getUsers();

            if (CollectionUtils.isEmpty(users)) {

                // no users in database create the admin user
                if (log.isInfoEnabled()) {
                    log.info("No user in database, will create default " +
                             "users.");
                }

                service.createDefaultUsers();
            }

            if (configuration.isEmbedded()) {

                if (log.isInfoEnabled()) {
                    log.info("Will try t create default working db " +
                             "configuration for embedded db.");
                }
                // try to create a default embedded working db configuration
                serviceContext.getService(WorkingDbConfigurationService.class).
                        createEmbeddedWorkingDbConfiguration();
            }
        } finally {
            serviceContext.setInternalTransaction(null);
            EchoBaseEntityHelper.closeConnection(tx);
        }
    }

    protected void copyDriverFiles(EchoBaseServiceContext serviceContext) throws IOException {
        EmbeddedApplicationService service =
                serviceContext.getService(EmbeddedApplicationService.class);
        EchoBaseConfiguration configuration = serviceContext.getConfiguration();
        File libDirectory = configuration.getLibDirectory();
        for (DriverType driverType : DriverType.values()) {
            String pilotFileName = driverType.getPilotFileName(configuration);
            File pilotFile = new File(libDirectory, pilotFileName);
            if (!pilotFile.exists()) {

                // copy it from class-path
                if (log.isInfoEnabled()) {
                    log.info("Copy driver " + pilotFileName +
                             " to directory " + libDirectory);
                }
                service.copyEmbeddedBinaryFile(pilotFileName, libDirectory);
            }
        }
    }
}
