/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: AbstractWaitAndExecAction.java 589 2012-09-02 15:11:02Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-1.1/echobase-ui/src/main/java/fr/ifremer/echobase/ui/actions/AbstractWaitAndExecAction.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui.actions;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.Preparable;
import fr.ifremer.echobase.persistence.EchoBaseEntityHelper;
import fr.ifremer.echobase.services.AbstractEchobaseActionConfiguration;
import fr.ifremer.echobase.services.EchoBaseService;
import org.nuiton.topia.TopiaContext;

import java.io.IOException;

/**
 * Abstract long action using the exec and wait interceptor.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public abstract class AbstractWaitAndExecAction<M extends AbstractEchobaseActionConfiguration, S extends EchoBaseService> extends EchoBaseActionSupport implements Preparable {

    private static final long serialVersionUID = 1L;

    /**
     * Model type ot use.
     *
     * @since 1.1
     */
    private final Class<M> modelType;

    /**
     * Service type to use in this action.
     *
     * @since 1.1
     */
    private final Class<S> serviceType;

    /** Configuration of the action. */
    private M model;

    /** Shared actionContext to reuse all invocation of this same action. */
    private ActionContext actionContext;

    protected AbstractWaitAndExecAction(Class<M> modelType,
                                        Class<S> serviceType) {
        this.modelType = modelType;
        this.serviceType = serviceType;
    }

    protected abstract void startAction(S service, M model) throws Exception;

    public final M getModel() {
        return model;
    }

    @Override
    public void prepare() throws Exception {

        if (actionContext == null) {

            // keep it since exec and wait then use another thread
            actionContext = ActionContext.getContext();
        } else {

            // having the action context here means we already came here,
            // now we need to propagate it
            ActionContext.setContext(actionContext);
        }
        model = getEchoBaseSession().getActionConfiguration(modelType);
    }

    @Override
    public final String execute() throws Exception {

        model.beginAction();

        // having the action context here means we already came here,
        // now we need to propagate it
        ActionContext.setContext(actionContext);

        // we must use a standalone transaction since it will pass through
        // in more than one request

        TopiaContext tx = EchoBaseEntityHelper.newTransactionFromRootContext(
                getServiceContext().getTransaction());

        try {
            getServiceContext().setTransaction(tx);

            S service = getService(serviceType);

            startAction(service, model);

            model.endAction();

            return SUCCESS;
        } finally {

            try {
                EchoBaseEntityHelper.closeConnection(tx);
            } finally {
                closeAction(model);
            }
        }
    }

    protected void closeAction(M model) throws Exception {
        // by default do nothing
    }

    protected void destroyModel(M model) throws IOException {

        try {
            // clean configuration
            model.destroy();
        } finally {

            // remove configuration from session
            getEchoBaseSession().removeActionConfiguration(modelType);
        }
    }
}