/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseApplicationListener.java 55 2011-11-13 22:22:54Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-0.1/echobase-ui/src/main/java/fr/ifremer/echobase/ui/EchoBaseApplicationListener.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui;

import com.google.common.base.Supplier;
import com.google.common.collect.Lists;
import fr.ifremer.echobase.EchoBaseConfiguration;
import fr.ifremer.echobase.EchoBaseTechnicalException;
import fr.ifremer.echobase.EchoBaseTopiaRootContextSupplierFactory;
import fr.ifremer.echobase.entities.EchoBaseEntityEnum;
import fr.ifremer.echobase.entities.EchoBaseUser;
import fr.ifremer.echobase.entities.EchoBaseUserImpl;
import fr.ifremer.echobase.entities.meta.DbMeta;
import fr.ifremer.echobase.services.EchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseServiceContextImpl;
import fr.ifremer.echobase.services.EchoBaseServiceFactory;
import fr.ifremer.echobase.services.UserService;
import fr.ifremer.echobase.ui.actions.EchoBaseActionSupport;
import fr.ird.converter.FloatConverter;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaUtil;
import org.nuiton.util.converter.ConverterUtil;

import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

/**
 * To listen start or end of the application.
 * <p/>
 * On start we will load the configuration and check connection to internal
 * database, creates schema and create an admin user in none found in database.
 * <p/>
 * On stop, just release the application configuration.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseApplicationListener implements ServletContextListener {

    /** Logger. */
    protected static final Log log =
            LogFactory.getLog(EchoBaseApplicationListener.class);

    private Supplier<TopiaContext> rootContextSupplier;

    @Override
    public void contextInitialized(ServletContextEvent sce) {

        if (log.isInfoEnabled()) {
            log.info("Application starting at " + new Date() + "...");
        }

        // init I18n
        DefaultI18nInitializer i18nInitializer =
                new DefaultI18nInitializer("echobase-i18n");
        i18nInitializer.setMissingKeyReturnNull(true);
        I18n.init(i18nInitializer, Locale.getDefault());

        EchoBaseApplicationContext applicationContext = new EchoBaseApplicationContext();
        sce.getServletContext().setAttribute(EchoBaseActionSupport.APPLICATION_CONTEXT_PARAMETER, applicationContext);

        // initialize configuration
        EchoBaseConfiguration configuration = new EchoBaseConfiguration();
        applicationContext.setConfiguration(configuration);

        if (log.isInfoEnabled()) {
            log.info("Initializing RootContextSupplier...");
        }
        EchoBaseTopiaRootContextSupplierFactory factory =
                new EchoBaseTopiaRootContextSupplierFactory();
        rootContextSupplier = factory.newDatabaseFromConfig(configuration);
        applicationContext.setRootContextSupplier(rootContextSupplier);
        List<EchoBaseEntityEnum> entityEnums =
                Lists.newArrayList(EchoBaseEntityEnum.values());
//        entityEnums.remove(EchoBaseEntityEnum.EchoBaseUser);
        DbMeta dbMeta = new DbMeta(entityEnums.toArray(new EchoBaseEntityEnum[entityEnums.size()]));

        applicationContext.setDbMeta(dbMeta);

        // register our not locale dependant converter
        Converter converter = ConverterUtil.getConverter(Float.class);
        if (converter != null) {
            ConvertUtils.deregister(Float.class);
        }
        ConvertUtils.register(new FloatConverter(), Float.class);

        // init database (and create minimal admin user if required)
        try {
            boolean schemaExist = isSchemaCreated();
            if (!schemaExist ||
                configuration.getOptionAsBoolean("updateSchema")) {

                updateSchema(configuration);
            }

            createAdminUser(applicationContext);
        } catch (TopiaException e) {
            throw new EchoBaseTechnicalException("Could not init db", e);
        }
    }

    @Override
    public void contextDestroyed(ServletContextEvent sce) {

        if (log.isInfoEnabled()) {
            log.info("Application is ending at " + new Date() + "...");
        }
        if (rootContextSupplier != null) {
            if (log.isInfoEnabled()) {
                log.info("Shuting down RootContextSupplier...");
            }
            TopiaContext rootContext = rootContextSupplier.get();
            if (!rootContext.isClosed()) {
                try {
                    rootContext.closeContext();
                } catch (TopiaException te) {
                    if (log.isErrorEnabled()) {
                        log.error("Could not close rootContext", te);
                    }
                }
            }
        }
    }

    protected void updateSchema(EchoBaseConfiguration configuration) throws TopiaException {
        if (log.isInfoEnabled()) {
            log.info("Will create or update schema for db.");
        }
        // must create the schema

        Properties dbConf = configuration.getProperties();

        dbConf.put("hibernate.hbm2ddl.auto", "update");

        EchoBaseTopiaRootContextSupplierFactory factory =
                new EchoBaseTopiaRootContextSupplierFactory();
        Supplier<TopiaContext> topiaContextSupplier =
                factory.newDatabaseFromProperties(dbConf);

        // start a connexion to load schema
        TopiaContext tx = null;

        try {
            tx = topiaContextSupplier.get().beginTransaction();

        } finally {

            // no more update of schema...
            dbConf.put("hibernate.hbm2ddl.auto", "none");

            closeTransaction(tx);
        }
    }

    /**
     * Creates the adminsitrator ({@code admin/admin}) on the database.
     *
     * @param applicationContext application context
     * @throws TopiaException if could not create the user.
     */
    protected void createAdminUser(EchoBaseApplicationContext applicationContext) throws TopiaException {

        EchoBaseConfiguration configuration =
                applicationContext.getConfiguration();

        EchoBaseServiceFactory serviceFactory =
                new EchoBaseServiceFactory();
        TopiaContext transaction = rootContextSupplier.get().beginTransaction();

        try {
            EchoBaseServiceContext serviceContext = EchoBaseServiceContextImpl.newContext(
                    Locale.getDefault(),
                    transaction,
                    configuration,
                    applicationContext.getDbMeta(),
                    serviceFactory
            );

            UserService service =
                    serviceFactory.newService(UserService.class, serviceContext);

            List<EchoBaseUser> users = service.getUsers();

            if (CollectionUtils.isEmpty(users)) {

                // no users in database create the admin user
                if (log.isInfoEnabled()) {
                    log.info("No user in database, will create default " +
                             "admin user (password admin).");
                }

                service.createDefaultAdminUser();
            }
        } finally {
            transaction.closeContext();
        }
    }

    protected boolean isSchemaCreated() throws TopiaException {

        TopiaContextImplementor tx =
                (TopiaContextImplementor)
                        rootContextSupplier.get();
        try {
            boolean schemaFound = TopiaUtil.isSchemaExist(
                    tx,
                    EchoBaseUserImpl.class.getName()
            );

            return schemaFound;

        } finally {
            closeTransaction(tx);
        }
    }

    /**
     * Try to close the given transaction.
     *
     * @param tx the transaction to close
     * @throws TopiaException if could not close the transaction
     */
    protected void closeTransaction(TopiaContext tx) throws TopiaException {
        if (tx != null && !tx.isClosed()) {
            tx.closeContext();
        }
    }

}
