package com.franciaflex.faxtomail.services.service;

/*
 * #%L
 * FaxToMail :: Service
 * $Id: MailFolderService.java 146 2014-06-06 16:26:07Z echatellier $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.1/faxtomail-service/src/main/java/com/franciaflex/faxtomail/services/service/MailFolderService.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaEntities;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;
import org.nuiton.util.pagination.PaginationParameter;

import com.franciaflex.faxtomail.persistence.entities.EmailTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.EtatAttente;
import com.franciaflex.faxtomail.persistence.entities.MailFilter;
import com.franciaflex.faxtomail.persistence.entities.MailFilterTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.persistence.entities.MailFolderImpl;
import com.franciaflex.faxtomail.persistence.entities.MailFolderTopiaDao;
import com.franciaflex.faxtomail.services.FaxToMailServiceSupport;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 0.1
 */
public class MailFolderService extends FaxToMailServiceSupport {

    private static final Log log = LogFactory.getLog(MailFolderService.class);

    public MailFolder getFolderForRecipient(String recipient) {
        MailFilterTopiaDao filterTopiaDao = getPersistenceContext().getMailFilterDao();

        PaginationParameter tpb = PaginationParameter.of(0, -1, MailFilter.PROPERTY_POSITION, false);

        String query = "FROM " + MailFilter.class.getCanonicalName() + " WHERE :recipient LIKE " + MailFilter.PROPERTY_EXPRESSION;
        Map<String, Object> params = new HashMap<>();
        params.put("recipient", recipient);

        Collection<MailFilter> filters = filterTopiaDao.find(query, params, tpb);
        MailFolder result;
        if (CollectionUtils.isNotEmpty(filters)) {
            result = filters.iterator().next().getMailFolder();
        } else {
            result = null;
        }
        return result;
    }

    public List<MailFolder> getAllMailFolders() {
        MailFolderTopiaDao dao = getPersistenceContext().getMailFolderDao();
        return new ArrayList<MailFolder>(dao.findAll());
    }
    
    public List<MailFolder> getRootMailFolders() {
        MailFolderTopiaDao dao = getPersistenceContext().getMailFolderDao();
        List<MailFolder> result = dao.forParentEquals(null).findAll();
        return result;
    }

    public List<MailFolder> getMailFolders(Collection<String> ids) {
        MailFolderTopiaDao dao = getPersistenceContext().getMailFolderDao();
        return new ArrayList<MailFolder>(dao.forTopiaIdIn(ids).findAll());
    }

    public void saveMailFolders(Collection<MailFolder> newMailFolders) {
        // get current folders
        MailFolderTopiaDao dao = getPersistenceContext().getMailFolderDao();
        List<MailFolder> mailFolders = dao.findAll();
        Map<String, MailFolder> mailFolderMap = new HashMap<>(Maps.uniqueIndex(mailFolders, TopiaEntities.getTopiaIdFunction()));

        // recursive update
        saveMailFolders(dao, mailFolderMap, null, newMailFolders);

        // if map is not empty after recursive iteration, remaining folder must be deleted
        dao.deleteAll(mailFolderMap.values());

        getPersistenceContext().commit();
    }
    
    protected Collection<MailFolder> saveMailFolders(MailFolderTopiaDao dao, Map<String, MailFolder> mailFolderMap,
            MailFolder parent, Collection<MailFolder> mailFolders) {
        
        Collection<MailFolder> result = Lists.newArrayList();
        if (mailFolders == null) {
            return result;
        }

        Binder<MailFolder, MailFolder> binderMailFolder = BinderFactory.newBinder(MailFolder.class);
        for (MailFolder mailFolder : mailFolders) {

            MailFolder currentMailFolder;
            if (StringUtils.isBlank(mailFolder.getTopiaId()) || mailFolder.getTopiaId().startsWith("new_")) {
                currentMailFolder = new MailFolderImpl();
            } else {
                currentMailFolder = mailFolderMap.remove(mailFolder.getTopiaId());
            }
            
            binderMailFolder.copyExcluding(mailFolder, currentMailFolder,
                    MailFolder.PROPERTY_TOPIA_ID,
                    MailFolder.PROPERTY_TOPIA_CREATE_DATE,
                    MailFolder.PROPERTY_TOPIA_VERSION,
                    MailFolder.PROPERTY_CHILDREN,
                    MailFolder.PROPERTY_PARENT);

            currentMailFolder.setParent(parent);

            if (!currentMailFolder.isPersisted()) {
                currentMailFolder = dao.create(currentMailFolder);
            }

            Collection<MailFolder> children = saveMailFolders(dao, mailFolderMap, currentMailFolder, mailFolder.getChildren());
            currentMailFolder.setChildren(children);
            dao.update(currentMailFolder);

            result.add(currentMailFolder);
        }

        return result;
    }

    public Collection<MailFolder> getFoldersWithEtatAttente(EtatAttente etatAttente) {
        Collection<MailFolder> result = new HashSet<>();

        MailFolderTopiaDao dao = getPersistenceContext().getMailFolderDao();
        List<MailFolder> roots = dao.forParentEquals(null).findAll();

        for (MailFolder root : roots) {
            result.addAll(getFoldersWithEtatAttente(root, new HashSet<EtatAttente>(), etatAttente));
        }

        return result;
    }

    protected Collection<MailFolder> getFoldersWithEtatAttente(MailFolder folder,
                                                               Collection<EtatAttente> parentEtatAttente,
                                                               EtatAttente etatAttente) {
        Collection<EtatAttente> etatAttentes = folder.getEtatAttentes();
        if (CollectionUtils.isEmpty(etatAttentes)) {
            etatAttentes = parentEtatAttente;
        } else {
            parentEtatAttente = etatAttentes;
        }

        Collection<MailFolder> result = new HashSet<>();
        Collection<MailFolder> children = folder.getChildren();
        if (children != null) {
            for (MailFolder child : children) {
                result.addAll(getFoldersWithEtatAttente(child, parentEtatAttente, etatAttente));
            }
        }

        if (etatAttentes.contains(etatAttente)) {
            result.add(folder);
        }

        return result;

    }

    /**
     * Retourne une map avec les identifiants des {@code mailFolder}s et le nombre de mails qu'ils
     * contiennent pour pouvoir empecher la suppression.
     * Seulement pour les dossiers qui ont au moins un email.
     * 
     * @return le nombre de mail par dossier
     */
    public Map<String, Long> getMailFoldersUsage() {
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();
        Map<String, Long> result = dao.getMailCountByFolder();
        return result;
    }

}
