/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyExtension.java 832 2011-04-22 16:37:22Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.1/wikitty-api/src/main/java/org/nuiton/wikitty/entities/WikittyExtension.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty.entities;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.nuiton.wikitty.WikittyUtil;

/**
 *
 * @author poussin
 * @version $Revision: 832 $
 *
 * Last update: $Date: 2011-04-22 18:37:22 +0200 (ven., 22 avril 2011) $
 * by : $Author: bpoussin $
 */
public class WikittyExtension implements Serializable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -3598621577607442972L;

    /** Field name pattern only word character [a-zA-Z_0-9] is accepted */
    static protected Pattern fieldNamePattern = Pattern.compile("^\\w+$");

    /**
     * Property change support.
     * 
     * Warning, this transient field is null after deserialization.
     */
    protected transient PropertyChangeSupport propertyChangeSupport;

    /** Name of this extension. */
    protected String name;

    /**
     * Name of other extension needed to put this extension to object.
     * 
     * Warning : Multiples extensions are not supported yet.
     * 
     */
    protected String requires;

    /**
     * use to know version objet, when you change field number, type or other
     * you must change version number.
     */
    protected String version = WikittyUtil.DEFAULT_VERSION;

    /** used to store tag/value used by client side ex: updatedDate=101212 */
    protected Map<String, String> tagValues = new HashMap<String, String>();

    /**
     * fields use ordered map, to keep order insertion of field
     * key: field name
     * value: field type
     */
    protected LinkedHashMap<String, FieldType> fields = new LinkedHashMap<String, FieldType>();

    /**
     * Default constructor.
     * 
     * Used by hibernate.
     */
    public WikittyExtension() {
        
    }

    public WikittyExtension(String name) {
        setName(name);
    }

    public WikittyExtension(String name, String version,
            String requires, LinkedHashMap<String, FieldType> fields) {
        if (version == null) {
            throw new IllegalArgumentException("Version must not be null");
        }
        setName(name);
        this.version = WikittyUtil.normalizeVersion(version);
        this.requires = requires;
        if (fields != null) {
            for (Map.Entry<String, FieldType> entry : fields.entrySet()) {
                String fieldName = entry.getKey();
                FieldType fieldType = entry.getValue();
                addField(fieldName, fieldType);
            }
        }
    }

    protected PropertyChangeSupport getPropertyChangeSupport() {
        if (propertyChangeSupport == null) {
            propertyChangeSupport = new PropertyChangeSupport(this);
        }
        return propertyChangeSupport;
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        getPropertyChangeSupport().addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        getPropertyChangeSupport().removePropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName,
            PropertyChangeListener listener) {
        getPropertyChangeSupport().addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(String propertyName,
            PropertyChangeListener listener) {
        getPropertyChangeSupport().removePropertyChangeListener(propertyName, listener);
    }

    public WikittyExtension cloneForUpgrade() {
        String nextRevision = WikittyUtil.incrementMajorRevision(getVersion());
        
        LinkedHashMap<String, FieldType> nextFields = null;
        if (fields != null) {
            nextFields = new LinkedHashMap<String, FieldType>();
            for (Map.Entry<String, FieldType> entry : fields.entrySet()) {
                FieldType type = entry.getValue();
                FieldType nextType = new FieldType(
                        type.getType(), type.getLowerBound(), type.getUpperBound());
                Set<String> tagNames = type.getTagNames();
                if (tagNames != null) {
                    for (String tagName : tagNames) {
                        String tagValue = type.getTagValue(tagName);
                        nextType.addTagValue(tagName, tagValue);
                    }
                }
                nextFields.put(entry.getKey(), nextType);
            }
        }

        WikittyExtension result = new WikittyExtension(name, nextRevision, requires, nextFields);
        return result;
    }

    /**
     * Compute id for extension name and version in argument.
     * 
     * @param name extension name
     * @param version extension version
     * @return extension string id
     */
    static public String computeId(String name, String version) {
        String result = name + "[" + version + "]";
        return result;
    }

    /**
     * Extract name from extension id
     *
     * @param id id like MonExtension[3.0]
     * @return extension name. Example 'MonExtension'
     */
    static public String computeName(String id) {
        int i = id.lastIndexOf("[");
        String result = id;
        if (i != -1) {
            result = id.substring(0, i);
        }
        return result;
    }

    /**
     * Extract extension version from extension id. If id contains no version
     * this method return '0.0'.
     *
     * @param id id like MonExtension[3.0]
     * @return extension version. Example '3.0'
     */
    static public String computeVersion(String id) {
        int b = id.lastIndexOf("[");
        int e = id.lastIndexOf("]");
        String result = null;
        if (b != -1 && e != -1) {
            result = id.substring(b+1, e);
        }
        result = WikittyUtil.normalizeVersion(result);
        return result;
    }

    /**
     * Extract extension name from fully qualified field name
     *
     * @param fqFieldName fully qualified field name like 'WikittyUser.login'
     * @return return extension name. Example 'WikittyUser'
     * @throws WikittyException if bad fqFieldName format
     */
    static public String extractExtensionName(String fqFieldName) {
        int i = fqFieldName.indexOf(WikittyUtil.FQ_FIELD_NAME_SEPARATOR);
        if (i > 0) {
            String result = fqFieldName.substring(0, i);
            return result;
        } else {
            throw new IllegalArgumentException(String.format(
                    "Your argument '%s' is not fully qualified field name", fqFieldName));
        }
    }

    /**
     * Extract field name from fully qualified field name (suppression [n/m] if
     * field is collection
     *
     * @param fqFieldName fully qualified field name like 'WikittyUser.login'
     * @return return field name. Example 'login'
     */
    static public String extractFieldName(String fqFieldName) {
        int i = fqFieldName.indexOf(WikittyUtil.FQ_FIELD_NAME_SEPARATOR);
        if (i > 0) {
            String result = fqFieldName.substring(i+1);
            int b = result.lastIndexOf("[");
            if (b > 0) {
                result = result.substring(0, b);
            }

            return result;
        } else {
            throw new IllegalArgumentException(String.format(
                    "Your argument '%s' is not fully qualified field name", fqFieldName));
        }
    }

    public String getId() {
        String result = computeId(getName(), getVersion());
        return result;
    }

    public String getName() {
        return name;
    }

    /**
     * Set extension name.
     * 
     * Check for invalid extension name (non alphanumeric characters).
     * 
     * @param name name
     */
    public void setName(String name) {

        // check alphanumeric characters
        if (name == null) {
            throw new IllegalArgumentException("Name must not be null");
        }
        if (!name.matches("\\w+")) {
            throw new IllegalArgumentException("Name contains non alphanumeric characters");
        }

        this.name = name;
    }

    public String getVersion() {
        return version;
    }

    public String getRequires() {
        return requires;
    }
    
    public FieldType getFieldType(String fieldName) {
        return fields.get(fieldName);
    }

    public Collection<String> getFieldNames() {
        Collection<String> result = fields.keySet();
        return result;
    }

    public void addField(String fieldName, FieldType type) {
        Matcher matcher = fieldNamePattern.matcher(fieldName);
        if(matcher.find()) {
            fields.put(fieldName, type);
            // TODO EC20100610 null for old value
            getPropertyChangeSupport().firePropertyChange("fields", null, fields);
        } else {
            throw new IllegalArgumentException("For field name [" + fieldName +"], only word character [a-zA-Z_0-9] is accepted");
        }
    }

    public void removeField(String fieldName) {
        fields.remove(fieldName);
        // TODO EC20100610 null for old value
        getPropertyChangeSupport().firePropertyChange("fields", null, fields);
    }

    @Override
    public int hashCode() {
        return getId().hashCode();
    }

    public void addTagValue(String tag, String value) {
        tagValues.put(tag, value);
        // TODO EC20100610 null for old value
        getPropertyChangeSupport().firePropertyChange("tagValues", null, tagValues);
    }

    public String getTagValue(String tag) {
        String result = tagValues.get(tag);
        return result;
    }

    public Set<String> getTagNames() {
        return tagValues.keySet();
    }

    public Map<String, String> getTagValues() {
        return tagValues;
    }

    public void setTagValues(Map<String, String> tagValues) {
        Map<String, String> oldValue = this.tagValues;
        this.tagValues = tagValues;
        getPropertyChangeSupport().firePropertyChange("tagValues", oldValue, tagValues);
    }

    @Override
    public boolean equals(Object obj) {
        boolean result = false;
        if (obj instanceof WikittyExtension) {
            WikittyExtension other = (WikittyExtension)obj;
            result = this.getId().equals(other.getId());
        }
        return result;
    }

    @Override
    public String toString() {
        return getId();
    }
    
    public String toDefinition() {
        String result = "Extension " + getId();
        result += WikittyUtil.tagValuesToString(tagValues);
        result += " {\n";
        for (String fieldName : fields.keySet()) {
            result += fields.get(fieldName).toDefinition(fieldName) + "\n";
        }
        result += "}";
        return result;
    }

}
