/*
 * #%L
 * Wikitty :: api
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.api;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.entities.ExtensionFactory;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyImpl;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;

/**
 * User: couteau
 * Date: 05/04/11
 */
public abstract class AbstractSearchTest {

    public static final String EXT_PRODUCT = "Product";
    public static final String EXT_CATEGORY = "Category";
    public static final String PRODUCT_PRICE = "price";
    public static final String PRODUCT_NAME = "name";
    public static final String PRODUCT_CATEGORY = "category";
    public static final String CATEGORY_NAME = "name";
    public static final String VERSION = "1.0";
    public static String W_ID ="";

    protected WikittyProxy proxy = new WikittyProxy(getWikittyService());

    @Before
    public void initData(){

        ///////////////////////////////////////////////////////////////////////
        //                   CREATE EXTENSIONS NEEDED FOR TESTS              //
        ///////////////////////////////////////////////////////////////////////

        //Create product extension
        WikittyExtension product = ExtensionFactory.create(EXT_PRODUCT, VERSION)
                .addField(PRODUCT_PRICE, FieldType.TYPE.NUMERIC)
                .addField(PRODUCT_NAME, FieldType.TYPE.STRING)
                .addField(PRODUCT_CATEGORY, FieldType.TYPE.WIKITTY)
                .extension();
        proxy.storeExtension(product);
        
        //Create category extension
        WikittyExtension category = ExtensionFactory.create(EXT_CATEGORY, VERSION)
                .addField(CATEGORY_NAME, FieldType.TYPE.STRING)
                .extension();
        proxy.storeExtension(category);

        ///////////////////////////////////////////////////////////////////////
        //                    CREATE CATEGORIES                              //
        ///////////////////////////////////////////////////////////////////////

        //Create wikitty category named "Hardware"
        Wikitty category1 = new WikittyImpl();
        category1.addExtension(category);
        category1.setField(EXT_CATEGORY, CATEGORY_NAME, "Hardware");
        proxy.store(category1);

        //Create wikitty category named "Wrong category"
        Wikitty category2 = new WikittyImpl();
        category2.addExtension(category);
        category2.setField(EXT_CATEGORY, CATEGORY_NAME, "Wrong category");
        proxy.store(category2);

        ///////////////////////////////////////////////////////////////////////
        //                    CREATE PRODUCTS                                //
        ///////////////////////////////////////////////////////////////////////

        //Create product named Paint, price is 20 category is category1
        Wikitty product1 = new WikittyImpl();
        product1.addExtension(product);
        product1.setField(EXT_PRODUCT, PRODUCT_NAME, "Paint");
        product1.setField(EXT_PRODUCT, PRODUCT_PRICE, 20);
        product1.setField(EXT_PRODUCT, PRODUCT_CATEGORY, category1.getId());
        proxy.store(product1);

        //Create product named Screwdriver, price is 3, category is category2
        Wikitty product2 = new WikittyImpl();
        product2.addExtension(product);
        product2.setField(EXT_PRODUCT, PRODUCT_NAME, "Screwdriver");
        product2.setField(EXT_PRODUCT, PRODUCT_PRICE, 3);
        product2.setField(EXT_PRODUCT, PRODUCT_CATEGORY, category2.getId());
        proxy.store(product2);

        //Create product named Paint Blue, price is 22, category is category2
        Wikitty product3 = new WikittyImpl();
        product3.addExtension(product);
        product3.setField(EXT_PRODUCT, PRODUCT_NAME, "Paint Blue");
        product3.setField(EXT_PRODUCT, PRODUCT_PRICE, 22);
        product3.setField(EXT_PRODUCT, PRODUCT_CATEGORY, category2.getId());
        proxy.store(product3);

        W_ID = product3.getId();

    }

    @Test
    public void testEq() throws Exception {

        Search query = Search.query().eq("Product.price", "3");

        Criteria eqCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(eqCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));


        //Test using pattern matching
        Search query2 = Search.query().eq("Product.name", "*dri*");

        Criteria eqCriteria2 = query2.criteria();

        PagedResult<Wikitty> results2 = proxy.findAllByCriteria(eqCriteria2);

        Assert.assertEquals(1, results2.getAll().size());

        Wikitty resultW2 = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW2.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW2.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));

        //Test using pattern matching
        Search query3 = Search.query().eq("Product.name", "*dri?er");

        Criteria eqCriteria3 = query3.criteria();

        PagedResult<Wikitty> results3 = proxy.findAllByCriteria(eqCriteria3);

        Assert.assertEquals(1, results3.getAll().size());

        Wikitty resultW3 = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW3.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW3.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));
    }

    @Test
    public void testNeq() throws Exception {
        Search query = Search.query().neq("Product.price", "3");

        Criteria neqCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(neqCriteria);

        Assert.assertEquals(4, results.getAll().size());

    }

    @Test
    public void testBw() throws Exception {

        Search query = Search.query().bw("Product.price", "3", "5");

        Criteria bwCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(bwCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));
    }

    @Test
    public void testLt() throws Exception {

        Search query = Search.query().lt("Product.price", "20");

        Criteria bwCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(bwCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));

    }

        @Test
    public void testLe() throws Exception {

        Search query = Search.query().le("Product.price", "20");

        Criteria bwCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(bwCriteria);

        Assert.assertEquals(2, results.getAll().size());
    }

        @Test
    public void testGt() throws Exception {

        Search query = Search.query().gt("Product.price", "20");

        Criteria bwCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(bwCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Paint Blue", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(22, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));

    }

        @Test
    public void testGe() throws Exception {

        Search query = Search.query().ge("Product.price", "20");

        Criteria bwCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(bwCriteria);

        Assert.assertEquals(2, results.getAll().size());

    }

    @Test
    public void testContains() throws Exception {

        Search query = Search.query().contains("Product.price", "3");

        Criteria inCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(inCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));

        //TODO JC-07-04-2011 do a test with multi-valued field

    }

    @Test
    public void testIn() throws Exception {

        Search query = Search.query().in("Product.price","1","2","3","4","5");

        Criteria inCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(inCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));
    }

    @Test
    public void testExteq() throws Exception {
        Search query = Search.query().exteq(EXT_PRODUCT);

        Criteria exteqCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(exteqCriteria);

        Assert.assertEquals(3, results.getAll().size());
    }

    @Test
    public void testExtneq() throws Exception {
        Search query = Search.query().extneq(EXT_PRODUCT);

        Criteria extneqCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(extneqCriteria);

        Assert.assertEquals(2, results.getAll().size());

    }

    @Test
    public void testIdeq() throws Exception {
        Search query = Search.query().ideq(W_ID);

        Criteria ideqCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(ideqCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Paint Blue", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(22, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));

    }

    @Test
    public void testIdneq() throws Exception {

        Search query = Search.query().idneq(W_ID);

        Criteria idneqCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(idneqCriteria);

        Assert.assertEquals(4, results.getAll().size());
    }

    @Test
    public void testUnlike() throws Exception {
        /*Search query = Search.query().unlike("Product.name", "*dri*");

        Criteria likeCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(likeCriteria);

        Assert.assertEquals(4, results.getAll().size());*/
    }

    @Test
    public void testSw() throws Exception {
        Search query = Search.query().sw("Product.name", "Scre");

        Criteria swCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(swCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));

    }

    @Test
    public void testNsw() throws Exception {

        Search query = Search.query().nsw("Product.name", "Scre");

        Criteria nswCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(nswCriteria);

        Assert.assertEquals(4, results.getAll().size());
    }

    @Test
    public void testEw() throws Exception {

        Search query = Search.query().ew("Product.name", "ver");

        Criteria ewCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(ewCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Screwdriver", resultW.getFieldAsString(EXT_PRODUCT,
                PRODUCT_NAME));
        Assert.assertEquals(3, resultW.getFieldAsInt(EXT_PRODUCT,
                PRODUCT_PRICE));

    }

    @Test
    public void testNotew() throws Exception {

        Search query = Search.query().notew("Product.name", "ver");

        Criteria notewCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(notewCriteria);

        Assert.assertEquals(4, results.getAll().size());

    }

    @Test
    public void testKeyword() throws Exception {

        Search query = Search.query().keyword("Paint");

        Criteria keywordCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(keywordCriteria);

        Assert.assertEquals(2, results.getAll().size());

    }

    @Test
    public void testIsNull() throws Exception {

        Search query = Search.query().isNull("Product.name");

        Criteria isNull = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(isNull);

        Assert.assertEquals(0, results.getAll().size());

    }

    @Test
    public void testIsNotNull() throws Exception {

        Search query = Search.query().isNotNull("Product.name");

        Criteria isNotNull = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(isNotNull);

        Assert.assertEquals(3, results.getAll().size());

    }

    @Test
    public void testFalse() throws Exception {

        Search query = Search.query().rFalse();

        Criteria rFalse = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(rFalse);

        Assert.assertEquals(0, results.getAll().size());

    }

    @Test
    public void testTrue() throws Exception {

        Search query = Search.query().rTrue();

        Criteria rTrue = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(rTrue);

        Assert.assertEquals(5, results.getAll().size());

    }

    @Test
    public void testAnd() throws Exception {

        Search query = Search.query().and().bw("Product.price", "15", "25")
                .sw("Product.name", "Paint");

        Criteria andCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(andCriteria);

        Assert.assertEquals(2, results.getAll().size());

    }

    @Test
    public void testOr() throws Exception {

        Search query = Search.query();

        query.or().gt("Product.price", "20").sw("Product.name", "Screw");

        Criteria bwCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(bwCriteria);

        Assert.assertEquals(2, results.getAll().size());

    }

    @Test
    public void testNot() throws Exception {

        Search query = Search.query();

        query.not().ge("Product.price", "20");

        Criteria bwCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(bwCriteria);

        Assert.assertEquals(3, results.getAll().size());
    }

    @Test
    public void testAssociated() throws Exception {

        /* Following request mean is :
         * Looking for any product which price is between 15 and 25,
         * and name starts with 'Paint',
         * and with an associated category which name is 'Hardware'
         */

        Search query = Search.query().bw("Product.price", "15", "25")
                .sw("Product.name", "Paint").associated("Product.category")
                .eq("Category.name", "Hardware");

        Criteria associatedCriteria = query.criteria();

        PagedResult<Wikitty> results = proxy.findAllByCriteria(associatedCriteria);

        Assert.assertEquals(1, results.getAll().size());

        Wikitty resultW = results.getFirst();

        Assert.assertEquals("Paint", resultW.getFieldAsString(EXT_PRODUCT, PRODUCT_NAME));
        Assert.assertEquals(20, resultW.getFieldAsInt(EXT_PRODUCT, PRODUCT_PRICE));



        //Test with an associated search that returns nothing
        query = Search.query().bw("Product.price", "15", "25")
                .sw("Product.name", "Paint").associated("Product.category")
                .eq("Category.name", "don't return anything");

        associatedCriteria = query.criteria();

        results = proxy.findAllByCriteria(associatedCriteria);

        Assert.assertEquals(0, results.getAll().size());
    }

    @Test
    public void testComplexQueries() throws Exception {
        //TODO JC-07-04-2011 Write test
    }

    //TODO JC-07-04-2011 Write test on sortBy, facets, and with other types (date,...)

    public abstract WikittyService getWikittyService();
}
