/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyProxy.java 646 2010-12-22 17:50:50Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.1/wikitty-api/src/main/java/org/nuiton/wikitty/WikittyProxy.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.TimeTrace;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.entities.BusinessEntityImpl;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.services.WikittyEvent;
import org.nuiton.wikitty.services.WikittyServiceEnhanced;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.TimeTrace;
import org.nuiton.wikitty.entities.WikittyTreeNode;

/**
 * Wikitty proxy is used to transform wikitty object used by {@link WikittyService}
 * into business objects used by applications.
 * 
 * It also manage {@link #securityToken} for {@link org.nuiton.wikitty.services.WikittyServiceSecurity}.
 * 
 * @author poussin
 * @version $Revision: 646 $
 *
 * Last update: $Date: 2010-12-22 18:50:50 +0100 (mer., 22 déc. 2010) $
 * by : $Author: bpoussin $
 */
public class WikittyProxy {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyProxy.class);

    static protected TimeTrace timeTrace = new TimeTrace();

    /** Delegated wikitty service. */
    protected WikittyServiceEnhanced wikittyService;

    /**
     * Security token.
     * 
     * @see org.nuiton.wikitty.services.WikittyServiceSecurity#login(String, String)
     */
    protected String securityToken;

    /**
     * Empty constructor (uninitialized wikittyService).
     */
    public WikittyProxy() {
    }

    public WikittyProxy(ApplicationConfig config) {
        if (config != null) {
            long timeToLogInfo = config.getOptionAsInt(WikittyConfig.WikittyOption.
                    WIKITTY_PROXY_TIME_TO_LOG_INFO.getKey());
            long timeToLogWarn = config.getOptionAsInt(WikittyConfig.WikittyOption.
                    WIKITTY_PROXY_TIME_TO_LOG_WARN.getKey());
            timeTrace.setTimeToLogInfo(timeToLogInfo);
            timeTrace.setTimeToLogWarn(timeToLogWarn);
        }
    }

    /**
     * Constructor with wikittyService.
     * 
     * @param wikittyService wikitty service
     */
    public WikittyProxy(WikittyService wikittyService) {
        this();
        setWikittyService(wikittyService);
    }

    public WikittyProxy(ApplicationConfig config, WikittyService wikittyService) {
        this(config);
        setWikittyService(wikittyService);
    }

    static public TimeTrace getTimeTrace() {
        return timeTrace;
    }

    static public Map<String, TimeTrace.CallStat> getCallCount() {
        return timeTrace.getCallCount();
    }

    public void login(String login, String password) {
        long start = TimeTrace.getTime();
        String result = wikittyService.login(login, password);
        setSecurityToken(result);
        
        timeTrace.add(start, "login");
    }

    public void logout() {
        long start = TimeTrace.getTime();
        wikittyService.logout(securityToken);
        
        timeTrace.add(start, "logout");
    }

    public String getSecurityToken() {
        return securityToken;
    }

    public void setSecurityToken(String securityToken) {
        this.securityToken = securityToken;
    }

    public WikittyService getWikittyService() {
        return wikittyService.getDelegate();
    }

    public void setWikittyService(WikittyService wikittyService) {
        this.wikittyService = new WikittyServiceEnhanced(wikittyService);
    }

    /**
     * Cast Business object to another Business Object
     * If source don't have target extension, this extension is added
     *
     * @param source business entity source
     * @param target business entity class wanted
     * @return new instance of object wanted
     */
    public <E extends BusinessEntity> E cast(BusinessEntity source, Class<E> target) {
        long start = TimeTrace.getTime();
        E result = WikittyUtil.newInstance(
                securityToken, wikittyService, target, ((BusinessEntityImpl)source).getWikitty());
        
        timeTrace.add(start, "cast");
        return result;
    }

    public <E extends BusinessEntity> E store(E e) {
        Wikitty w = ((BusinessEntityImpl)e).getWikitty();
        w = store(w);
        return e;
    }

    public Wikitty store(Wikitty w) {
        long start = TimeTrace.getTime();
        WikittyEvent resp = wikittyService.store(securityToken, w);
        // update object
        resp.update(w);
        
        timeTrace.add(start, "store");
        return w;
    }


    /**
     * Store to WikittyService objects.
     * 
     * @param <E> object type
     * @param objets list
     * @return updated objects list
     */
    public <E extends BusinessEntity> List<E> store(List<E> objets) {
        long start = TimeTrace.getTime();
        // prepare data to send to service
        List<Wikitty> wikitties = new ArrayList<Wikitty>(objets.size());
        for (E e : objets) {
            Wikitty w = ((BusinessEntityImpl)e).getWikitty();
            wikitties.add(w);
        }

        // call the service with Wikitty
        WikittyEvent resp = wikittyService.store(securityToken, wikitties);

        // update object
        for (Wikitty w : wikitties) {
            resp.update(w);
        }

        timeTrace.add(start, "store<list>");
        return objets;
    }

    public  List<Wikitty> storeWikitty(List<Wikitty> wikitties) {
        long start = TimeTrace.getTime();

        // call the service with Wikitty
        WikittyEvent resp = wikittyService.store(securityToken, wikitties);

        // update object
        for (Wikitty w : wikitties) {
            resp.update(w);
        }

        timeTrace.add(start, "storeWikitty<list>");
        return wikitties;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param id entity id
     * @param checkExtension if true check that Wikitty result has all extension
     * declared in clazz
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public <E extends BusinessEntity> E restore(Class<E> clazz, String id, boolean checkExtension) {
        try {
            long start = TimeTrace.getTime();
            E result = null;
            if (id != null) {
                HashSet<String> extNames = null;

                Wikitty wikitty = wikittyService.restore(securityToken, id);
                if (wikitty != null) {
                    if (checkExtension) {
                        extNames = new HashSet<String>(wikitty.getExtensionNames());
                    }
                    result = WikittyUtil.newInstance(
                            securityToken, wikittyService, clazz, wikitty);

                    if (checkExtension) {
                        // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
                        BusinessEntityImpl b = (BusinessEntityImpl) result;
                        Collection<WikittyExtension> BusinessEntityStaticExtensions = b.getStaticExtensions();
                        for (WikittyExtension ext : BusinessEntityStaticExtensions) {
                            String extensionName = ext.getName();
                            if (!extNames.contains(extensionName)) {
                                // extension wanted by BusinessEntity (clazz)
                                // is not in wikitty, then wikitty is not good type
                                // for business
                                result = null;
                                break;
                            }
                        }
                    }
                }
            }
            
            timeTrace.add(start, "restore<Business>");
            return result;
        } catch (SecurityException eee) {
            throw eee;
        } catch (Exception eee) {
            throw new WikittyException("Can't restore wikitty", eee);
        }
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     * 
     * @param id entity id
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public Wikitty restore(String id) {
        long start = TimeTrace.getTime();
        Wikitty result = null;
        if (id != null) {
            result = wikittyService.restore(securityToken, id);
        }
        
        timeTrace.add(start, "restore");
    	return result;
    }
    
    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param id entity id
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public <E extends BusinessEntity> E restore(Class<E> clazz, String id) {
        E result = restore(clazz, id, false);
        return result;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be be found.
     *
     * @param id entity id
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public List<Wikitty> restore(List<String> id) {
        long start = TimeTrace.getTime();

        List<Wikitty> result = wikittyService.restore(securityToken, id);

        timeTrace.add(start, "restoreWikitty<list>");
        return result;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be be found.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param id entity id
     * @param checkExtension if true check that Wikitty result has all extension
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public <E extends BusinessEntity> List<E> restore(
            Class<E> clazz, List<String> id, boolean checkExtension) {
        long start = TimeTrace.getTime();

        List<Wikitty> wikitties = wikittyService.restore(securityToken, id);
        List<E> result = new ArrayList<E>();
        for (Wikitty w : wikitties) {
            HashSet<String> extNames = null;
            if (checkExtension) {
                extNames = new HashSet<String>(w.getExtensionNames());
            }

            E dto = WikittyUtil.newInstance(
                    securityToken, wikittyService, clazz, w);
            if (checkExtension) {
                // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
                BusinessEntityImpl b = (BusinessEntityImpl) dto;
                for (WikittyExtension ext : b.getStaticExtensions()) {
                    if (!extNames.contains(ext.getName())) {
                        // extension wanted by BusinessEntity (clazz)
                        // is not in wikitty, then wikitty is not good type
                        // for business
                        result = null;
                        break;
                    }
                }
                // le break au dessus permet de soritr d'une boucle
                // met si result est null il faut sortir de la methode et donc
                // sortir de la deuxieme boucle
                if (result == null) {
                    break;
                }
            }
            result.add(dto);
        }
        
        timeTrace.add(start, "restore<list>");
        return result;
    }

    public <E extends BusinessEntity> List<E> restore(Class<E> clazz, List<String> id) {
        List<E> result = restore(clazz, id, false);
        return result;
    }

    public void delete(String id) {
        long start = TimeTrace.getTime();
        wikittyService.delete(securityToken, id);
        
        timeTrace.add(start, "delete");
    }

    public void delete(Collection<String> ids) {
        long start = TimeTrace.getTime();
        wikittyService.delete(securityToken, ids);
        
        timeTrace.add(start, "delete<list>");
    }

    /**
     * Null field are not used in search request.
     *
     * @param e sample wikitty
     * @param firstIndex
     * @param endIndex
     * @param fieldFacet
     * @return
     */
    public <E extends BusinessEntityImpl> PagedResult<E> findAllByExample(E e,
            int firstIndex, int endIndex, String ... fieldFacet ) {
        long start = TimeTrace.getTime();

        Criteria criteria = Search.query(e.getWikitty()).criteria()
                .setFirstIndex(firstIndex).setEndIndex(endIndex)
                .setFacetField(fieldFacet);

        PagedResult<String> pagedResult = wikittyService.findAllByCriteria(
                securityToken, criteria);
        // we can use autoconvert = true because search by example add automaticaly
        // restriction on extension
        PagedResult<E> result = (PagedResult<E>)pagedResult.cast(
                this, e.getClass(), true);
        
        timeTrace.add(start, "findAllByExample<limit>");
        return result;
    }

    /**
     * Null field are not used in search request.
     * 
     * @param e sample wikitty
     * @return
     */
    public <E extends BusinessEntityImpl> E findByExample(E e) {
        long start = TimeTrace.getTime();
        Criteria criteria = Search.query(e.getWikitty()).criteria();

        String id = wikittyService.findByCriteria(securityToken, criteria);
        E result = null;
        if (id != null) {
            Wikitty w = wikittyService.restore(securityToken, id);
            result = (E) WikittyUtil.newInstance(
                    securityToken, wikittyService, e.getClass(), w);
        }
        
        timeTrace.add(start, "findByExample");
        return result;
    }

    /**
     * Search object that correspond to criteria and that have all extension
     * needed by BusinessEntity (clazz). If criteria is null, find all extensions
     * else if criteria is empty return nothing.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param criteria criteria
     * @return paged result
     */
    public <E extends BusinessEntity> PagedResult<E> findAllByCriteria(
            Class<E> clazz, Criteria criteria) {
        long start = TimeTrace.getTime();

        // newInstance only return BusinessEntityWikittyImpl
        BusinessEntityImpl sample =
                (BusinessEntityImpl)WikittyUtil.newInstance(clazz);

        Wikitty wikitty = sample.getWikitty();
        Collection<String> extensions = wikitty.getExtensionNames();

        Criteria serviceCriteria;
        if(criteria == null) {
            Search search = Search.query();
            search = search.eq(Element.ELT_EXTENSION, extensions);

            serviceCriteria = search.criteria();
        } else {
            Search search = Search.query(criteria);
            search = search.eq(Element.ELT_EXTENSION, extensions);

            serviceCriteria = search.criteria(criteria.getName());
            serviceCriteria.setFirstIndex(criteria.getFirstIndex());
            serviceCriteria.setEndIndex(criteria.getEndIndex());

            List<String> facetField = criteria.getFacetField();
            if(facetField != null) {
                String[] facetFieldTab = new String[facetField.size()];
                serviceCriteria.setFacetField(facetField.toArray(facetFieldTab));
            }

            List<Criteria> facetCriteria = criteria.getFacetCriteria();
            if(facetCriteria != null) {
                Criteria[] facetCriteriaTab = new Criteria[facetCriteria.size()];
                serviceCriteria.setFacetCriteria(facetCriteria.toArray(facetCriteriaTab));
            }

            List<String> sortAscending = criteria.getSortAscending();
            if(sortAscending != null) {
                String[] sortAscendingTab = new String[sortAscending.size()];
                serviceCriteria.setSortAscending(sortAscending.toArray(sortAscendingTab));
            }

            List<String> sortDescending = criteria.getSortDescending();
            if(sortDescending != null) {
                String[] sortDescendingTab = new String[sortDescending.size()];
                serviceCriteria.setSortDescending(sortDescending.toArray(sortDescendingTab));
            }
        }

        PagedResult<String> pagedResult = wikittyService.findAllByCriteria(
                securityToken, serviceCriteria);
        
        // we can cast with autoconvert true because we have add restriction
        // on extension
        PagedResult<E> result = (PagedResult<E>)pagedResult.cast(
                this, sample.getClass(), true);
        
        timeTrace.add(start, "findAllByCriteria<Business>");
        return result;
    }

    public PagedResult<Wikitty> findAllByCriteria(Criteria criteria) {
        long start = TimeTrace.getTime();
    	PagedResult<String> resultId = wikittyService.findAllByCriteria(securityToken, criteria);
    	PagedResult<Wikitty> result = resultId.cast(securityToken, wikittyService);

        timeTrace.add(start, "findAllByCriteria");
    	return result;
    }

    public PagedResult<String> findAllIdByCriteria(Criteria criteria) {
        long start = TimeTrace.getTime();
    	PagedResult<String> result = wikittyService.findAllByCriteria(securityToken, criteria);

        timeTrace.add(start, "findAllByCriteria");
    	return result;
    }

    public <E extends BusinessEntity> E findByCriteria(Class<E> clazz, Criteria criteria) {
        long start = TimeTrace.getTime();
        BusinessEntityImpl sample =
                (BusinessEntityImpl)WikittyUtil.newInstance(clazz);

        Wikitty wikitty = sample.getWikitty();
        Collection<String> extensions = wikitty.getExtensionNames();
        Search search = Search.query(criteria);
        search = search.eq(Element.ELT_EXTENSION, extensions);

        criteria = search.criteria(criteria.getName());

        String id = wikittyService.findByCriteria(securityToken, criteria);
        Wikitty w = wikittyService.restore(securityToken, id);
        E result = WikittyUtil.newInstance(
                securityToken, wikittyService, clazz, w);
        
        timeTrace.add(start, "findByCriteria<Business>");
        return result;
    }

    public Wikitty findByCriteria(Criteria criteria) {
        long start = TimeTrace.getTime();
        String id = wikittyService.findByCriteria(securityToken, criteria);
        Wikitty wikitty = wikittyService.restore(securityToken, id);
        timeTrace.add(start, "findByCriteria");
    	return wikitty;
    }
    
    public WikittyTree restoreTree(String wikittyId) {
        long start = TimeTrace.getTime();
        WikittyTree result = wikittyService.restoreTree(securityToken, wikittyId);
        
        timeTrace.add(start, "restoreTree");
        return result;
    }

    /**
     * Delete specified tree node and all sub nodes.
     * 
     * @param treeNodeId tree node id to delete
     * @return {@true} if at least one node has been deleted
     */
    public WikittyEvent deleteTree(String treeNodeId) {
        long start = TimeTrace.getTime();
        WikittyEvent result = wikittyService.deleteTree(securityToken,treeNodeId);
        
        timeTrace.add(start, "deleteTree");
        return result;
    }

    public <E extends BusinessEntity> Map.Entry<E, Integer> restoreNode(
            Class<E> clazz, String wikittyId, Criteria filter) {
        Entry<E, Integer> result = restoreNode(clazz, wikittyId, filter, false);
        return result;
    }

    public <E extends BusinessEntity> Map.Entry<E, Integer> restoreNode(
            Class<E> clazz, String wikittyId, Criteria filter, boolean checkExtension) {
        long start = TimeTrace.getTime();
        Map.Entry<E, Integer> result = null;

        Map.Entry<String, Integer> node = wikittyService.restoreNode(
                securityToken, wikittyId, filter);
        if (node != null) {
            E bean = restore(clazz, wikittyId, checkExtension);
            if (bean != null) {
                result = new HashMap.SimpleEntry<E, Integer>(bean, node.getValue());
            }
        }
        
        timeTrace.add(start, "restoreNode");
        return result;
    }

    public <E extends BusinessEntity> Map<E, Integer> restoreChildren(
            Class<E> clazz, String wikittyId, Criteria filter) {
        Map<E, Integer> result = restoreChildren(clazz, wikittyId, filter, false);
        return result;
    }

    public <E extends BusinessEntity> Map<E, Integer> restoreChildren(
            Class<E> clazz, String wikittyId, Criteria filter, boolean checkExtension) {
        long start = TimeTrace.getTime();

        Map<E, Integer> convertedResult = null;
        
        Map<String, Integer> result = wikittyService.restoreChildren(
                securityToken, wikittyId, filter);

        if(result != null) {
            List<String> ids = new ArrayList<String>(result.keySet());
            List<E> keys = restore(clazz, ids, checkExtension);

            convertedResult = new LinkedHashMap<E, Integer>();
            for (E e : keys) {
                Integer size = result.get(e.getWikittyId());
                convertedResult.put(e, size);
            }
        }
        
        timeTrace.add(start, "restoreChildren");
        return convertedResult;
    }

    public Wikitty restoreVersion(String wikittyId, String version) {
        long start = TimeTrace.getTime();
        Wikitty result = wikittyService.restoreVersion(
                securityToken, wikittyId, version);
        
        timeTrace.add(start, "restoreVersion");
        return result;
    }

    public <E extends BusinessEntity> boolean hasType(Class<E> clazz, String wikittyId) {
        try {
            long start = TimeTrace.getTime();
            
            boolean result = true;

            E businessObject;
            HashSet<String> extNames;
            Wikitty wikitty = wikittyService.restore(securityToken, wikittyId);
            if (wikitty == null) {
                result = false;
            } else {
                extNames = new HashSet<String>(wikitty.getExtensionNames());
                businessObject = WikittyUtil.newInstance(
                        securityToken, wikittyService, clazz, wikitty);
                // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
                BusinessEntityImpl b = (BusinessEntityImpl) businessObject;
                for (WikittyExtension ext : b.getStaticExtensions()) {
                    if (!extNames.contains(ext.getName())) {
                        // extension wanted by BusinessEntity (clazz)
                        // is not in wikitty, then wikitty is not good type
                        // for business
                        result = false;
                        break;
                    }
                }
            }
            
            timeTrace.add(start, "hasType");
            return result;
        } catch (SecurityException eee) {
            throw eee;
        } catch (Exception eee) {
            throw new WikittyException(
                    "Can't retrieve wikitty needed for hasType test", eee);
        }
    }

    /**
     * Manage Update and creation.
     *
     * @param ext extension to be persisted
     * @return update response
     */
    public WikittyEvent storeExtension(WikittyExtension ext) {
        long start = TimeTrace.getTime();
        WikittyEvent response =
                wikittyService.storeExtension(securityToken, ext);
        
        timeTrace.add(start, "storeExtension");
        return response;
    }

    /**
     * Manage Update and creation.
     *
     * @param exts list of wikitty extension to be persisted
     * @return update response
     */
    public WikittyEvent storeExtension(Collection<WikittyExtension> exts) {
        long start = TimeTrace.getTime();
        WikittyEvent response =
                wikittyService.storeExtension(securityToken, exts);
        
        timeTrace.add(start, "storeExtension<list>");
        return response;
    }

    /**
     * Load extension from id. Id is 'name[version]'.
     * 
     * @param extensionId extension id to restore
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtension(String extensionId) {
        long start = TimeTrace.getTime();
        WikittyExtension extension = wikittyService.restoreExtension(securityToken, extensionId);
        
        timeTrace.add(start, "restoreExtension");
        return extension;
    }

    /**
     * Search extension with name in last version.
     * 
     * @param extensionName extension name
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtensionLastVersion(String extensionName) {
        long start = TimeTrace.getTime();
        WikittyExtension extension = wikittyService.restoreExtensionLastVersion(securityToken, extensionName);
        
        timeTrace.add(start, "restoreExtensionLastVersion");
        return extension;
    }

    public void deleteExtension(String extName) {
        long start = TimeTrace.getTime();
        wikittyService.deleteExtension(securityToken, extName);

        timeTrace.add(start, "deleteExtension");
    }

    public void deleteExtension(Collection<String> extNames) {
        long start = TimeTrace.getTime();
        wikittyService.deleteExtension(securityToken, extNames);

        timeTrace.add(start, "deleteExtension<list>");
    }

    /**
     * Return all extension id (ex: "extName[version])").
     * 
     * @return extension id list
     */
    public List<String> getAllExtensionIds() {
        long start = TimeTrace.getTime();
        List<String> result = wikittyService.getAllExtensionIds(securityToken);
        
        timeTrace.add(start, "getAllExtensionIds");
        return result;
    }
    
    /**
     * Return all extension id (ex: "extName[version])") where
     * {@code extensionName} is required.
     * 
     * @param extensionName extension name
     * @return extensions
     */
    public List<String> getAllExtensionsRequires(String extensionName) {
        long start = TimeTrace.getTime();
        List<String> result = wikittyService.getAllExtensionsRequires(securityToken, extensionName);
        
        timeTrace.add(start, "getAllExtensionsRequires");
        return result;
    }

    /**
     * Use with caution : It will delete ALL indexes from search engine !
     * This operation should be disabled in production environment.
     */
    public WikittyEvent clear() {
        long start = TimeTrace.getTime();
        WikittyEvent result = wikittyService.clear(securityToken);
        
        timeTrace.add(start, "clear");
        return result;
    }

    /**
     * Synchronize search engine with wikitty storage engine, i.e. clear and
     * reindex all wikitties.
     */
    public void syncSearchEngine() {
        long start = TimeTrace.getTime();
        wikittyService.syncSearchEngine(securityToken);
        
        timeTrace.add(start, "syncSearchEngine");
    }

    /**
     * Method to get the Wikitty encapsulated into a BusinessEntity
     * @param entity the BusinessEntity encapsulating the Wikitty
     * @return the wikitty encapsulated
     */
    public Wikitty getWikitty(BusinessEntity entity){
        long start = TimeTrace.getTime();
        Wikitty result = WikittyUtil.getWikitty(wikittyService, securityToken, entity);
        
        timeTrace.add(start, "getWikitty");
        return result;
    }

}
