package org.nuiton.wikitty.search;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.Wikitty;
import org.nuiton.wikitty.WikittyUtil;

/**
 * Helper to create a criteria with a restriction
 *
 * Element :
 * <extensionName>.<fieldName>[.<fieldType>] : search on an extension and field with specific type (optional)
 * Criteria.ALL_EXTENSIONS.<fieldName>.<fieldType> : search on all extension and field name with specific type
 *
 * <fieldType> specify search on field as NUMERIC, STRING, WIKITTY, BOOLEAN, DATE. Use Element.ElementType to specify type.
 */
public class Search {

    /** Default operator type between all search condition. */
    public enum KIND {
        AND, OR, NOT
    }

    /** Defaut kind to {@link KIND#AND}. */
    protected KIND kind = KIND.AND;

    protected List<Restriction> restrictions;

    protected List<Search> subSearchs;

    public Search() {
        restrictions = new ArrayList<Restriction>();
        subSearchs = new ArrayList<Search>();
    }

    /**
     * Create Search query with field in wikitty argument.
     *
     * @param wikitty example use to create query
     * @return query
     */
    static public Search query(Wikitty wikitty) {
        Search result = Search.query();
        result.kind = KIND.AND;
        // result object must have same extension that wikitty example
        for (String extName : wikitty.getExtensionNames()) {
            result.eq(Element.ELT_EXTENSION, extName);
        }

        for (String fqfieldName : wikitty.fieldNames()) {
            Object value = wikitty.getFqField(fqfieldName);
            if (value != null) { // FIXME poussin 20090830 how to search for null value in field ?
                FieldType type = wikitty.getFieldType(fqfieldName);
                if (type.isCollection()) {
                    Collection<?> collection = (Collection<?>) value;
                    for (Object o : collection) {
                        String strValue = WikittyUtil.toString(type, o);
                        result.eq(fqfieldName, strValue);
                    }
                } else {
                    String strValue = WikittyUtil.toString(type, value);
                    result.eq(fqfieldName, strValue);
                }
            }
        }

        return result;
    }

    /**
     * Create new {@code Search} object with default kind to {@link KIND#AND}.
     * 
     * @return Search helper
     */
    public static Search query() {
        Search search = query(KIND.AND);
        return search;
    }

    /**
     * Create new {@code Search} object with custom kind.
     * 
     * @param kind kind
     * @return Search helper
     */
    public static Search query(KIND kind) {
        Search search = new Search();
        search.kind = kind;
        return search;
    }

    /**
     * Create new query on existent criteria to add new constraint to existent
     * criteria.
     * 
     * @param criteria
     * @return
     */
    public static Search query(Criteria criteria) {
        Search search = query();
        if (criteria != null) {
            search.restrictions.add(criteria.getRestriction());
        }
        return search;
    }
    
    protected static Element elt(String element) {
        Element elm = new Element();
        elm.setName(element);
        return elm;
    }

    /**
     * Contains.
     * 
     * @param element
     * @param values
     * @return
     */
    public Search contains(String element, Collection<String> values) {
        restrictions.add(RestrictionHelper.contains(elt(element),
                new ArrayList<String>(values)));
        return this;
    }

    /**
     * Equals.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search eq(String element, String value) {
        restrictions.add(RestrictionHelper.eq(elt(element), value));
        return this;
    }

    /**
     * Equals each collection elements.
     * 
     * @param element
     * @param values
     * @return {@code this}
     */
    public Search eq(String element, Collection<String> values) {
        for (String value : values) {
            restrictions.add(RestrictionHelper.eq(elt(element), value));
        }
        return this;
    }

    /**
     * Like.
     * 
     * @param element
     * @param value
     * @param searchAs
     * @return {@code this}
     */
    public Search like(String element, String value, Like.SearchAs searchAs) {
        restrictions.add(RestrictionHelper.like(elt(element), value, searchAs));
        return this;
    }

    /**
     * Unlike.
     * 
     * @param element
     * @param value
     * @param searchAs
     * @return {@code this}
     */
    public Search unlike(String element, String value, Like.SearchAs searchAs) {
        restrictions.add(RestrictionHelper.unlike(elt(element), value, searchAs));
        return this;
    }

    /**
     * Not equals.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search neq(String element, String value) {
        restrictions.add(RestrictionHelper.neq(elt(element), value));
        return this;
    }

    /**
     * Greater than.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search gt(String element, String value) {
        restrictions.add(RestrictionHelper.great(elt(element), value));
        return this;
    }

    /**
     * Greater or equals.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search ge(String element, String value) {
        restrictions.add(RestrictionHelper.greatEq(elt(element), value));
        return this;
    }

    /**
     * Less than.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search lt(String element, String value) {
        restrictions.add(RestrictionHelper.less(elt(element), value));
        return this;
    }

    /**
     * Less or equals.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search le(String element, String value) {
        restrictions.add(RestrictionHelper.lessEq(elt(element), value));
        return this;
    }

    /**
     * Between.
     * 
     * @param element
     * @param lowerValue
     * @param upperValue
     * @return {@code this}
     */
    public Search bw(String element, String lowerValue, String upperValue) {
        restrictions.add(RestrictionHelper.between(elt(element), lowerValue, upperValue));
        return this;
    }

    /**
     * Starts with.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search sw(String element, String value) {
        restrictions.add(RestrictionHelper.start(elt(element), value));
        return this;
    }

    /**
     * Not starts with.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search nsw(String element, String value) {
        restrictions.add(RestrictionHelper.not(
                        RestrictionHelper.start(elt(element), value)));
        return this;
    }

    /**
     * Ends with.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search ew(String element, String value) {
        restrictions.add(RestrictionHelper.end(elt(element), value));
        return this;
    }

    /**
     * Not ends with.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search notew(String element, String value) {
        restrictions.add(RestrictionHelper.not(
                        RestrictionHelper.end(elt(element), value)));
        return this;
    }

    /**
     * Keyword.
     * 
     * @param value
     * @return {@code this}
     */
    public Search keyword(String value) {
        restrictions.add(RestrictionHelper.keyword(value));
        return this;
    }

    /**
     * Not (sub query).
     * 
     * @return sub query
     */
    public Search not() {
        Search not = Search.query(KIND.NOT);
        subSearchs.add(not);

        Search search = Search.query(kind);
        not.subSearchs.add(search);

        return search;
    }

    /**
     * Or (sub query).
     * 
     * @return sub query
     */
    public Search or() {
        Search search = Search.query(KIND.OR);
        subSearchs.add(search);
        return search;
    }

    /**
     * And (sub query).
     * 
     * @return sub query
     */
    public Search and() {
        Search search = Search.query(KIND.AND);
        subSearchs.add(search);
        return search;
    }

    public Search associated(String foreignFieldName) {
        Search search = new SubSearch(foreignFieldName, this);
        return search;
    }
    
    /**
     * Return named criteria.
     * 
     * @param name name of criteria
     * @return new criteria
     */
    public Criteria criteria(String name) {
        Criteria criteria = new Criteria(name);
        Restriction result = getRestrictions();
        criteria.setRestriction(result);
        return criteria;
    }

    /**
     * Return unnamed criteria.
     * 
     * @return new criteria
     */
    public Criteria criteria() {
        Criteria criteria = criteria(null);
        return criteria;
    }

    protected Restriction getRestrictions() throws UnsupportedOperationException {
        Restriction result;
        if (restrictions.isEmpty() && subSearchs.isEmpty()) {
            result = RestrictionHelper.rFalse();
            
        } else if (restrictions.size() == 1 && subSearchs.isEmpty()) {
            result = restrictions.remove(0);

        } else if (subSearchs.size() == 1 && restrictions.isEmpty()) {
            Search subSearch = subSearchs.get(0);
            result = subSearch.getRestrictions();

            if (kind == KIND.NOT) {
                result = RestrictionHelper.not(result);
            }

        } else {
            List<Restriction> allRestrictions = new ArrayList<Restriction>(restrictions);
            for (Search subSearch : subSearchs) {
                Restriction restriction = subSearch.getRestrictions();
                allRestrictions.add(restriction);
            }
            switch (kind) {
                case OR:
                    result = RestrictionHelper.or(allRestrictions);
                    break;
                case AND:
                    result = RestrictionHelper.and(allRestrictions);
                    break;
                default:
                    throw new UnsupportedOperationException("Can't handle restriction kind " + kind.name());
            }
        }
        return result;
    }

    
}
