package org.nuiton.topia.persistence.internal;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaHibernateSessionRegistry.java 2981 2014-01-17 17:38:55Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-9/topia-persistence/src/main/java/org/nuiton/topia/persistence/internal/TopiaHibernateSessionRegistry.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.lang.ref.WeakReference;
import java.util.WeakHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;
import org.nuiton.topia.persistence.TopiaPersistenceContext;

import com.google.common.base.Preconditions;

/**
 * Class used to keep an association between an Hibernate Session and a TopiaPersistenceContext
 *
 * @author Arnaud Thimel : thimel@codelutin.com
 * @since 3.0
 */
public class TopiaHibernateSessionRegistry {

    private static final Log log = LogFactory.getLog(TopiaHibernateSessionRegistry.class);

    protected WeakHashMap<Session, WeakReference<TopiaPersistenceContext>> registry =
            new WeakHashMap<Session, WeakReference<TopiaPersistenceContext>>();

    /**
     * Register the Session<->TopiaPersistenceContext couple
     *
     * @param session            the currently used Session
     * @param persistenceContext the current TopiaPersistenceContext using this Session
     */
    public void register(Session session, TopiaPersistenceContext persistenceContext) {
        if (log.isDebugEnabled()) {
            log.debug("New Session<->TopiaPersistenceContext registration");
        }
        Preconditions.checkArgument(session != null);
        Preconditions.checkArgument(persistenceContext != null);
        WeakReference<TopiaPersistenceContext> reference = new WeakReference<TopiaPersistenceContext>(persistenceContext);
        registry.put(session, reference);
    }

    /**
     * Look for the TopiaPersistenceContext based on the given Hibernate session
     *
     * @param session the Hibernate Session to use
     * @return the TopiaPersistenceContext using this Session, or null if not found
     */
    public TopiaPersistenceContext getPersistenceContext(Session session) {
        Preconditions.checkArgument(session != null);
        WeakReference<TopiaPersistenceContext> reference = registry.get(session);
        TopiaPersistenceContext result = reference.get();
        if (log.isDebugEnabled()) {
            log.debug("Get TopiaPersistenceContext from Session : " + (result == null ? "Not found" : "HIT !"));
        }
        return result;
    }

    /**
     * Removes the Session<->TopiaPersistenceContext association from the registry
     *
     * @param session the Hibernate Session to remove
     */
    public void unregister(Session session) {
        Preconditions.checkArgument(session != null);
        WeakReference<TopiaPersistenceContext> reference = registry.remove(session);
        if (log.isDebugEnabled()) {
            log.debug("Remove TopiaPersistenceContext from Session : " + (reference != null));
        }
    }

}
