package org.nuiton.topia.persistence.internal;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: HibernateProvider.java 2955 2013-12-20 16:19:35Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-8/topia-persistence/src/main/java/org/nuiton/topia/persistence/internal/HibernateProvider.java $
 * %%
 * Copyright (C) 2004 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.io.File;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.event.service.spi.EventListenerRegistry;
import org.hibernate.event.spi.EventType;
import org.hibernate.service.ServiceRegistry;
import org.hibernate.service.ServiceRegistryBuilder;
import org.hibernate.service.jdbc.connections.spi.ConnectionProvider;
import org.hibernate.service.spi.ServiceRegistryImplementor;
import org.hibernate.service.spi.Stoppable;
import org.nuiton.topia.persistence.TopiaConfigurationConstants;
import org.nuiton.topia.persistence.TopiaNotFoundException;
import org.nuiton.topia.persistence.TopiaService;
import org.nuiton.topia.persistence.support.TopiaServiceSupport;
import org.nuiton.topia.persistence.internal.support.TopiaHibernateEventListener;
import org.nuiton.topia.persistence.util.TopiaUtil;

import com.google.common.collect.Lists;

/**
 * @author Arnaud Thimel <thimel@codelutin.com>
 */
public class HibernateProvider {

    private static final Log log = LogFactory.getLog(HibernateProvider.class);

    protected SessionFactory hibernateSessionFactory;
    protected Configuration hibernateConfiguration;

    protected Map<String, String> configuration;
    protected TopiaServiceSupport topiaServiceSupport;
    protected TopiaHibernateSessionRegistry sessionRegistry;

    /**
     * List of persistent classes
     */
    protected List<Class<?>> persistentClasses = Lists.newArrayList();

    public HibernateProvider(Map<String, String> configuration,
                             TopiaServiceSupport topiaServiceSupport,
                             TopiaHibernateSessionRegistry sessionRegistry) {
        this.configuration = configuration;
        this.topiaServiceSupport = topiaServiceSupport;
        this.sessionRegistry = sessionRegistry;
    }

    protected String getProperty(String key) {
        return getProperty(key, null);
    }

    protected String getProperty(String key, String defaultValue) {
        String result = defaultValue;
        if (configuration.containsKey(key)) {
            result = configuration.get(key);
        }

        return result;
    }

    public List<Class<?>> getPersistentClasses() {
        if (persistentClasses.isEmpty()) {
            // Force configuration load
            getHibernateConfiguration();
        }
        return persistentClasses;
    }

    public Configuration getHibernateConfiguration() {
        if (hibernateConfiguration == null) {
            hibernateConfiguration = new Configuration();

            // ajout des repertoires contenant les mappings hibernate
            String[] dirs = getProperty(
                    TopiaConfigurationConstants.CONFIG_PERSISTENCE_DIRECTORIES, "").split(",");
            for (String dir : dirs) {
                dir = dir.trim();
                if (StringUtils.isNotEmpty(dir)) {
                    if (log.isDebugEnabled()) {
                        log.debug("Load persistence from dir : " + dir);
                    }
                    hibernateConfiguration.addDirectory(new File(dir));
                }
            }

            // ajout des classes dites persistentes
            Set<Class<?>> hibernatePersistanceClasses = new HashSet<Class<?>>();
            for (TopiaService service : topiaServiceSupport.getServices().values()) {
                Class<?>[] classes = service.getPersistenceClasses();

                // certains service n'ont pas de classe persistantes
                if (classes != null) {
                    // sletellier 20110411 : http://www.nuiton.org/issues/show/1454
                    hibernatePersistanceClasses.addAll(Arrays.asList(classes));
//                        for (Class<?> clazz : classes) {
//                            hibernateConfiguration.addClass(clazz);
//                        }
                }
            }

            String listPersistenceClasses = getProperty(
                    TopiaConfigurationConstants.CONFIG_PERSISTENCE_CLASSES, "");

            String[] classes = listPersistenceClasses.split(",");
            for (String classname : classes) {
                classname = classname.trim();
                if (StringUtils.isNotEmpty(classname)) {
                    if (log.isDebugEnabled()) {
                        log.debug("Load persistent class : " + classname);
                    }

                    // XXX echatellier 20111007 ce qui est dommage ici, c'est
                    // la definition de cette classe ne sert a rien (apart security)
                    // car pour hibernate hibernateConfiguration.addClass(persistanceClass)
                    // il ne se sert pas de la classe en fait et fait seulement
                    // un  classname.replace( '.', '/' ) + ".hbm.xml";
                    // pour obtenir le mapping et la reinstancier ensuite

                    Class<?> clazz;
                    try {
                        clazz = Class.forName(classname);
                    } catch (ClassNotFoundException eee) {
                        if (log.isDebugEnabled()) {
                            log.debug("Class " + classname + " not found");
                        }
                        throw new TopiaNotFoundException(
                                String.format("Persistence class %1$s not found",
                                        classname));
                    }
                    persistentClasses.add(clazz);

                    // sletellier 20110411 : http://www.nuiton.org/issues/show/1454
//                        hibernateConfiguration.addClass(clazz);
                    hibernatePersistanceClasses.add(clazz);
                }
            }

            // sletellier 20110411 : http://www.nuiton.org/issues/show/1454
            // Add persistance classes in hibernate config
            for (Class<?> persistanceClass : hibernatePersistanceClasses) {
                hibernateConfiguration.addClass(persistanceClass);
            }

            Properties prop = new Properties();
            prop.putAll(hibernateConfiguration.getProperties());
            prop.putAll(configuration);

            // Strange behavior, all properties are already loaded from
            // constructor. Difficult to use this behavior, need to have
            // TOPIA_PERSISTENCE_PROPERTIES_FILE in config.
            Properties propertiesFromClasspath =
                    TopiaUtil.getProperties(getProperty(TopiaConfigurationConstants.CONFIG_PERSISTENCE_PROPERTIES_FILE));

            if (!propertiesFromClasspath.isEmpty()) {
                if (log.isDebugEnabled()) {
                    log.debug("Load properties from file : " +
                            propertiesFromClasspath);
                }
                prop.putAll(propertiesFromClasspath);
            }

            hibernateConfiguration.setProperties(prop);

            // tchemit 2011-05-26 When using hibernate > 3.3, need to make sure all mappings are loaded (the one from directory files are not still done).
            hibernateConfiguration.buildMappings();
        }
        return hibernateConfiguration;
    }

    public void close() {
        if (hibernateSessionFactory != null) {
            hibernateSessionFactory.close();
            // close connection provider if possible (http://nuiton.org/issues/2757)
            SessionFactoryImplementor sessionFactoryImplementor = (SessionFactoryImplementor) hibernateSessionFactory;
            ServiceRegistryImplementor serviceRegistry = sessionFactoryImplementor.getServiceRegistry();
            ConnectionProvider service = serviceRegistry.getService(ConnectionProvider.class);

            // TODO AThimel 18/12/13 Check this code compatibility with Hibernate 4.3.0.Final
//            ConnectionProvider service = hibernateSessionFactory.getSessionFactoryOptions().getServiceRegistry().getService(ConnectionProvider.class);
            if (service instanceof Stoppable) {
                Stoppable stoppable = (Stoppable) service;
                stoppable.stop();
            }
        }
    }

    public SessionFactory getSessionFactory() {

        if (hibernateSessionFactory == null) {

            // init service registry
            Properties properties = getHibernateConfiguration().getProperties();
            ServiceRegistryBuilder builder = new ServiceRegistryBuilder().applySettings(properties);
            ServiceRegistry serviceRegistry = builder.buildServiceRegistry();

            hibernateSessionFactory = getHibernateConfiguration().buildSessionFactory(serviceRegistry);

            // TODO AThimel 18/12/13 Check this code compatibility with Hibernate 4.3.0.Final
//            StandardServiceRegistryBuilder builder = new StandardServiceRegistryBuilder();
//            Properties properties = getHibernateConfiguration().getProperties();
//            StandardServiceRegistry standardServiceRegistry = builder.applySettings(properties).build();
//
//            hibernateSessionFactory = getHibernateConfiguration().buildSessionFactory(standardServiceRegistry);

            // we can't reuse original serviceRegistry instance
            // we must call getServiceRegistry on factory to get a working one
            ServiceRegistry serviceRegistryInit = ((SessionFactoryImplementor) hibernateSessionFactory).getServiceRegistry();
            EventListenerRegistry eventListenerRegistry = serviceRegistryInit.getService(EventListenerRegistry.class);

            TopiaHibernateEventListener listener = new TopiaHibernateEventListener(sessionRegistry);
            eventListenerRegistry.appendListeners(EventType.PRE_INSERT, listener);
            eventListenerRegistry.appendListeners(EventType.PRE_LOAD, listener);
            eventListenerRegistry.appendListeners(EventType.PRE_UPDATE, listener);
            eventListenerRegistry.appendListeners(EventType.PRE_DELETE, listener);
            eventListenerRegistry.appendListeners(EventType.POST_INSERT, listener);
            eventListenerRegistry.appendListeners(EventType.POST_LOAD, listener);
            eventListenerRegistry.appendListeners(EventType.POST_UPDATE, listener);
            eventListenerRegistry.appendListeners(EventType.POST_DELETE, listener);

            // following listeners must be called before hibernate
            eventListenerRegistry.prependListeners(EventType.SAVE_UPDATE, listener);
        }
        return hibernateSessionFactory;
    }

}
