/*
 * *##% Nuiton utilities library
 * Copyright (C) 2004 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.util.beans;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;


/**
 * Model of a {@link Binder}.
 *
 * TODO-TC20100225 should have special cases for collections treatment.
 *
 * @author tchemit <chemit@codelutin.com>
 * @param <S> the source type
 * @param <T> the target type
 * @since 1.1.5
 */
public class BinderModel<S, T> implements java.io.Serializable {

    /**
     * source type
     */
    protected final Class<S> sourceType;
    /**
     * destination type
     */
    protected final Class<T> targetType;

    /**
     * source type descriptors (key are property names)
     */
    protected final Map<String, PropertyDescriptor> sourceDescriptors;
    /**
     * destination descriptors (key are property names)
     */
    protected final Map<String, PropertyDescriptor> targetDescriptors;
    /**
     * properties mapping (key are source properties, value are destination
     * properties)
     */
    protected final Map<String, String> propertiesMapping;

    private static final long serialVersionUID = 1L;

    public BinderModel(Class<S> sourceType, Class<T> targetType) {
        this.sourceType = sourceType;
        this.targetType = targetType;
        this.sourceDescriptors = new TreeMap<String, PropertyDescriptor>();
        this.targetDescriptors = new TreeMap<String, PropertyDescriptor>();
        this.propertiesMapping = new TreeMap<String, String>();
    }

    /**
     * Gets the type of the binder's source.
     *
     * @return the type of the source object in the binder
     */
    public Class<S> getSourceType() {
        return sourceType;
    }

    /**
     * Gets the type of the binder's destination
     *
     * @return the type of the destination object in the binder
     */
    public Class<T> getTargetType() {
        return targetType;
    }

    /**
     * Gets all registred property names of the binder's source type.
     *
     * @return the array of all source object properties names to bind
     */
    public String[] getSourceDescriptors() {
        Set<String> universe = sourceDescriptors.keySet();
        return universe.toArray(new String[sourceDescriptors.size()]);
    }

    /**
     * Gets all registred property names of the binder's destination type.
     *
     * @return the array of all source object properties names to bind
     */
    public String[] getTargetDescriptors() {
        Set<String> universe = targetDescriptors.keySet();
        return universe.toArray(new String[targetDescriptors.size()]);
    }

    /**
     * Gets the destination property name given the
     *
     * @param sourceProperty the name of the source property to bind
     * @return the name of the destination object property to bind, or
     *         {@code null} if {@code propertySrc} is unknown in the model
     */
    public String getTargetProperty(String sourceProperty) {
        if (!containsSourceProperty(sourceProperty)) {
            return null;
        }
        String dstProperty = propertiesMapping.get(sourceProperty);
        return dstProperty;
    }

    /**
     * Gets the bean descriptor of the source type for the given
     * destination property.
     *
     * @param sourceProperty name of the source type property name
     * @return the descriptor or {@code null} if not found.
     */
    public PropertyDescriptor getSourceDescriptor(String sourceProperty) {
        // check src property is registred
        if (!containsSourceProperty(sourceProperty)) {
            return null;
        }
        PropertyDescriptor descriptor = sourceDescriptors.get(sourceProperty);
        return descriptor;
    }

    /**
     * @param srcProperty the name of a property of the source object.
     * @return the method to read in a source object for the given property.
     */
    public Method getSourceReadMethod(String srcProperty) {
        PropertyDescriptor descriptor = getSourceDescriptor(srcProperty);
        Method readMethod = null;
        if (descriptor != null) {
            readMethod = descriptor.getReadMethod();
        }
        return readMethod;
    }

    /**
     * @param sourceProperty the name of a property of the source object.
     * @return the method to write in a source object for the given property.
     */
    public Method getSourceWriteMethod(String sourceProperty) {
        PropertyDescriptor descriptor = getSourceDescriptor(sourceProperty);
        Method writeMethod = null;
        if (descriptor != null) {
            writeMethod = descriptor.getWriteMethod();
        }
        return writeMethod;
    }

    /**
     * Gets the bean descriptor of the destination type for the given
     * destination property.
     *
     * @param targetProperty name of the destination type property name
     * @return the descriptor or {@code null} if not found.
     */
    public PropertyDescriptor getTargetDescriptor(String targetProperty) {
        // check dst property is registred
        if (!containsTargetProperty(targetProperty)) {
            return null;
        }
        PropertyDescriptor descriptor = targetDescriptors.get(targetProperty);
        return descriptor;
    }

    /**
     * @param targetProperty the name of a property of the destination object.
     * @return the method to read in a destination object for the given
     *         property.
     */
    public Method getTargetReadMethod(String targetProperty) {
        PropertyDescriptor descriptor = getTargetDescriptor(targetProperty);
        Method readMethod = null;
        if (descriptor != null) {
            readMethod = descriptor.getReadMethod();
        }
        return readMethod;
    }

    /**
     * @param targetProperty the name of a property of the destination object.
     * @return the method to write in a destination object for the given
     *         property.
     */
    public Method getTargetWriteMethod(String targetProperty) {
        PropertyDescriptor descriptor = getTargetDescriptor(targetProperty);
        Method writeMethod = null;
        if (descriptor != null) {
            writeMethod = descriptor.getWriteMethod();
        }
        return writeMethod;
    }

    protected boolean containsSourceProperty(String sourceProperty) {
        return propertiesMapping.containsKey(sourceProperty);
    }

    protected boolean containsTargetProperty(String targetProperty) {
        return propertiesMapping.containsValue(targetProperty);
    }

    protected void addBinding(PropertyDescriptor sourceDescriptor,
                              PropertyDescriptor targetDescriptor) {

        String sourceProperty = sourceDescriptor.getName();
        String targetProperty = targetDescriptor.getName();
        sourceDescriptors.put(sourceProperty, sourceDescriptor);
        targetDescriptors.put(targetProperty, targetDescriptor);
        propertiesMapping.put(sourceProperty, targetProperty);
    }

    protected Map<String, String> getPropertiesMapping() {
        return propertiesMapping;
    }
}
