/* *##% Nuiton utilities library
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */

/* *
 * MonitorMapper.java
 *
 * Created: 13 sept. 06 11:39:06
 *
 * @author poussin
 * @version $Revision: 1637 $
 *
 * Last update: $Date: 2009-06-21 04:53:31 +0200 (dim., 21 juin 2009) $
 * by : $Author: tchemit $
 */

package org.nuiton.log;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.log.LutinProgressEvent.ProgressType;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import java.awt.Button;
import java.awt.Component;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.Statement;


/**
 * Helper to monitor LutinLog. This class permit to say which component used
 * to display text, which component used to display progress and what is the
 * cancel button
 *
 * @author poussin
 */

public class MonitorMapper implements LutinLogListener, ActionListener {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(MonitorMapper.class);

    protected LutinLog lutinLog;
    protected Container parentContainer;
    protected Component label;
    protected JProgressBar progress;
    protected Component cancel;
    protected boolean changeCancelText;

    /**
     * @param lutinLog         LutinLog to monitor
     * @param parentContainer  component that contains all other component
     *                         if not null, then when task begin setVisible(true) is called
     *                         and when task end setVisible(false) is called
     * @param label            component that have setText method to display user log
     *                         ({@link JLabel}, {@link JTextField}, ...)
     * @param progress         progress widget
     * @param cancel           componant that have addActionListener
     *                         ({@link JButton},{@link  Button}, ...)
     * @param changeCancelText if true try to change text display in cancel
     *                         component switch between "Cancel" and "Don't cancel"
     */
    public MonitorMapper(LutinLog lutinLog, Container parentContainer,
                         Component label, JProgressBar progress, Component cancel,
                         boolean changeCancelText) {
        this.lutinLog = lutinLog;
        this.parentContainer = parentContainer;
        this.label = label;
        this.progress = progress;
        this.cancel = cancel;
        this.changeCancelText = changeCancelText;

        lutinLog.addLutinLogListener(this);
        addActionListener(cancel, this);
    }

    /**
     * Try to call addActionListener on component, do nothing if call failed
     *
     * @param c
     * @param action
     */
    protected void addActionListener(Component c, ActionListener action) {
        try {
            if (c != null) {
                Statement statement = new Statement(c, "addActionListener", new Object[]{action});
                statement.execute();
            }
        } catch (Exception eee) {
            if (log.isTraceEnabled()) {
                log.trace("Can't modify call addActionListener on: " + c, eee);
            }
        }
    }

    /**
     * try to call setText on component, do nothing if call failed
     *
     * @param c
     * @param text text to put in component
     */
    protected void setText(Component c, String text) {
        try {
            if (c != null) {
                Statement statement = new Statement(c, "setText", new Object[]{text});
                statement.execute();
            }
        } catch (Exception eee) {
            if (log.isTraceEnabled()) {
                log.trace("Can't modify call setText on: " + c, eee);
            }
        }
    }

    /* (non-Javadoc)
    * @see org.nuiton.log.LutinLogListener#logEvent(org.nuiton.log.LutinLogEvent)
    */
    public void logEvent(LutinLogEvent e) {
        if (e.getLogType() == LutinLogEvent.LogType.user) {
            SwingUtilities.invokeLater(new HandleLogEvent(e));
        }
    }

    /* (non-Javadoc)
     * @see org.nuiton.log.LutinLogListener#progressEvent(org.nuiton.log.LutinProgressEvent)
     */
    public void progressEvent(LutinProgressEvent e) {
        SwingUtilities.invokeLater(new HandleProgressEvent(e));
    }

    /* (non-Javadoc)
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent e) {
        boolean oldValue = lutinLog.isAskStopTask();
        lutinLog.setAskStopTask(!oldValue);
    }


    /**
     * Used to update UI
     *
     * @author poussin
     */
    class HandleLogEvent implements Runnable {
        protected LutinLogEvent e;

        public HandleLogEvent(LutinLogEvent e) {
            this.e = e;
        }

        public void run() {
            setText(label, e.getMsg());
        }
    }

    /**
     * Used to update UI
     *
     * @author poussin
     */
    class HandleProgressEvent implements Runnable {
        protected LutinProgressEvent e;

        public HandleProgressEvent(LutinProgressEvent e) {
            this.e = e;
        }

        public void run() {
            if (e.getType() == ProgressType.start) {
                int min = e.getLutinLog().getMin();
                int max = e.getLutinLog().getMax();
                int value = e.getLutinLog().getValue();
                progress.setMinimum(min);
                progress.setMaximum(max);
                progress.setValue(value);
                String text = value + " of " + max; // TODO i18n
                progress.setString(text);
                if (parentContainer != null) {
                    parentContainer.setVisible(true);
                }
            } else if (e.getType() == ProgressType.min) {
                int min = e.getLutinLog().getMin();
                progress.setMinimum(min);
            } else if (e.getType() == ProgressType.max) {
                int max = e.getLutinLog().getMax();
                progress.setMaximum(max);
                int value = e.getLutinLog().getValue();
                String text = value + " of " + max; // TODO i18n
                progress.setString(text);
            } else if (e.getType() == ProgressType.value) {
                int max = e.getLutinLog().getMax();
                int value = e.getLutinLog().getValue();
                progress.setValue(value);
                String text = value + " of " + max; // TODO i18n
                progress.setString(text);
            } else if (e.getType() == ProgressType.askStop) {
                if (changeCancelText) {
                    setText(cancel, "Don't cancel"); // TODO i18n
                }
            } else if (e.getType() == ProgressType.cancelAskStop) {
                if (changeCancelText) {
                    setText(cancel, "Cancel"); // TODO i18n
                }
            } else if (e.getType() == ProgressType.end) {
                if (parentContainer != null) {
                    parentContainer.setVisible(false);
                }
                progress.setMinimum(0);
                progress.setMaximum(0);
                progress.setValue(0);
                progress.setString("");
            }
        }
    }
}


