/* *##% Nuiton utilities library
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */

/**
 * Language.java
 */

package org.nuiton.i18n;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URLClassLoader;
import java.util.Enumeration;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.bundle.I18nBundleEntry;
import org.nuiton.util.RecursiveProperties;

/**
 * This class is used by the i18n class. It encapsulates the translation
 * resource for a given language.
 * <p/>
 * Encoding to be used to read properties files will be ISO-8859-1 since java does it like this.
 */
public class Language {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(Language.class);

    public static URLClassLoader getLoader() {
        ClassLoader loader = Language.class.getClassLoader();
        if (loader instanceof URLClassLoader) {
            return (URLClassLoader) loader;
        }
        log.warn("could not find the URLClassLoader : "+loader);
        return null;
    }

    /** toutes les traductions pour cette langue */
    protected Properties resource;

    /** la locale de la langue */
    protected Locale locale;

    /** @param l the current locale of the language */
    public Language(Locale l) {
        this.locale = l;
    }

    /**
     * charge les traductions de la languea partir d'une liste donnee de
     * fichiers de traduction.
     *
     * @param bundleEntries the used bundles entries to load
     */
    public void load(I18nBundleEntry[] bundleEntries) {
        
        resource = new RecursiveProperties();
        
        try {
            for (I18nBundleEntry e : bundleEntries) {
                e.load(resource);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * translate takes a sentence and returns its translation if found, the very
     * same string otherwise.
     *
     * @param sentence sentence to translate
     * @return translated sentence
     */
    public String translate(String sentence) {
        if (resource == null) {
            recordNotFound(sentence);
            return sentence;
        }
        try {
            String result = resource.getProperty(sentence);
            if (null != result && !"".equals(result)) {
                return result;
            }
            recordNotFound(sentence);
            return sentence;
        } catch (MissingResourceException eee) {
            log.warn("Resource " + sentence + " unavailable", eee);
            return sentence;
        } catch (Exception eee) {
            log.error("Unexpected error while translating : ", eee);
            return sentence;
        }
    }

    private void recordNotFound(String key) {
        if (I18n.recordFilePath != null && key != null && !"".equals(key)) {
            File f = new File(I18n.recordFilePath);
            Properties recordProps = new Properties();
            try {
                if (f.exists()) {
                    FileInputStream fis = new FileInputStream(f);
                    recordProps.load(fis);
                    fis.close();
                }
                recordProps.put(key, "");
                FileOutputStream fos = new FileOutputStream(f);
                recordProps.store(fos, "Adding the key : " + key);
                fos.close();
            } catch (FileNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * untranslate takes a translated sentence and returns the original one if
     * found, the very same string otherwise.
     *
     * @param sentence sentence to untranslate
     * @return untranslated sentence
     */
    public String untranslate(String sentence) {
        if (resource == null) {
            return sentence;
        }
        try {
            Enumeration<?> e = resource.propertyNames();
            // Look for the given sentence through all translations
            while (e.hasMoreElements()) {
                String key = (String) e.nextElement();
                String translation = resource.getProperty(key);
                // If found returns the corresponding key
                if (sentence.equals(translation)) {
                    return key;
                }
            }
        } catch (MissingResourceException eee) {
            // Well, this can't happen...
        }
        // No such translated sentence in our resourceBundle
        return sentence;
    }

    public Locale getLocale() {
        return locale;
    }

    public int size() {
        return resource == null ? 0 : resource.size();
    }

    public void close() {
        if (resource != null) {
            log.info(this);
            resource.clear();
            resource = null;
        }
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }

    @Override
    public boolean equals(Object o) {
        return this == o || o instanceof Language && locale.equals(((Language) o).locale);
    }

    @Override
    public int hashCode() {
        return locale.hashCode();
    }

    @Override
    public String toString() {
        return "Language <locale: " + locale + ",nbStences:" + (size()) + ">";
    }
}
