package jaxx.runtime.swing.editor.config.model;

import javax.swing.*;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * CallBack manager.
 *
 * @author tchemit < chemit@codelutin.com >
 * @since 2.0
 */
public class CallBacksManager {

    /**
     * lists of registred callback.
     */
    protected List<CallBackEntry> callbacks;

    public CallBacksManager() {
        callbacks = new ArrayList<CallBackEntry>();
    }

    /**
     * Registers a new callback.
     * <p/>
     * <b>Note:</b> the order of registred callback is used to determine
     * the higher priority of callback to launch if required.
     *
     * @param name        the unique name of a callback
     * @param description the i18n key to describe the action
     * @param icon        icon of callBack (used in ui)
     * @param action      the action of the callback
     */
    public void registerCallBack(String name,
                                 String description,
                                 Icon icon,
                                 Runnable action) {
        if (name == null) {
            throw new NullPointerException("parameter 'name' can not be null");
        }
        if (action == null) {
            throw new NullPointerException("parameter 'action' can not be null");
        }
        if (description == null) {
            throw new NullPointerException("parameter 'description' can " +
                    "not be null");
        }
        if (icon == null) {
            throw new NullPointerException("parameter 'icon' can not be null");
        }
        if (getCallBack(name) != null) {
            throw new IllegalArgumentException("there is already a callback " +
                    "with name '" + name + "'");
        }
        callbacks.add(new CallBackEntry(name, description, icon, action));
    }

    /**
     * Registers a option into a known callback.
     *
     * @param name   the name of the callback
     * @param option the option to register for the given callback
     */
    public void registerOption(String name, OptionModel option) {
        if (name == null) {
            throw new NullPointerException("parameter 'name' can not be null");
        }
        if (option == null) {
            throw new NullPointerException("parameter 'option' can not be null");
        }
        CallBackEntry callback = getCallBack(name);
        if (callback == null) {
            throw new IllegalArgumentException("could not find a callback " +
                    "with name '" + name + "'");
        }
        callback.addOption(option);
    }

    /**
     * Scan a model and grab per callBack the options saved.
     *
     * @param model the model to scan
     * @return the dictionnary of options for each callback to launch
     */
    public Map<CallBackEntry, List<OptionModel>> getCallBacksForSaved(
            ConfigUIModel model) {

        Map<CallBackEntry, List<OptionModel>> result;
        result = new LinkedHashMap<CallBackEntry, List<OptionModel>>();

        for (CategoryModel categoryModel : model) {
            Map<CallBackEntry, List<OptionModel>> callBacks =
                    getCallBacksForSaved(categoryModel);
            for (Map.Entry<CallBackEntry, List<OptionModel>> entry :
                    callBacks.entrySet()) {
                CallBackEntry key = entry.getKey();
                List<OptionModel> value = entry.getValue();
                if (result.containsKey(key)) {
                    result.get(key).addAll(value);
                } else {
                    result.put(key, value);
                }
            }
            callBacks.clear();
        }
        return result;
    }

    /**
     * Scan a category and grab per callBack the options saved.
     *
     * @param category the category to scan
     * @return the dictionnary of options for each callBack to launch
     */
    public Map<CallBackEntry, List<OptionModel>> getCallBacksForSaved(
            CategoryModel category) {

        Map<CallBackEntry, List<OptionModel>> result;
        result = new LinkedHashMap<CallBackEntry, List<OptionModel>>();

        for (OptionModel optionModel : category) {
            if (optionModel.isSaved()) {
                CallBackEntry callBackEntry = getCallBack(optionModel);
                if (callBackEntry != null) {
                    List<OptionModel> models = result.get(callBackEntry);
                    if (models == null) {
                        models = new ArrayList<OptionModel>();
                        result.put(callBackEntry, models);
                    }
                    models.add(optionModel);
                }
            }
        }
        return result;
    }

    /**
     * Get the first callBack for a given option.
     *
     * @param option the option
     * @return the first callBack (so the most important) on which the given
     *         option is attacjed. (can be null)
     */
    protected CallBackEntry getCallBack(OptionModel option) {
        for (CallBackEntry callback : callbacks) {
            if (callback.getOptions().contains(option)) {
                return callback;
            }
        }
        return null;
    }

    /**
     * Obtain a registred callBack from his name.
     *
     * @param name the name of the searched callBack
     * @return the callBack for the given name (or {@code null} if not found).
     */
    protected CallBackEntry getCallBack(String name) {
        for (CallBackEntry callback : callbacks) {
            if (callback.getName().equals(name)) {
                return callback;
            }
        }
        return null;
    }

}