package jaxx.runtime;

/*
 * #%L
 * JAXX :: Runtime
 * $Id: FileChooserUtil.java 2711 2013-08-22 19:54:43Z tchemit $
 * $HeadURL: https://nuiton.org/svn/jaxx/tags/jaxx-2.8.1/jaxx-runtime/src/main/java/jaxx/runtime/FileChooserUtil.java $
 * %%
 * Copyright (C) 2008 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;

import javax.swing.JFileChooser;
import java.awt.Component;
import java.io.File;

/**
 * To help choosing a file / directory.
 *
 * Was previously in {@code org.nuiton.util.FileUtil} from
 * {@code nuiton-utils} project.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.5.16
 */
public class FileChooserUtil {

    /** Logger. */
    private static final Log log = LogFactory.getLog(FileChooserUtil.class);

    public static final File DEFAULT_CURRENT_DIRECTORY_FILE = new File(".");

    protected static File currentDirectory = DEFAULT_CURRENT_DIRECTORY_FILE;

    public static void setCurrentDirectory(File dir) {
        currentDirectory = dir;
    }

    public static File getCurrentDirectory() {
        return currentDirectory;
    }

    public static boolean isCurrentDirectoryDefault() {
        return currentDirectory.equals(DEFAULT_CURRENT_DIRECTORY_FILE);
    }

    protected FileChooserUtil() {
        // no instance please
    }


    public static class PatternChooserFilter extends javax.swing.filechooser.FileFilter {
        protected String pattern;

        protected String description;

        public PatternChooserFilter(String pattern, String description) {
            this.pattern = pattern;
            this.description = description;
        }

        @Override
        public boolean accept(File f) {
            return f.isDirectory() || f.getAbsolutePath().matches(pattern);
        }

        @Override
        public String getDescription() {
            return description;
        }

    }


    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param patternOrDescriptionFilters les filtres a utiliser, les chaines doivent etre données
     *                                    par deux, le pattern du filtre + la description du filtre
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     * @see #getFile(javax.swing.filechooser.FileFilter...)
     */
    public static File getFile(String... patternOrDescriptionFilters) {
        File result;
        result = getFile(null, patternOrDescriptionFilters);
        return result;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param filters les filtres a ajouter
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     */
    public static File getFile(javax.swing.filechooser.FileFilter... filters) {
        File result = getFile(null, filters);
        return result;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param parent                      le component parent du dialog
     * @param patternOrDescriptionFilters les filtres a utiliser, les chaines doivent etre données
     *                                    par deux, le pattern du filtre + la description du filtre
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     * @see #getFile(javax.swing.filechooser.FileFilter...)
     */
    public static File getFile(Component parent,
                               String... patternOrDescriptionFilters) {
        File result;
        result = getFile("Ok", "Ok", parent, patternOrDescriptionFilters);
        return result;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param title                       le titre de la boite de dialogue
     * @param approvalText                le label du boutton d'acceptation
     * @param parent                      le component parent du dialog
     * @param patternOrDescriptionFilters les filtres a utiliser, les chaines doivent etre données
     *                                    par deux, le pattern du filtre + la description du filtre
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     * @see #getFile(javax.swing.filechooser.FileFilter...)
     */
    public static File getFile(String title,
                               String approvalText,
                               Component parent,
                               String... patternOrDescriptionFilters) {

        if (patternOrDescriptionFilters.length % 2 != 0) {
            throw new IllegalArgumentException(
                    "Arguments must be (pattern, description) couple");
        }
        javax.swing.filechooser.FileFilter[] filters =
                new javax.swing.filechooser.FileFilter[
                        patternOrDescriptionFilters.length / 2];
        for (int i = 0; i < filters.length; i++) {
            String pattern = patternOrDescriptionFilters[i * 2];
            String description = patternOrDescriptionFilters[i * 2 + 1];
            filters[i] = new PatternChooserFilter(pattern, description);
        }
        File result;
        result = getFile(title, approvalText, parent, filters);
        return result;
    }


    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param parent  le component parent du dialog
     * @param filters les filtres a ajouter
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     */
    public static File getFile(Component parent,
                               javax.swing.filechooser.FileFilter... filters) {
        File result = getFile("Ok", "Ok", parent, filters);
        return result;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param title        le titre de la boite de dialogue
     * @param approvalText le label du boutton d'acceptation
     * @param parent       le component parent du dialog
     * @param filters      les filtres a ajouter
     * @return le fichier accepté, ou null si rien n'est choisi ou l'utilisateur a annulé
     */
    public static File getFile(String title,
                               String approvalText,
                               Component parent,
                               javax.swing.filechooser.FileFilter... filters) {
        try {
            JFileChooser chooser = new JFileChooser(currentDirectory);

            chooser.setDialogType(JFileChooser.CUSTOM_DIALOG);
            if (filters.length > 0) {
                if (filters.length == 1) {
                    chooser.setFileFilter(filters[0]);
                } else {
                    for (javax.swing.filechooser.FileFilter filter : filters) {
                        chooser.addChoosableFileFilter(filter);
                    }
                }
            }
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            chooser.setDialogTitle(title);
            int returnVal = chooser.showDialog(parent, approvalText);
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File theFile = chooser.getSelectedFile();
                if (theFile != null) {
                    currentDirectory = theFile;
                    return theFile.getAbsoluteFile();
                }
            }
        } catch (Exception eee) {
            log.warn("Erreur:", eee);
        }
        return null;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param title                       le titre de la boite de dialogue
     * @param approvalText                le label du boutton d'acceptation
     * @param parent                      le component parent du dialog
     * @param patternOrDescriptionFilters les filtres a utiliser, les chaines doivent etre données
     *                                    par deux, le pattern du filtre + la description du filtre
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     */
    public static File saveFile(String filename,
                                String title,
                                String approvalText,
                                Component parent,
                                String... patternOrDescriptionFilters) {

        if (patternOrDescriptionFilters.length % 2 != 0) {
            throw new IllegalArgumentException(
                    "Arguments must be (pattern, description) couple");
        }
        javax.swing.filechooser.FileFilter[] filters =
                new javax.swing.filechooser.FileFilter[
                        patternOrDescriptionFilters.length / 2];
        for (int i = 0; i < filters.length; i++) {
            String pattern = patternOrDescriptionFilters[i * 2];
            String description = patternOrDescriptionFilters[i * 2 + 1];
            filters[i] = new FileUtil.PatternChooserFilter(pattern, description);
        }
        File result;
        result = saveFile(filename, title, approvalText, parent, filters);
        return result;
    }

    /**
     * Retourne le nom du fichier entre dans la boite de dialogue.
     * Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne null.
     *
     * @param title        le titre de la boite de dialogue
     * @param approvalText le label du boutton d'acceptation
     * @param parent       le component parent du dialog
     * @param filters      les filtres a ajouter
     * @return le fichier accepté, ou null si rien n'est chois ou l'utilisateur a annulé
     */
    public static File saveFile(String filename,
                                String title,
                                String approvalText,
                                Component parent,
                                javax.swing.filechooser.FileFilter... filters) {
        File selectedFile = new File(getCurrentDirectory(), filename);
        JFileChooser chooser = new JFileChooser(selectedFile);

        chooser.setDialogType(JFileChooser.SAVE_DIALOG);
        chooser.setSelectedFile(selectedFile);
        if (filters.length > 0) {
            if (filters.length == 1) {
                chooser.setFileFilter(filters[0]);
            } else {
                for (javax.swing.filechooser.FileFilter filter : filters) {
                    chooser.addChoosableFileFilter(filter);
                }
            }
        }
        chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
        chooser.setDialogTitle(title);

        File result = null;

        int returnVal = chooser.showDialog(parent, approvalText);
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            result = chooser.getSelectedFile();
            if (result != null) {
                setCurrentDirectory(result);
                result = result.getAbsoluteFile();
            }
        }
        return result;
    }

    /**
     * @return le nom du repertoire entre dans la boite de dialogue.
     *         Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne
     *         null.
     */
    public static String getDirectory() {
        return getDirectory(null, "Ok", "Ok");
    }

    /**
     * @param title        le nom de la boite de dialogue
     * @param approvalText le texte de l'action d'acceptation du répertoire dans le file chooser
     * @return le nom du repertoire entre dans la boite de dialogue.
     *         Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne
     *         null.
     */
    public static String getDirectory(String title, String approvalText) {
        String result = getDirectory(null, title, approvalText);
        return result;
    }

    /**
     * @param parent       le component parent du dialog
     * @param title        le nom de la boite de dialogue
     * @param approvalText le texte de l'action d'acceptation du répertoire dans le file chooser
     * @return le nom du repertoire entre dans la boite de dialogue.
     *         Si le bouton annuler est utilisé, ou qu'il y a une erreur retourne
     *         null.
     */
    public static String getDirectory(Component parent,
                                      String title,
                                      String approvalText) {
        try {
            JFileChooser chooser = new JFileChooser(currentDirectory);
            chooser.setDialogType(JFileChooser.CUSTOM_DIALOG);
            chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            chooser.setDialogTitle(title);
            int returnVal = chooser.showDialog(parent, approvalText);
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File theFile = chooser.getSelectedFile();
                if (theFile != null) {
                    currentDirectory = theFile;
                    if (theFile.isDirectory()) {
                        return theFile.getAbsolutePath();
                    }
                }
            } else {
                return null;
            }
        } catch (Exception eee) {
            log.warn("Erreur:", eee);
        }
        return null;
    }
}
