/*
 * *##%
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.navigation.tree.NavigationTreeModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Model of the tree used for a navigation tree. It is composed of {@link E}
 * nodes.
 *
 * @author letellier
 * @param <E> the type of nodes in model
 * @since 2.0.1
 */
public abstract class AbstractNavigationModel<E extends NavigationNode<E>> implements NavigationModel<E> {

    /** Logger */
    static private final Log log = LogFactory.getLog(NavigationTreeModel.class);

    /** the delegate model */
    protected final TreeModel delegate;

    /**
     * The path separator used to build the {@link NavigationNode#getFullPath()}.
     *
     * @see NavigationNode#getNodePath()
     * @see NavigationNode#getFullPath()
     */
    protected final String pathSeparator;

    /** Context to retrieve beans */
    private JAXXContext context;
    /**
     * if sets to {@code true} will not fires any event (this is a convinient
     * state when we are building the tree to avoid reload of nodes)
     */
    protected boolean adjustingValue;

    public AbstractNavigationModel(TreeModel delegate,
                                   String pathSeparator,
                                   JAXXContext context) {
        this.pathSeparator = pathSeparator;
        this.context = context;
        this.delegate = delegate;
    }

    //--------------------------------------------------------------------------
    // NavigationModel implementation
    //--------------------------------------------------------------------------

    public boolean isAdjustingValue() {
        return adjustingValue;
    }

    public void setAdjustingValue(boolean adjustingValue) {
        this.adjustingValue = adjustingValue;
    }

    @Override
    public JAXXContext getContext() {
        return context;
    }

    public TreeModel getDelegate() {
        return delegate;
    }

    @Override
    public final String getPathSeparator() {
        return pathSeparator;
    }

    @Override
    public final E getRoot() {
        return (E) getDelegate().getRoot();
    }

    @Override
    public final Object getBean(String navigationPath) {
        Object result;
        E node = findNode(navigationPath, (Pattern) null);
        result = getBean(node);
        return result;
    }

    @Override
    public final Object getBean(E node) {
        if (node == null) {
            return null;
            //fixme should throw a NPE exception
            //throw new NullPointerException("node can not be null");
        }
        return node.getBean(getContext());
    }

    @Override
    public final E findNode(String path) {
        return findNode(getRoot(), path, (Pattern) null);
    }

    @Override
    public final E findNode(String path, String regex) {
        return findNode(getRoot(), path, regex);
    }

    @Override
    public final E findNode(String path, Pattern regex) {
        return findNode(getRoot(), path, regex);
    }

    @Override
    public final E findNode(E root, String path) {
        return findNode(root, path, (Pattern) null);
    }

    @Override
    public final E findNode(E root, String path, String regex) {
        return findNode(root, path,
                        regex == null ? null : Pattern.compile(regex));
    }

    @Override
    public final E findNode(E root, String path, Pattern regex) {
        if (regex != null) {
            Matcher matcher = regex.matcher(path);
            if (!matcher.matches() || matcher.groupCount() < 1) {
                log.warn("no matching regex " + regex + " to " + path);
                return null;
            }
            path = matcher.group(1);
            if (log.isDebugEnabled()) {
                log.debug("matching regex " + regex + " : " + path);
            }
        }
        StringTokenizer stk = new StringTokenizer(path, pathSeparator);
        E result = root;
        // pas the first token (matches the root node)
        if (root.isRoot() && stk.hasMoreTokens()) {
            String rootPath = stk.nextToken();
            if (!rootPath.equals(root.getNodePath())) {
                return null;
            }
        }
        while (stk.hasMoreTokens()) {
            result = result.getChild(stk.nextToken());
        }
        return result;
    }

    @Override
    public final void nodeChanged(E node) {
        if (adjustingValue) {
            return;
        }
        nodeChanged(node, false);
        if (log.isDebugEnabled()) {
            log.debug(node);
        }
    }

    protected final void reload(E node) {
        if (adjustingValue) {
            return;
        }
        reload(node, false);
    }

    protected final void reload(E node, boolean deep) {
        if (node == null) {
            return;
        }
        if (adjustingValue) {
            return;
        }
        node.reload(getContext());

        if (deep) {
            Enumeration<? extends E> childs = node.children();
            while (childs.hasMoreElements()) {
                E o = childs.nextElement();
                reload(o, true);
            }
        }
    }

    //--------------------------------------------------------------------------
    // TreeModel implementation
    //--------------------------------------------------------------------------

    @Override
    public final Object getChild(Object parent, int index) {
        return getDelegate().getChild(parent, index);
    }

    @Override
    public final int getChildCount(Object parent) {
        return getDelegate().getChildCount(parent);
    }

    @Override
    public final boolean isLeaf(Object node) {
        return getDelegate().isLeaf(node);
    }

    @Override
    public final void valueForPathChanged(TreePath path, Object newValue) {
        if (adjustingValue) {
            return;
        }
        getDelegate().valueForPathChanged(path, newValue);
    }

    @Override
    public final int getIndexOfChild(Object parent, Object child) {
        return getDelegate().getIndexOfChild(parent, child);
    }

    @Override
    public final void addTreeModelListener(TreeModelListener l) {
        getDelegate().addTreeModelListener(l);
    }

    @Override
    public final void removeTreeModelListener(TreeModelListener l) {
        getDelegate().removeTreeModelListener(l);
    }
}
