/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: JavaFile.java 1847 2010-04-16 12:27:48Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.2.3/jaxx-compiler/src/main/java/jaxx/compiler/java/JavaFile.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler.java;

import jaxx.compiler.types.TypeManager;
import org.apache.commons.lang.StringUtils;

import java.lang.reflect.Modifier;
import java.util.*;

/**
 * A Java source file being generated for output.  Once the class is completely initialized, use the
 * {@link #toString} method to generate source code for it.
 */
public class JavaFile extends JavaElement {

    protected static final String GETTER_PATTERN = "return %1$s;";
    protected static final String BOOLEAN_GETTER_PATTERN = "return %1$s !=null && %1$s;";
    protected static final String SETTER_PATTERN = "%1$s oldValue = this.%2$s;\nthis.%2$s = newValue;\nfirePropertyChange(%3$s, oldValue, newValue);";
    private Set<String> imports = new HashSet<String>();
    private List<JavaField> fields = new ArrayList<JavaField>();
    private List<JavaMethod> methods = new ArrayList<JavaMethod>();
    private List<JavaFile> innerClasses = new ArrayList<JavaFile>();
    private String superClass;
    private List<String> interfaces;
    private StringBuffer rawBodyCode = new StringBuffer();
    private boolean superclassIsJAXXObject;
    private boolean abstractClass;
    private String genericType;
    private String superGenericType;

    public JavaFile() {
        super(0, "");
    }

    public JavaFile(int modifiers, String className, String superClass) {
        this(modifiers, className, superClass, null);
    }

    public JavaFile(int modifiers, String className, String superClass, List<String> interfaces) {
        super(modifiers, className);
        this.superClass = superClass;
        this.interfaces = interfaces;
    }

    public void addImport(String importString) {
        imports.add(importString);
    }

    public void addImport(Class<?> importString) {
        addImport(importString.getName());
    }

    public String[] getImports() {
        List<String> result = new ArrayList<String>(imports);
        Collections.sort(result);
        return result.toArray(new String[result.size()]);
    }

    public String getSuperClass() {
        return superClass;
    }

    public void setSuperClass(String superClass) {
        this.superClass = superClass;
    }

    public List<String> getInterfaces() {
        if (interfaces == null) {
            interfaces = new ArrayList<String>();
        }
        return interfaces;
    }

    public void setInterfaces(List<String> interfaces) {
        this.interfaces = interfaces;
    }

    public void setGenericType(String genericType) {
        this.genericType = genericType;
    }

    public void addMethod(JavaMethod method) {
        methods.add(method);
    }

    public List<JavaMethod> getMethods() {
        return methods;
    }

    public void addField(JavaField field) {
        addField(field, false);
    }

    public void addField(JavaField field, boolean javaBean) {
        addSimpleField(field);
        String id = field.getName();
        String capitalizedName = StringUtils.capitalize(id);
        // add getter file
        String content = String.format(GETTER_PATTERN, id);
        addMethod(new JavaMethod(
                Modifier.isProtected(field.getModifiers()) ? Modifier.PUBLIC : Modifier.PROTECTED,
                field.getType(), "get" + capitalizedName, null, null, content, field.isOverride()));

        if (javaBean) {
            // add full javabean support accessor + mutator + constante with name of property to make it easier to use
            // compute the property constante
            String constantId = TypeManager.convertVariableNameToConstantName("property" + capitalizedName);
            addSimpleField(JavaFileGenerator.newField(Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL, String.class.getSimpleName(), constantId, false, "\"" + id + "\""));

            if (Boolean.class.getName().equals(field.getType())) {
                content = String.format(BOOLEAN_GETTER_PATTERN, id);
                addMethod(new JavaMethod(Modifier.PUBLIC, field.getType(), "is" + capitalizedName, null, null, content, field.isOverride()));
            }
            content = String.format(SETTER_PATTERN, field.getType(), id, constantId);
            JavaArgument arg = new JavaArgument(field.getType(), "newValue");
            addMethod(new JavaMethod(Modifier.PUBLIC, "void", "set" + capitalizedName, new JavaArgument[]{arg}, null, content, field.isOverride()));
        }
    }

    public void addSimpleField(JavaField field) {
        fields.add(field);
    }

    public List<JavaField> getFields() {
        return fields;
    }

    public void addBodyCode(String bodyCode) {
        rawBodyCode.append(bodyCode);
    }

    public boolean isAbstractClass() {
        return abstractClass;
    }

    public String getGenericType() {
        return genericType;
    }

    public List<JavaFile> getInnerClasses() {
        return innerClasses;
    }

    public StringBuffer getRawBodyCode() {
        return rawBodyCode;
    }

    public String getSuperGenericType() {
        return superGenericType;
    }

    public void addInterface(String canonicalName) {
        if (interfaces == null || !interfaces.contains(canonicalName)) {
            getInterfaces().add(canonicalName);
        }
    }

    public void addInterfaces(String[] canonicalNames) {
        if (canonicalNames == null) {
            return;
        }
        for (String canonicalName : canonicalNames) {
            if (interfaces == null || !interfaces.contains(canonicalName)) {
                getInterfaces().add(canonicalName);
            }
        }
    }

    public boolean isSuperclassIsJAXXObject() {
        return superclassIsJAXXObject;
    }

    public void setSuperclassIsJAXXObject(boolean superclassIsJAXXObject) {
        this.superclassIsJAXXObject = superclassIsJAXXObject;
    }

    public void setAbstractClass(boolean abstractClass) {
        this.abstractClass = abstractClass;
    }

    public void setSuperGenericType(String superGenericType) {
        this.superGenericType = superGenericType;
    }
}
