/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.decorators;

import jaxx.compiler.CompiledObject;
import jaxx.compiler.CompiledObjectDecorator;
import jaxx.compiler.CompilerException;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.java.JavaFile;
import jaxx.compiler.java.JavaFileGenerator;
import jaxx.compiler.script.ScriptInitializer;

import java.lang.reflect.Modifier;
import java.util.Map.Entry;

/**
 * The default compiledObjectDecorator.
 * 
 * @author tony
 * @since 1.2
 */
public class DefaultCompiledObjectDecorator implements CompiledObjectDecorator {

    @Override
    public String getName() {
        return "default";
    }

    @Override
    public void finalizeCompiler(JAXXCompiler compiler, CompiledObject root, CompiledObject object, JavaFile javaFile, String packageName, String className, String fullClassName) {

        if (!object.isOverride() && !(object instanceof ScriptInitializer)) {
            String id = object.getId();
            int access = id.startsWith("$") ? Modifier.PRIVATE : Modifier.PROTECTED;
            if (object == root) {
                javaFile.addSimpleField(JavaFileGenerator.newField(access, className, id, false, "this"));
//                javaFile.addField(new JavaField(access, fullClassName, id, false, "this"));
            } else {
                //TC -20081017 can have generic on compiled Object
                javaFile.addField(JavaFileGenerator.newField(access, JAXXCompiler.getCanonicalName(object), id, object.isOverride()), object.isJavaBean());
            }
        }

        if (!compiler.inlineCreation(object) && object != root) {
            javaFile.addMethod(JavaFileGenerator.newMethod(Modifier.PROTECTED, "void", object.getCreationMethodName(), getCreationCode(compiler, object), object.isOverride()));
        }
    }

    @Override
    public String getCreationCode(JAXXCompiler compiler, CompiledObject object) throws CompilerException {
        if (object instanceof ScriptInitializer) {
            return object.getInitializationCode(compiler);
        }
        String eol = JAXXCompiler.getLineSeparator();
        StringBuffer result = new StringBuffer();
        if (compiler.getRootObject() == object || compiler.inlineCreation(object)) {
            result.append("// inline creation of " + object.getId());
        }
        if (object.isOverride() && object.getOverrideType() == object.getObjectClass()) {
            //TC-20090309 on utilise le super code quand l'objet est de meme type
            result.append("super.").append(object.getCreationMethodName()).append("();");
        } else {
            String init = object.getId() + " = ";
            if (object.isJavaBean() && object.getJavaBeanInitCode() != null) {
                init += object.getJavaBeanInitCode();
            } else if (object.getInitializer() != null) {
                init += object.getInitializer();
            } else {
                //TC - 20081017 compiledObject can have generics
                String canonicalName = JAXXCompiler.getCanonicalName(object);
                init += "new " + canonicalName + "(";
                String constructorParams = object.getConstructorParams();
                if (constructorParams != null) {
                    init += constructorParams;
                }
                init += ")";
            }
            result.append(eol);
            result.append("$objectMap.put(").append(compiler.getJavaCode(object.getId())).append(", ").append(init).append(");");
        }
        result.append(eol);
        String initCode = object.getInitializationCode(compiler);
        if (initCode != null && initCode.length() > 0) {
            result.append(initCode);
        }
        //TC-20091025 generate client properties at creation time (not at setup time)
        // in some case can save to create a setup method (when there is only client properties
        // to store)
        addClientProperties(object, result, eol);
        return result.toString();
    }

    @Override
    public String createCompleteSetupMethod(JAXXCompiler compiler, CompiledObject object, JavaFile javaFile, StringBuffer initDataBindings) {
        StringBuffer code = new StringBuffer();
        String eol = JAXXCompiler.getLineSeparator();
        //TC-20091025 generate client properties at creation time (not at setup time)
        // in some case can save to create a setup method (when there is only client properties
        // to store)
//        addClientProperties(object);
        //TC-20091025 only generate the code if not empty
        if (object.getId().startsWith("$")) {
            String additionCode = object.getAdditionCode();
            if (!additionCode.isEmpty()) {
                code.append("// inline complete setup of " + object.getId()).append(eol);
                code.append(additionCode);
            }
        } else {
            String additionCode = object.getAdditionCode();
            if (additionCode.length() > 0) {
                code.append(object.getAdditionMethodName()).append("();").append(eol);
                additionCode = "if (!allComponentsCreated) {" + eol + "    return;" + eol + "}" + eol + additionCode;
                javaFile.addMethod(JavaFileGenerator.newMethod(Modifier.PROTECTED, "void", object.getAdditionMethodName(), additionCode, false));
            }
        }
        String result = code.toString();
        return result;
    }

    @Override
    public boolean createInitializer(JAXXCompiler compiler, CompiledObject root, CompiledObject object, StringBuffer code, boolean lastWasMethodCall) {
        String eol = JAXXCompiler.getLineSeparator();
        if (object == root) {
            String rootCode = root.getInitializationCode(compiler);
            if (rootCode != null && rootCode.length() > 0) {
                code.append("// inline creation of " + object.getId()).append(eol);
                code.append(rootCode);
                //TC-20091025 generate client properties at creation time (not at setup time)
                // in some case can save to create a setup method (when there is only client properties
                // to store)
                addClientProperties(object, code, eol);
                code.append(eol);
            }
        } else {
            if (!object.isOverride()) {
                if (compiler.inlineCreation(object)) {
                    if (lastWasMethodCall) {
                        lastWasMethodCall = false;
                    }
                    code.append(getCreationCode(compiler, object));
                } else {
                    code.append(object.getCreationMethodName()).append("();");
                    code.append(eol);
                    lastWasMethodCall = true;
                }
            }
        }
        return lastWasMethodCall;
    }

    protected void addClientProperties(CompiledObject object, StringBuffer code, String eol) {
        //TC-20090327 generate client properties
        if (object.hasClientProperties()) {
            // generate putClientProperty invocations
            for (Entry<String, String> entry : object.getClientProperties().entrySet()) {
                code.append(object.getJavaCode() + ".putClientProperty(\"" + entry.getKey() + "\", " + entry.getValue() + ");").append(eol);
            }
        }
    }
}
