/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix;

import java.io.File;
import java.io.PrintWriter;
import java.io.StringReader;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.guix.generator.GuixGenerator;
import org.nuiton.guix.generator.JavaArgument;
import org.nuiton.guix.generator.JavaField;
import org.nuiton.guix.generator.JavaFile;
import org.nuiton.guix.generator.JavaMethod;
import org.nuiton.guix.model.ClassDescriptor;
import org.nuiton.guix.model.GuixModelObject;
import org.nuiton.guix.parser.JavaParser;
import org.nuiton.guix.parser.JavaParserTreeConstants;
import org.nuiton.guix.parser.SimpleNode;
import org.nuiton.guix.tags.TagManager;
import org.xmlpull.v1.XmlPullParserFactory;
import org.xmlpull.v1.XmlSerializer;

/**
 * Launch Guix files compilation
 *
 * @author morin
 */
public class GuixLauncher {

    /** log */
    protected static final Log log = LogFactory.getLog(GuixLauncher.class);
    /** original list of files to compile */
    protected final File[] files;
    /** original list of classes to compile */
    protected final List<String> classNames = new ArrayList<String>();
    /** Files to be treated while compilation. */
    protected List<File> guixFiles = new ArrayList<File>();
    /** Class names corresponding to the files in the guixFiles list. */
    protected List<String> guixFileClassNames = new ArrayList<String>();
    /** Maps the root GuixModelObjects being compiled to the compiler instance
     * handling the compilation. */
    protected Map<GuixModelObject, Long> rootModelObjects = new HashMap<GuixModelObject, Long>();
    /** CLassDescriptor met during the compilation */
    protected List<ClassDescriptor> classDescriptors = new ArrayList<ClassDescriptor>();
    /** List of the compiled files */
    private List<File> compiledFiles = new ArrayList<File>();
    /** Maps the generator with the generated JavaFile */
    private Map<GuixGenerator, JavaFile> generatedFiles = new HashMap<GuixGenerator, JavaFile>();
     /** Source directory of the project */
    private File srcDirectory;
    /** Directory where to save the generated files */
    private File targetDirectory;
    /** Main class of the generated application */
    private String mainClass;
    /** Class of teh generator to use */
    private Class<? extends GuixGenerator> generatorClass;
    /** Language of generation*/
    private String generationLanguage;
    /** Name of the class that will launch the application */
    private String launcherName;
    /** Maps the ClassDescriptors to generate to the corresponding root GuixModelObject */
    private Map<ClassDescriptor, GuixModelObject> rootClassDescriptors = new HashMap<ClassDescriptor, GuixModelObject>();
    /** Dependencies between the different files to generate */
    private Map<GuixModelObject, ArrayList<ClassDescriptor>> dependencies = new HashMap<GuixModelObject, ArrayList<ClassDescriptor>>();

    /**
     * Constructor
     *
     * @param files the files to compile
     * @param targetDirectory the directory where to save the generated files
     * @param srcDir the directory of the source files
     * @param mainClass main class of the application
     * @param generatorClass class of the generator
     * @param generationLanguage the output graphical library
     * @param launcherName name of the class that will launch the application
     */
    public GuixLauncher(File[] files, File targetDirectory, File srcDir, String mainClass, Class<? extends GuixGenerator> generatorClass,String generationLanguage, String launcherName) {
        // Set up a simple configuration that logs on the console.
        this.files = files;
        this.srcDirectory = srcDir;
        this.targetDirectory = targetDirectory;
//        this.rootPackage = rootPackage != null ? rootPackage : "";
        if (files != null) {
            for (int i = 0; i < files.length; i++) {
                String path = files[i].getAbsolutePath();
                if (srcDir != null) {
                    classNames.add(path.substring(srcDir.getAbsolutePath().length() + 1, path.lastIndexOf('.')).replace(File.separatorChar, '.'));
//                    classNames.add((rootPackage.length() > 0 ? rootPackage + "." : "") + path.substring(baseDir.getAbsolutePath().length() + 1,
//                            path.lastIndexOf('.')).replace(File.separatorChar, '.'));
                }
                else {
                    classNames.add(path.substring(0, path.lastIndexOf('.')));
                }
            }
        }
        this.mainClass = mainClass != null ? mainClass : "";
        this.launcherName = launcherName != null ? launcherName : "Main";
        this.generatorClass = generatorClass;
        this.generationLanguage = generationLanguage;
    }

    public File getSrcDirectory() {
        return srcDirectory;
    }

    public File getTargetDirectory() {
        return targetDirectory;
    }

    public List<String> getGuixFileClassNames() {
        return guixFileClassNames;
    }

    /**
     * Compiles a set of files.
     *
     * @return <code>true</code> if compilation succeeds,
     * <code>false</code> otherwise
     */
    public synchronized boolean compile() {
        if (files != null) {
            if (log.isInfoEnabled()) {
                log.info("Start compiling");
            }
            guixFiles.addAll(Arrays.asList(files));
            guixFileClassNames.addAll(classNames);
            boolean success = true;
            
            try {
                assert guixFiles.size() == guixFileClassNames.size();

                for (int i = 0; i < guixFiles.size(); i++) {
                    File file = guixFiles.get(i);
                    String className = guixFileClassNames.get(i);

                    //if we have not compiled the file yet
                    if (!compiledFiles.contains(file)) {
                        if (log.isInfoEnabled()) {
                            log.info("Compiling class " + className);
                        }
                        compiledFiles.add(file);

                        String classPackage;
                        if (targetDirectory != null) {
                            int dotPos = className.lastIndexOf(".");
                            if (dotPos != -1) {
                                classPackage = className.substring(0, dotPos);
                            }
                            else {
                                classPackage = "";
                            }
                        }
                        else {
                            targetDirectory = file.getParentFile();
                            classPackage = targetDirectory.getAbsolutePath().replace(File.separatorChar, '.');
                        }
                        if(!targetDirectory.exists()) {
                            targetDirectory.mkdirs();
                        }
                        //compile the file
                        GuixCompiler compiler = new GuixCompiler(file, this, classPackage, generationLanguage);
                        GuixModelObject rootModelObject = compiler.compile();
                        addRootModelObject(rootModelObject, compiler.getLastModification());

                        if (compiler.isFailed()) {
                            success = false;
                        }
                    }
                    else {
                        if (log.isWarnEnabled()) {
                            log.warn(file.getName() + " has already been compiled.");
                        }
                    }
                }

                //if the compilation successed
                if (success) {
                    //check if all the objects have an existing clas or a class being generated
                    for(ClassDescriptor classDescriptor : classDescriptors) {
                        if(classDescriptor.getPackageName() == null && TagManager.getGuixClassHandler(classDescriptor.getName()) == null) {
                            if(log.isWarnEnabled()) {
                                log.warn("The class '" + classDescriptor.getName() + "' has no package. Default package will be applied.");
                            }
                        }
                    }
                    if (generatorClass != null) {
                        //creates the XML serializer for the spring conf
                        XmlPullParserFactory factory =
                                XmlPullParserFactory.newInstance(
                                System.getProperty(XmlPullParserFactory.PROPERTY_NAME),
                                null);
                        XmlSerializer serializer = factory.newSerializer();

                        File config = new File(targetDirectory, "configSpring.xml");

                        serializer.setOutput(new PrintWriter(config));
                        serializer.startDocument("UTF-8", null);
                        serializer.setProperty("http://xmlpull.org/v1/doc/properties.html#serializer-line-separator", "\n");
                        serializer.setProperty("http://xmlpull.org/v1/doc/properties.html#serializer-indentation", "\t");
                        serializer.text("\n");
                        serializer.docdecl(" beans SYSTEM \"http://www.springframework.org/dtd/spring-beans.dtd\"");
                        serializer.text("\n");
                        serializer.startTag("", "beans");

                        Map<String, JavaFile> classes = new HashMap<String, JavaFile>();
                        for(String c : classNames) {
                            classes.put(c, null);
                        }

                        List<GuixModelObject> rootMos = new ArrayList<GuixModelObject>();

                        for(GuixModelObject mo : rootModelObjects.keySet()) {
                            if(!orderDependencies(mo, rootMos, rootClassDescriptors, -1)) {
                                log.error("I'm bad, I'm bad, you know it !");
                            }
                        }
                        for (GuixModelObject mo : rootMos) {
                            try {
                                //init generator
                                GuixGenerator gen = (GuixGenerator) generatorClass.newInstance();
                                gen.setSrcDir(srcDirectory);
                                gen.setDestDir(targetDirectory);
                                gen.setGmo(mo);
                                gen.setLastModification(rootModelObjects.get(mo));
                                gen.setMainClass((mo.getClassDescriptor().getPackageName() + "." + mo.getClassDescriptor().getName()).equals(mainClass));
                                gen.setSerializer(serializer);
                                gen.setClasses(classes);
                                gen.setLauncherName(launcherName);
                                gen.setCSSFiles(mo.getCssFiles());
                                //generates the file without databinding
                                JavaFile generatedFile = gen.generate();
                                if(generatedFile != null) {
                                    generatedFiles.put(gen, generatedFile);
                                    classes.put(mo.getClassDescriptor().toString(), generatedFile);
                                }
                            }
                            catch (InstantiationException eee) {
                                if (log.isErrorEnabled()) {
                                    log.error(eee);
                                }
                            }
                            catch (IllegalAccessException eee) {
                                if (log.isErrorEnabled()) {
                                    log.error(eee);
                                }
                            }
                        }
                        //generation of the databinding
                        for (GuixGenerator gen : generatedFiles.keySet()) {
                            //int dataSourceNumber = 0;
                            //databinding creation method
                            StringBuffer dbCreation = new StringBuffer();
                            //databinding deletion method
                            StringBuffer dbDeletion = new StringBuffer();
                            //databinding execution method
                            StringBuffer dbProcess = new StringBuffer();
                            JavaFile jf = generatedFiles.get(gen);
                            //for each field which binds the value of others objects
                            for (String field : gen.getBindingsToGenerate().keySet()) {
                                for (String attr : gen.getBindingsToGenerate().get(field).keySet()) {
                                    try {
                                        dbCreation.append(dbCreation.length() != 0 ? "else if(\"" : "if(\"").append(field).append(".").append(attr).append("\".equals(_binding)) {\n");
                                        dbDeletion.append(dbDeletion.length() != 0 ? "else if(\"" : "if(\"").append(field).append(".").append(attr).append("\".equals(_binding)) {\n");
                                        dbProcess.append(dbProcess.length() != 0 ? "else if(\"" : "if(\"").append(field).append(".").append(attr).append("\".equals(_binding)) {\n");
                                        List<Class> listeners = new ArrayList<Class>();

//                                        if(gen.getBindingsToGenerate().get(field).get(attr).startsWith("new ")) {
//                                            listeners = null;
//                                        }
//                                        else {
                                            List<String[]> bindings = new ArrayList<String[]>();
                                            //the parser which decompose the binding into the different elements to bind
                                            JavaParser p = new JavaParser(new StringReader(gen.getBindingsToGenerate().get(field).get(attr)));
                                            //start parsing the binding value
                                            while (!p.Line()) {
                                                SimpleNode node = p.popNode();
                                                if (node != null) {
                                                    List<String[]> l = BindingUtils.scanNode(node);
                                                    if(l != null) {
                                                        bindings.addAll(l);
                                                    }

//                                                    if(l != null) {
//                                                        for(String s : l) {
//                                                            bindings.add(s.split("\\.").length > 0 ? s.split("\\.") : new String[]{s});
//                                                        }
//                                                    }
                                                    //get the different elements to bind
//                                                    for (String s : browseNode(node)) {
//                                                        //decompose the element s into attributes
//                                                        List<String> l = new ArrayList<String>();
//                                                        //number of open brackets
//                                                        int parOuvertes = 0;
//                                                        StringBuffer read = new StringBuffer();
//                                                        for (char c : s.toCharArray()) {
//                                                            if (c == '(') {
//                                                                parOuvertes++;
//                                                                read.append(c);
//                                                            }
//                                                            else if (c == ')') {
//                                                                parOuvertes--;
//                                                                read.append(c);
//                                                            }
//                                                            else if (c == '.') {
//                                                                if (parOuvertes == 0) {
//                                                                    l.add(read.toString());
//                                                                    read = new StringBuffer();
//                                                                }
//                                                                else {
//                                                                    read.append(c);
//                                                                }
//                                                            }
//                                                            else {
//                                                                read.append(c);
//                                                            }
//                                                        }
//                                                        l.add(read.toString());
//                                                        bindings.add(l.toArray(new String[l.size()]));
//                                                    }
                                                }
                                            }
                                            //generates the code
                                            for (String[] binding : bindings) {
                                                StringBuffer methodToInvoke = new StringBuffer();
                                                //the binding value before the generator changes it
                                                StringBuffer oldBinding = new StringBuffer();
                                                //generates the method name for calling the databinding process method
                                                for (String s : binding) {
                                                    methodToInvoke.append(s.replaceAll("\\W", ""));
                                                    oldBinding.append(s).append(".");
                                                }
                                                oldBinding.setLength(oldBinding.length() - 1);
                                                //generates the code and modify the simple attribute name in the binding by the getter
                                                listeners.addAll(gen.generateBindings(dbCreation, dbDeletion, null, jf, null, binding, 0, null, methodToInvoke.toString(), generatedFiles));
                                                //JavaArgument[] args = new JavaArgument[]{new JavaArgument(parameterType.getName(), "event")};
                                                //new value of the binding with the getters
                                                StringBuffer newBinding = new StringBuffer();
                                                for (String s : binding) {
                                                    newBinding.append(s).append(".");
                                                }
                                                newBinding.setLength(newBinding.length() - 1);
                                                //replaces the old binding by the newer one
                                                gen.getBindingsToGenerate().get(field).put(attr, gen.getBindingsToGenerate().get(field).get(attr).replace(oldBinding.toString(), newBinding.toString()));

                                                //method called by the listener
                                                if (jf.getMethod("onChangeFrom" + methodToInvoke, /*args*/ null) == null) {
                                                    jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "onChangeFrom" + methodToInvoke.toString(), /*args*/ null, null, /*"_DataSource" + dataSourceNumber + ".propertyChange(null);"*/ "processDataBinding(\"" + field + "." + attr + "\");", "Method called when the objects bound by '" + attr + "' of '" + field + "' change"));
                                                }
                                                else {
                                                    jf.getMethod("onChangeFrom" + methodToInvoke.toString(), /*args*/ null).appendBodyCode(/*"_DataSource" + dataSourceNumber + ".propertyChange(null);"*/"processDataBinding(\"" + field + "." + attr + "\");", "\n");
                                                }
                                            }
//                                        }
                                        dbCreation.append("}\n");
                                        dbDeletion.append("}\n");
                                        dbProcess.append(field).append(".set").append(Character.toUpperCase(attr.charAt(0))).append(attr.substring(1)).append("(").append(gen.getBindingsToGenerate().get(field).get(attr)).append(");\n}\n");
                                        //if the binding got some elements to bind
                                        if (listeners != null) {
                                            //jf.addField(new JavaField(Modifier.PRIVATE, "java.beans.PropertyChangeListener", "_DataSource" + dataSourceNumber, "new org.nuiton.guix.runtime.DataBindingListener(this,\"" + field + "." + attr + "\")", null, null));
                                            //method to init the databinding
                                            jf.getMethod("initDataBinding", null).appendBodyCode("applyDataBinding(\"" + field + "." + attr + "\");", "\n");
    //                                        for (Class listener : listeners) {
    //                                            final List<Method> listenerMethods = Arrays.asList(listener.getMethods());
    //                                            Class parameterType = listenerMethods.get(0).getParameterTypes()[0];
    //                                        }
                                        }
                                        //dataSourceNumber++;
                                    }
                                    catch(NullPointerException eee) {
                                        dbCreation.append("}\n");
                                        dbDeletion.append("}\n");
                                        dbProcess.append("}\n");
                                    }
                                }
                            }
                            //if the generated file has a superclass which also is a generated class, apply the databinding of the superclass to the class
                            if (generatedFiles.get(jf.getSuperClass()) != null) {
                                dbCreation.append("else {\nsuper.applyDataBinding(_binding);\n" +
                                        "return;\n}");
                                dbDeletion.append("\nelse {\nsuper.removeDataBinding(_binding);\n}\n");
                            }
                            dbCreation.append("\nprocessDataBinding(_binding);");
                            dbProcess.insert(0, "if (activeBindings.contains(_binding)) {\n" +
                                    "return;\n}\n" +
                                    "activeBindings.add(_binding);\ntry{\n");
                            dbProcess.append("\n} finally {\nactiveBindings.remove(_binding);\n}\n");

                            if(!jf.isSuperclassIsGuixObject()) {
                                jf.addSimpleField(new JavaField(Modifier.PROTECTED, "java.util.List<String>", "activeBindings", "new java.util.ArrayList<String>()", "List of the active bindings", null));
                            }
                            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "applyDataBinding", new JavaArgument[]{new JavaArgument("String", "_binding")}, null,
                                    (jf.isSuperclassIsGuixObject() ? "super.applyDataBinding(_binding);\n" : "") + dbCreation.toString(), "Adds the listeners to the elements which are bound by another attribute"));
//                            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "removeDataBinding", new JavaArgument[]{new JavaArgument("String", "_binding")}, null,
//                                    (jf.isSuperclassIsGuixObject() ? "super.removeDataBinding(_binding);\n" : "") + dbDeletion.toString(), "Removes the listeners to the elements which are bound by another attribute"));
                            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "processDataBinding", new JavaArgument[]{new JavaArgument("String", "_binding")}, null,
                                    (jf.isSuperclassIsGuixObject() ? "super.processDataBinding(_binding);\n" : "") + dbProcess.toString(), "Executes the binding instruction"));
                            //saves the files
                            gen.saveFiles();
                        }
                        serializer.endTag("", "beans");
                        serializer.endDocument();
                    }
                    else {
                        if (log.isWarnEnabled()) {
                            log.warn("No generation language given");
                        }
                    }
                }
            }
            catch (Throwable e) {
                e.printStackTrace();
                return false;
            }
            return success;
        }
        else {
            if (log.isWarnEnabled()) {
                log.warn("No file to compile");
            }
            return true;
        }
    }

    /**
     * Registers a ClassDescriptor
     *
     * @param classDescriptor   the ClassDescripor to register
     * @return false            if another ClassDescriptor with the same classname
     *                          and package has a different script or superclass
     */
    public ClassDescriptor registerClassDescriptor(ClassDescriptor classDescriptor) {
        if (classDescriptor == null || classDescriptor.getName() == null) {
            return null;
        }
        int i = 0;
        while (i < classDescriptors.size() && (!classDescriptors.get(i).getName().equals(classDescriptor.getName()) || (classDescriptors.get(i).getPackageName() != null && classDescriptor.getPackageName() != null && !classDescriptors.get(i).getPackageName().equals(classDescriptor.getPackageName())))) {
            i++;
        }
        //if the ClassDescriptor does not exist
        if (i >= classDescriptors.size()) {
            ClassDescriptor cd = registerClassDescriptor(classDescriptor.getSuperClass());
            classDescriptor.setSuperClass(cd);
            classDescriptors.add(classDescriptor);
            if (log.isDebugEnabled()) {
                log.debug("new ClassDescriptor " + classDescriptor + " inserted");
            }
            return classDescriptor;
        }
        if(classDescriptor.getPackageName() != null) {
            classDescriptors.get(i).setPackageName(classDescriptor.getPackageName());
        }
        if (classDescriptor.getScript() != null) {
            if (classDescriptors.get(i).getScript() == null) {
                classDescriptors.get(i).setScript(classDescriptor.getScript());
                if (log.isDebugEnabled()) {
                    log.debug("add script to ClassDescriptor " + classDescriptor);
                }
            }
            else if (!classDescriptors.get(i).getScript().equals(classDescriptor.getScript())) {
                if (log.isDebugEnabled()) {
                    log.error("ClassDescriptor " + classDescriptor + " script already defined and different !");
                }
                return null;
            }
        }
        if (classDescriptor.getSuperClass() != null) {
            ClassDescriptor cd = registerClassDescriptor(classDescriptor.getSuperClass());
            if (classDescriptors.get(i).getSuperClass() == null) {
                classDescriptors.get(i).setSuperClass(cd);
                if (log.isDebugEnabled()) {
                    log.debug("add superclass to ClassDescriptor " + classDescriptor);
                }
            }
            else if (!classDescriptors.get(i).getSuperClass().equals(cd)) {
                if (log.isErrorEnabled()) {
                    log.error("ClassDescriptor " + classDescriptor + " superclass already defined and different !");
                }
                return null;
            }
        }
        return classDescriptors.get(i);
    }

    /**
     * Adds a root GuixModelObject with its last modification time
     *
     * @param gmo the root GuixModelObject to add
     * @param l its last modification time
     */
    public void addRootModelObject(GuixModelObject gmo, Long l) {
        rootModelObjects.put(gmo, l);
        rootClassDescriptors.put(gmo.getClassDescriptor(), gmo);
    }

    public void addClassName(String className) {
        if(!classNames.contains(className)) {
            classNames.add(className);
        }
    }

    /**
     * Ads a file to the compiled files list
     *
     * @param f the file to add
     */
    public void addCompiledFile(File f) {
        compiledFiles.add(f);
    }

    /**
     * Has the file f already been compiled
     *
     * @param f the file we'd like to know if it has already been compiled
     * @return true if f has already been compiled
     */
    public boolean isFileAlreadyCompiled(File f) {
        return compiledFiles.contains(f);
    }

    public void addDependency(GuixModelObject gmo, ClassDescriptor cd) {
        if(!dependencies.containsKey(gmo)) {
            dependencies.put(gmo, new ArrayList<ClassDescriptor>());
        }
        if(!dependencies.get(gmo).contains(cd)) {
            dependencies.get(gmo).add(cd);
        }
    }

    private boolean orderDependencies(GuixModelObject mo, List<GuixModelObject> rootMos, Map<ClassDescriptor, GuixModelObject> rootCDs, int i) {
        if(!rootMos.contains(mo)) {
            if(dependencies.get(mo) != null) {
                for(ClassDescriptor cd : dependencies.get(mo)) {
                    if(rootCDs.get(cd) != null && !rootMos.contains(rootCDs.get(cd))) {
                        if(!orderDependencies(rootCDs.get(cd), rootMos, rootCDs, -1)) {
                            return false;
                        }
                    }
                }
            }
            if(i < 0) {
                rootMos.add(mo);
            }
            else {
                rootMos.add(i, mo);
            }
        }
        else {
            if(dependencies.get(mo) != null) {
                for(ClassDescriptor cd : dependencies.get(mo)) {
                    if(rootCDs.get(cd) != null && !rootMos.contains(rootCDs.get(cd))) {
                        if(!orderDependencies(rootCDs.get(cd), rootMos, rootCDs, rootMos.indexOf(mo))) {
                            return false;
                        }
                    }
                    else if(rootCDs.get(cd) != null && rootMos.indexOf(mo) < rootMos.indexOf(rootCDs.get(cd))) {
                        return false;
                    }
                }
            }
        }
        return true;
    }

}
