/*      */ package org.quartz;
/*      */ 
/*      */ import java.util.Collections;
/*      */ import java.util.Date;
/*      */ import java.util.LinkedList;
/*      */ import java.util.List;
/*      */ import java.util.TimeZone;
/*      */ 
/*      */ public class TriggerUtils
/*      */ {
/*      */   public static final int SUNDAY = 1;
/*      */   public static final int MONDAY = 2;
/*      */   public static final int TUESDAY = 3;
/*      */   public static final int WEDNESDAY = 4;
/*      */   public static final int THURSDAY = 5;
/*      */   public static final int FRIDAY = 6;
/*      */   public static final int SATURDAY = 7;
/*      */   public static final int LAST_DAY_OF_MONTH = -1;
/*      */   public static final long MILLISECONDS_IN_MINUTE = 60000L;
/*      */   public static final long MILLISECONDS_IN_HOUR = 3600000L;
/*      */   public static final long SECONDS_IN_DAY = 86400L;
/*      */   public static final long MILLISECONDS_IN_DAY = 86400000L;
/*      */ 
/*      */   private static void validateDayOfWeek(int dayOfWeek)
/*      */   {
/*   90 */     if ((dayOfWeek < 1) || (dayOfWeek > 7))
/*   91 */       throw new IllegalArgumentException("Invalid day of week.");
/*      */   }
/*      */ 
/*      */   private static void validateHour(int hour) {
/*   95 */     if ((hour < 0) || (hour > 23))
/*   96 */       throw new IllegalArgumentException("Invalid hour (must be >= 0 and <= 23).");
/*      */   }
/*      */ 
/*      */   private static void validateMinute(int minute)
/*      */   {
/*  101 */     if ((minute < 0) || (minute > 59))
/*  102 */       throw new IllegalArgumentException("Invalid minute (must be >= 0 and <= 59).");
/*      */   }
/*      */ 
/*      */   private static void validateSecond(int second)
/*      */   {
/*  107 */     if ((second < 0) || (second > 59))
/*  108 */       throw new IllegalArgumentException("Invalid second (must be >= 0 and <= 59).");
/*      */   }
/*      */ 
/*      */   private static void validateDayOfMonth(int day)
/*      */   {
/*  113 */     if (((day < 1) || (day > 31)) && (day != -1))
/*  114 */       throw new IllegalArgumentException("Invalid day of month.");
/*      */   }
/*      */ 
/*      */   private static void validateMonth(int month) {
/*  118 */     if ((month < 1) || (month > 12))
/*  119 */       throw new IllegalArgumentException("Invalid month (must be >= 1 and <= 12.");
/*      */   }
/*      */ 
/*      */   private static void validateYear(int year)
/*      */   {
/*  124 */     if ((year < 1970) || (year > 2099))
/*  125 */       throw new IllegalArgumentException("Invalid year (must be >= 1970 and <= 2099.");
/*      */   }
/*      */ 
/*      */   public static void setTriggerIdentity(Trigger trig, String name)
/*      */   {
/*  139 */     setTriggerIdentity(trig, name, "DEFAULT");
/*      */   }
/*      */ 
/*      */   public static void setTriggerIdentity(Trigger trig, String name, String group)
/*      */   {
/*  154 */     trig.setName(name);
/*  155 */     trig.setGroup(group);
/*      */   }
/*      */ 
/*      */   public static Trigger makeDailyTrigger(int hour, int minute)
/*      */   {
/*  173 */     validateHour(hour);
/*  174 */     validateMinute(minute);
/*      */ 
/*  176 */     CronTrigger trig = new CronTrigger();
/*      */     try
/*      */     {
/*  179 */       trig.setCronExpression("0 " + minute + " " + hour + " ? * *");
/*      */     } catch (Exception ignore) {
/*  181 */       return null;
/*      */     }
/*      */ 
/*  184 */     trig.setStartTime(new Date());
/*      */ 
/*  186 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeDailyTrigger(String trigName, int hour, int minute)
/*      */   {
/*  206 */     Trigger trig = makeDailyTrigger(hour, minute);
/*  207 */     trig.setName(trigName);
/*  208 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeWeeklyTrigger(int dayOfWeek, int hour, int minute)
/*      */   {
/*  236 */     validateDayOfWeek(dayOfWeek);
/*  237 */     validateHour(hour);
/*  238 */     validateMinute(minute);
/*      */ 
/*  240 */     CronTrigger trig = new CronTrigger();
/*      */     try
/*      */     {
/*  243 */       trig.setCronExpression("0 " + minute + " " + hour + " ? * " + dayOfWeek);
/*      */     }
/*      */     catch (Exception ignore) {
/*  246 */       return null;
/*      */     }
/*      */ 
/*  249 */     trig.setStartTime(new Date());
/*      */ 
/*  251 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeWeeklyTrigger(String trigName, int dayOfWeek, int hour, int minute)
/*      */   {
/*  280 */     Trigger trig = makeWeeklyTrigger(dayOfWeek, hour, minute);
/*  281 */     trig.setName(trigName);
/*  282 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeMonthlyTrigger(int dayOfMonth, int hour, int minute)
/*      */   {
/*  310 */     validateDayOfMonth(dayOfMonth);
/*  311 */     validateHour(hour);
/*  312 */     validateMinute(minute);
/*      */ 
/*  314 */     CronTrigger trig = new CronTrigger();
/*      */     try
/*      */     {
/*  317 */       if (dayOfMonth != -1) trig.setCronExpression("0 " + minute + " " + hour + " " + dayOfMonth + " * ?");
/*      */       else
/*      */       {
/*  320 */         trig.setCronExpression("0 " + minute + " " + hour + " L * ?");
/*      */       }
/*      */     } catch (Exception ignore) {
/*  322 */       return null;
/*      */     }
/*      */ 
/*  325 */     trig.setStartTime(new Date());
/*      */ 
/*  327 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeMonthlyTrigger(String trigName, int dayOfMonth, int hour, int minute)
/*      */   {
/*  355 */     Trigger trig = makeMonthlyTrigger(dayOfMonth, hour, minute);
/*  356 */     trig.setName(trigName);
/*  357 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeImmediateTrigger(int repeatCount, long repeatInterval)
/*      */   {
/*  397 */     SimpleTrigger trig = new SimpleTrigger();
/*  398 */     trig.setStartTime(new Date());
/*  399 */     trig.setRepeatCount(repeatCount);
/*  400 */     trig.setRepeatInterval(repeatInterval);
/*  401 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeImmediateTrigger(String trigName, int repeatCount, long repeatInterval)
/*      */   {
/*  422 */     Trigger trig = makeImmediateTrigger(repeatCount, repeatInterval);
/*  423 */     trig.setName(trigName);
/*  424 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeSecondlyTrigger()
/*      */   {
/*  439 */     return makeSecondlyTrigger(1, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeSecondlyTrigger(String trigName)
/*      */   {
/*  456 */     return makeSecondlyTrigger(trigName, 1, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeSecondlyTrigger(int intervalInSeconds)
/*      */   {
/*  475 */     return makeSecondlyTrigger(intervalInSeconds, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeSecondlyTrigger(int intervalInSeconds, int repeatCount)
/*      */   {
/*  496 */     SimpleTrigger trig = new SimpleTrigger();
/*      */ 
/*  498 */     trig.setRepeatInterval(intervalInSeconds * 1000L);
/*  499 */     trig.setRepeatCount(repeatCount);
/*      */ 
/*  501 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeSecondlyTrigger(String trigName, int intervalInSeconds, int repeatCount)
/*      */   {
/*  522 */     Trigger trig = makeSecondlyTrigger(intervalInSeconds, repeatCount);
/*  523 */     trig.setName(trigName);
/*  524 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeMinutelyTrigger()
/*      */   {
/*  540 */     return makeMinutelyTrigger(1, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeMinutelyTrigger(String trigName)
/*      */   {
/*  557 */     return makeMinutelyTrigger(trigName, 1, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeMinutelyTrigger(int intervalInMinutes)
/*      */   {
/*  575 */     return makeMinutelyTrigger(intervalInMinutes, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeMinutelyTrigger(int intervalInMinutes, int repeatCount)
/*      */   {
/*  596 */     SimpleTrigger trig = new SimpleTrigger();
/*      */ 
/*  598 */     trig.setRepeatInterval(intervalInMinutes * 60000L);
/*  599 */     trig.setRepeatCount(repeatCount);
/*      */ 
/*  601 */     trig.setStartTime(new Date());
/*      */ 
/*  603 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeMinutelyTrigger(String trigName, int intervalInMinutes, int repeatCount)
/*      */   {
/*  624 */     Trigger trig = makeMinutelyTrigger(intervalInMinutes, repeatCount);
/*  625 */     trig.setName(trigName);
/*  626 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeHourlyTrigger()
/*      */   {
/*  642 */     return makeHourlyTrigger(1, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeHourlyTrigger(String trigName)
/*      */   {
/*  659 */     return makeHourlyTrigger(trigName, 1, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeHourlyTrigger(int intervalInHours)
/*      */   {
/*  677 */     return makeHourlyTrigger(intervalInHours, SimpleTrigger.REPEAT_INDEFINITELY);
/*      */   }
/*      */ 
/*      */   public static Trigger makeHourlyTrigger(int intervalInHours, int repeatCount)
/*      */   {
/*  698 */     SimpleTrigger trig = new SimpleTrigger();
/*      */ 
/*  700 */     trig.setRepeatInterval(intervalInHours * 3600000L);
/*  701 */     trig.setRepeatCount(repeatCount);
/*      */ 
/*  703 */     trig.setStartTime(new Date());
/*      */ 
/*  705 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Trigger makeHourlyTrigger(String trigName, int intervalInHours, int repeatCount)
/*      */   {
/*  726 */     Trigger trig = makeHourlyTrigger(intervalInHours, repeatCount);
/*  727 */     trig.setName(trigName);
/*  728 */     return trig;
/*      */   }
/*      */ 
/*      */   public static Date getEvenHourDate(Date date)
/*      */   {
/*  749 */     if (date == null) date = new Date();
/*      */ 
/*  751 */     java.util.Calendar c = java.util.Calendar.getInstance();
/*  752 */     c.setTime(date);
/*  753 */     c.setLenient(true);
/*      */ 
/*  755 */     c.set(11, c.get(11) + 1);
/*  756 */     c.set(12, 0);
/*  757 */     c.set(13, 0);
/*  758 */     c.set(14, 0);
/*      */ 
/*  760 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getEvenHourDateBefore(Date date)
/*      */   {
/*  780 */     if (date == null) date = new Date();
/*      */ 
/*  782 */     java.util.Calendar c = java.util.Calendar.getInstance();
/*  783 */     c.setTime(date);
/*      */ 
/*  785 */     c.set(12, 0);
/*  786 */     c.set(13, 0);
/*  787 */     c.set(14, 0);
/*      */ 
/*  789 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getEvenMinuteDate(Date date)
/*      */   {
/*  810 */     if (date == null) date = new Date();
/*      */ 
/*  812 */     java.util.Calendar c = java.util.Calendar.getInstance();
/*  813 */     c.setTime(date);
/*  814 */     c.setLenient(true);
/*      */ 
/*  816 */     c.set(12, c.get(12) + 1);
/*  817 */     c.set(13, 0);
/*  818 */     c.set(14, 0);
/*      */ 
/*  820 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getEvenMinuteDateBefore(Date date)
/*      */   {
/*  840 */     if (date == null) date = new Date();
/*      */ 
/*  842 */     java.util.Calendar c = java.util.Calendar.getInstance();
/*  843 */     c.setTime(date);
/*      */ 
/*  845 */     c.set(13, 0);
/*  846 */     c.set(14, 0);
/*      */ 
/*  848 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getEvenSecondDate(Date date)
/*      */   {
/*  863 */     if (date == null) date = new Date();
/*      */ 
/*  865 */     java.util.Calendar c = java.util.Calendar.getInstance();
/*  866 */     c.setTime(date);
/*  867 */     c.setLenient(true);
/*      */ 
/*  869 */     c.set(13, c.get(13) + 1);
/*  870 */     c.set(14, 0);
/*      */ 
/*  872 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getEvenSecondDateBefore(Date date)
/*      */   {
/*  892 */     if (date == null) date = new Date();
/*      */ 
/*  894 */     java.util.Calendar c = java.util.Calendar.getInstance();
/*  895 */     c.setTime(date);
/*      */ 
/*  897 */     c.set(14, 0);
/*      */ 
/*  899 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getNextGivenMinuteDate(Date date, int minuteBase)
/*      */   {
/*  996 */     if ((minuteBase < 0) || (minuteBase > 59)) {
/*  997 */       throw new IllegalArgumentException("minuteBase must be >=0 and <= 59");
/*      */     }
/*      */ 
/* 1000 */     if (date == null) date = new Date();
/*      */ 
/* 1002 */     java.util.Calendar c = java.util.Calendar.getInstance();
/* 1003 */     c.setTime(date);
/* 1004 */     c.setLenient(true);
/*      */ 
/* 1006 */     if (minuteBase == 0) {
/* 1007 */       c.set(11, c.get(11) + 1);
/* 1008 */       c.set(12, 0);
/* 1009 */       c.set(13, 0);
/* 1010 */       c.set(14, 0);
/*      */ 
/* 1012 */       return c.getTime();
/*      */     }
/*      */ 
/* 1015 */     int minute = c.get(12);
/*      */ 
/* 1017 */     int arItr = minute / minuteBase;
/*      */ 
/* 1019 */     int nextMinuteOccurance = minuteBase * (arItr + 1);
/*      */ 
/* 1021 */     if (nextMinuteOccurance < 60) {
/* 1022 */       c.set(12, nextMinuteOccurance);
/* 1023 */       c.set(13, 0);
/* 1024 */       c.set(14, 0);
/*      */ 
/* 1026 */       return c.getTime();
/*      */     }
/* 1028 */     c.set(11, c.get(11) + 1);
/* 1029 */     c.set(12, 0);
/* 1030 */     c.set(13, 0);
/* 1031 */     c.set(14, 0);
/*      */ 
/* 1033 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getNextGivenSecondDate(Date date, int secondBase)
/*      */   {
/* 1057 */     if ((secondBase < 0) || (secondBase > 59)) {
/* 1058 */       throw new IllegalArgumentException("secondBase must be >=0 and <= 59");
/*      */     }
/*      */ 
/* 1061 */     if (date == null) date = new Date();
/*      */ 
/* 1063 */     java.util.Calendar c = java.util.Calendar.getInstance();
/* 1064 */     c.setTime(date);
/* 1065 */     c.setLenient(true);
/*      */ 
/* 1067 */     if (secondBase == 0) {
/* 1068 */       c.set(12, c.get(12) + 1);
/* 1069 */       c.set(13, 0);
/* 1070 */       c.set(14, 0);
/*      */ 
/* 1072 */       return c.getTime();
/*      */     }
/*      */ 
/* 1075 */     int second = c.get(13);
/*      */ 
/* 1077 */     int arItr = second / secondBase;
/*      */ 
/* 1079 */     int nextSecondOccurance = secondBase * (arItr + 1);
/*      */ 
/* 1081 */     if (nextSecondOccurance < 60) {
/* 1082 */       c.set(13, nextSecondOccurance);
/* 1083 */       c.set(14, 0);
/*      */ 
/* 1085 */       return c.getTime();
/*      */     }
/* 1087 */     c.set(12, c.get(12) + 1);
/* 1088 */     c.set(13, 0);
/* 1089 */     c.set(14, 0);
/*      */ 
/* 1091 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getDateOf(int second, int minute, int hour)
/*      */   {
/* 1110 */     validateSecond(second);
/* 1111 */     validateMinute(minute);
/* 1112 */     validateHour(hour);
/*      */ 
/* 1114 */     Date date = new Date();
/*      */ 
/* 1116 */     java.util.Calendar c = java.util.Calendar.getInstance();
/* 1117 */     c.setTime(date);
/* 1118 */     c.setLenient(true);
/*      */ 
/* 1120 */     c.set(11, hour);
/* 1121 */     c.set(12, minute);
/* 1122 */     c.set(13, second);
/* 1123 */     c.set(14, 0);
/*      */ 
/* 1125 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getDateOf(int second, int minute, int hour, int dayOfMonth, int month)
/*      */   {
/* 1148 */     validateSecond(second);
/* 1149 */     validateMinute(minute);
/* 1150 */     validateHour(hour);
/* 1151 */     validateDayOfMonth(dayOfMonth);
/* 1152 */     validateMonth(month);
/*      */ 
/* 1154 */     Date date = new Date();
/*      */ 
/* 1156 */     java.util.Calendar c = java.util.Calendar.getInstance();
/* 1157 */     c.setTime(date);
/*      */ 
/* 1159 */     c.set(2, month - 1);
/* 1160 */     c.set(5, dayOfMonth);
/* 1161 */     c.set(11, hour);
/* 1162 */     c.set(12, minute);
/* 1163 */     c.set(13, second);
/* 1164 */     c.set(14, 0);
/*      */ 
/* 1166 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static Date getDateOf(int second, int minute, int hour, int dayOfMonth, int month, int year)
/*      */   {
/* 1191 */     validateSecond(second);
/* 1192 */     validateMinute(minute);
/* 1193 */     validateHour(hour);
/* 1194 */     validateDayOfMonth(dayOfMonth);
/* 1195 */     validateMonth(month);
/* 1196 */     validateYear(year);
/*      */ 
/* 1198 */     Date date = new Date();
/*      */ 
/* 1200 */     java.util.Calendar c = java.util.Calendar.getInstance();
/* 1201 */     c.setTime(date);
/*      */ 
/* 1203 */     c.set(1, year);
/* 1204 */     c.set(2, month - 1);
/* 1205 */     c.set(5, dayOfMonth);
/* 1206 */     c.set(11, hour);
/* 1207 */     c.set(12, minute);
/* 1208 */     c.set(13, second);
/* 1209 */     c.set(14, 0);
/*      */ 
/* 1211 */     return c.getTime();
/*      */   }
/*      */ 
/*      */   public static List computeFireTimes(Trigger trigg, Calendar cal, int numTimes)
/*      */   {
/* 1230 */     LinkedList lst = new LinkedList();
/*      */ 
/* 1232 */     Trigger t = (Trigger)trigg.clone();
/*      */ 
/* 1234 */     if (t.getNextFireTime() == null) {
/* 1235 */       t.computeFirstFireTime(cal);
/*      */     }
/*      */ 
/* 1238 */     for (int i = 0; i < numTimes; i++) {
/* 1239 */       Date d = t.getNextFireTime();
/* 1240 */       if (d == null) break;
/* 1241 */       lst.add(d);
/* 1242 */       t.triggered(cal);
/*      */     }
/*      */ 
/* 1247 */     return Collections.unmodifiableList(lst);
/*      */   }
/*      */ 
/*      */   public static List computeFireTimesBetween(Trigger trigg, Calendar cal, Date from, Date to)
/*      */   {
/* 1275 */     LinkedList lst = new LinkedList();
/*      */ 
/* 1277 */     Trigger t = (Trigger)trigg.clone();
/*      */ 
/* 1279 */     if (t.getNextFireTime() == null) {
/* 1280 */       t.setStartTime(from);
/* 1281 */       t.setEndTime(to);
/* 1282 */       t.computeFirstFireTime(cal);
/*      */     }
/*      */ 
/*      */     while (true)
/*      */     {
/* 1288 */       Date d = t.getNextFireTime();
/* 1289 */       if (d == null) break;
/* 1290 */       if (d.before(from)) {
/* 1291 */         t.triggered(cal);
/* 1292 */         continue;
/*      */       }
/* 1294 */       if (d.after(to)) break;
/* 1295 */       lst.add(d);
/* 1296 */       t.triggered(cal);
/*      */     }
/*      */ 
/* 1301 */     return Collections.unmodifiableList(lst);
/*      */   }
/*      */ 
/*      */   public static Date translateTime(Date date, TimeZone src, TimeZone dest)
/*      */   {
/* 1316 */     Date newDate = new Date();
/*      */ 
/* 1318 */     int offset = getOffset(date.getTime(), dest) - getOffset(date.getTime(), src);
/*      */ 
/* 1322 */     newDate.setTime(date.getTime() - offset);
/*      */ 
/* 1324 */     return newDate;
/*      */   }
/*      */ 
/*      */   public static int getOffset(long date, TimeZone tz)
/*      */   {
/* 1342 */     if (tz.inDaylightTime(new Date(date))) {
/* 1343 */       return tz.getRawOffset() + getDSTSavings(tz);
/*      */     }
/*      */ 
/* 1346 */     return tz.getRawOffset();
/*      */   }
/*      */ 
/*      */   public static int getDSTSavings(TimeZone tz)
/*      */   {
/* 1360 */     if (tz.useDaylightTime()) {
/* 1361 */       return 3600000;
/*      */     }
/* 1363 */     return 0;
/*      */   }
/*      */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.quartz.TriggerUtils
 * JD-Core Version:    0.6.0
 */