package org.hsqldb.persist;

import java.io.File;
import java.io.IOException;
import org.hsqldb.Database;
import org.hsqldb.HsqlException;
import org.hsqldb.Trace;
import org.hsqldb.lib.FileAccess;
import org.hsqldb.lib.FileUtil;
import org.hsqldb.lib.HsqlByteArrayOutputStream;
import org.hsqldb.lib.SimpleLog;
import org.hsqldb.lib.StopWatch;
import org.hsqldb.lib.Storage;
import org.hsqldb.lib.ZipUnzipFile;
import org.hsqldb.rowio.RowInputBinary;
import org.hsqldb.rowio.RowInputInterface;
import org.hsqldb.rowio.RowOutputBinary;
import org.hsqldb.rowio.RowOutputInterface;
import org.hsqldb.store.BitMap;

public class DataFileCache
{
  protected FileAccess fa;
  public static final int FLAG_ISSAVED = 2;
  public static final int FLAG_ROWINFO = 3;
  static final int LONG_EMPTY_SIZE = 4;
  static final int LONG_FREE_POS_POS = 12;
  static final int LONG_EMPTY_INDEX_POS = 20;
  static final int FLAGS_POS = 28;
  static final int INITIAL_FREE_POS = 32;
  DataFileBlockManager freeBlocks;
  private static final int initIOBufferSize = 256;
  protected String fileName;
  protected String backupFileName;
  protected Database database;
  protected boolean fileModified;
  protected int cacheFileScale;
  protected boolean cacheReadonly;
  protected boolean storeOnInsert;
  protected int cachedRowPadding = 8;
  protected boolean hasRowInfo = false;
  protected RowInputInterface rowIn;
  protected RowOutputInterface rowOut;
  public long maxDataFileSize;
  protected Storage dataFile;
  protected long fileFreePosition;
  protected int maxCacheSize;
  protected long maxCacheBytes;
  protected int maxFreeBlocks;
  protected Cache cache;

  public DataFileCache(Database paramDatabase, String paramString)
    throws HsqlException
  {
    initParams(paramDatabase, paramString);
    this.cache = new Cache(this);
  }

  protected void initParams(Database paramDatabase, String paramString)
    throws HsqlException
  {
    HsqlDatabaseProperties localHsqlDatabaseProperties = paramDatabase.getProperties();
    this.fileName = (paramString + ".data");
    this.backupFileName = (paramString + ".backup");
    this.database = paramDatabase;
    this.fa = paramDatabase.getFileAccess();
    int i = localHsqlDatabaseProperties.getIntegerProperty("hsqldb.cache_scale", 14, 8, 18);
    int j = localHsqlDatabaseProperties.getIntegerProperty("hsqldb.cache_size_scale", 10, 6, 20);
    int k = localHsqlDatabaseProperties.getIntegerProperty("hsqldb.cache_free_count_scale", 9, 6, 12);
    this.cacheFileScale = paramDatabase.getProperties().getIntegerProperty("hsqldb.cache_file_scale", 1);
    if (this.cacheFileScale != 1)
      this.cacheFileScale = 8;
    this.cacheReadonly = paramDatabase.isFilesReadOnly();
    int m = 1 << i;
    int n = 1 << j;
    this.maxCacheSize = (m * 3);
    this.maxCacheBytes = (this.maxCacheSize * n);
    this.maxDataFileSize = (this.cacheFileScale == 1 ? 2147483647L : 8589934588L);
    this.maxFreeBlocks = (1 << k);
    this.dataFile = null;
  }

  public void open(boolean paramBoolean)
    throws HsqlException
  {
    this.fileFreePosition = 0L;
    this.database.logger.appLog.logContext(SimpleLog.LOG_NORMAL, "start");
    try
    {
      boolean bool1 = this.database.isFilesInJar();
      long l = 0L;
      Object localObject;
      if ((!bool1) && (this.fa.isStreamElement(this.fileName)))
        if (this.database.isStoredFileAccess())
        {
          bool1 = true;
        }
        else
        {
          localObject = new File(this.fileName);
          bool1 = ((File)localObject).length() > 32L;
        }
      if (bool1)
      {
        localObject = this.database.getProperties().getProperty("hsqldb.cache_version");
        bool3 = "1.7.0".equals(localObject);
        boolean bool4 = "1.8.0".equals(localObject);
        if (!bool3)
          throw Trace.error(30);
      }
      boolean bool2 = this.database.getProperties().isPropertyTrue("hsqldb.nio_data_file");
      boolean bool3 = bool2;
      int i;
      if (this.database.isFilesInJar())
        i = 2;
      String str1 = this.database.getURLProperties().getProperty("storage_class_name");
      String str2 = this.database.getURLProperties().getProperty("storage_key");
      this.dataFile = ScaledRAFile.newScaledRAFile(this.database, this.fileName, paramBoolean, i, str1, str2);
      if (bool1)
      {
        this.dataFile.seek(28L);
        int j = this.dataFile.readInt();
        this.hasRowInfo = BitMap.isSet(j, 3);
        this.dataFile.seek(4L);
        l = this.dataFile.readLong();
        this.dataFile.seek(12L);
        this.fileFreePosition = this.dataFile.readLong();
        if (this.fileFreePosition < 32L)
          this.fileFreePosition = 32L;
      }
      else
      {
        this.fileFreePosition = 32L;
        this.dataFile.seek(12L);
        this.dataFile.writeLong(32L);
        this.dataFile.seek(28L);
        this.dataFile.writeInt(0);
      }
      initBuffers();
      this.fileModified = false;
      this.freeBlocks = new DataFileBlockManager(this.maxFreeBlocks, this.cacheFileScale, l);
      this.database.logger.appLog.logContext(SimpleLog.LOG_NORMAL, "end");
    }
    catch (Throwable localThrowable)
    {
      this.database.logger.appLog.logContext(localThrowable, "failed");
      close(false);
      throw Trace.error(29, 210, new Object[] { localThrowable, this.fileName });
    }
  }

  public void close(boolean paramBoolean)
    throws HsqlException
  {
    SimpleLog localSimpleLog = this.database.logger.appLog;
    try
    {
      if (this.cacheReadonly)
      {
        if (this.dataFile != null)
          this.dataFile.close();
        return;
      }
      StopWatch localStopWatch = new StopWatch();
      localSimpleLog.sendLine(SimpleLog.LOG_NORMAL, "DataFileCache.close(" + paramBoolean + ") : start");
      if (paramBoolean)
      {
        this.cache.saveAll();
        Trace.printSystemOut("saveAll: " + localStopWatch.elapsedTime());
        localSimpleLog.sendLine(SimpleLog.LOG_NORMAL, "DataFileCache.close() : save data");
        if ((this.fileModified) || (this.freeBlocks.isModified()))
        {
          this.dataFile.seek(4L);
          this.dataFile.writeLong(this.freeBlocks.getLostBlocksSize());
          this.dataFile.seek(12L);
          this.dataFile.writeLong(this.fileFreePosition);
          this.dataFile.seek(28L);
          i = BitMap.set(0, 2);
          if (this.hasRowInfo)
            i = BitMap.set(i, 3);
          this.dataFile.writeInt(i);
          localSimpleLog.sendLine(SimpleLog.LOG_NORMAL, "DataFileCache.close() : flags");
          if (this.dataFile.length() != this.fileFreePosition)
            this.dataFile.seek(this.fileFreePosition);
          localSimpleLog.sendLine(SimpleLog.LOG_NORMAL, "DataFileCache.close() : seek end");
          Trace.printSystemOut("pos and flags: " + localStopWatch.elapsedTime());
        }
      }
      if (this.dataFile != null)
      {
        this.dataFile.close();
        localSimpleLog.sendLine(SimpleLog.LOG_NORMAL, "DataFileCache.close() : close");
        this.dataFile = null;
        Trace.printSystemOut("close: " + localStopWatch.elapsedTime());
      }
      int i = this.fileFreePosition == 32L ? 1 : 0;
      if (i != 0)
      {
        this.fa.removeElement(this.fileName);
        this.fa.removeElement(this.backupFileName);
      }
    }
    catch (Throwable localThrowable)
    {
      localSimpleLog.logContext(localThrowable, null);
      throw Trace.error(29, 211, new Object[] { localThrowable, this.fileName });
    }
  }

  protected void initBuffers()
  {
    if ((this.rowOut == null) || (((RowOutputBinary)this.rowOut).getBuffer().length > 256))
      this.rowOut = new RowOutputBinary(256);
    if ((this.rowIn == null) || (((RowInputBinary)this.rowIn).getBuffer().length > 256))
      this.rowIn = new RowInputBinary(new byte[256]);
  }

  public void defrag()
    throws HsqlException
  {
    if (this.cacheReadonly)
      return;
    if (this.fileFreePosition == 32L)
      return;
    this.database.logger.appLog.logContext(SimpleLog.LOG_NORMAL, "start");
    try
    {
      boolean bool = this.dataFile.wasNio();
      this.cache.saveAll();
      DataFileDefrag localDataFileDefrag = new DataFileDefrag(this.database, this, this.fileName);
      localDataFileDefrag.process();
      close(false);
      deleteFile(bool);
      renameDataFile();
      backupFile();
      this.database.getProperties().setProperty("hsqldb.cache_version", "1.7.0");
      this.database.getProperties().save();
      this.cache.clear();
      this.cache = new Cache(this);
      open(this.cacheReadonly);
      localDataFileDefrag.updateTableIndexRoots();
      localDataFileDefrag.updateTransactionRowIDs();
    }
    catch (Throwable localThrowable)
    {
      this.database.logger.appLog.logContext(localThrowable, null);
      throw new HsqlException(localThrowable, Trace.getMessage(98), 98);
    }
    this.database.logger.appLog.logContext(SimpleLog.LOG_NORMAL, "end");
  }

  public synchronized void remove(int paramInt, PersistentStore paramPersistentStore)
    throws IOException
  {
    CachedObject localCachedObject = release(paramInt);
    int i = localCachedObject == null ? getStorageSize(paramInt) : localCachedObject.getStorageSize();
    this.freeBlocks.add(paramInt, i);
  }

  public synchronized void removePersistence(int paramInt, PersistentStore paramPersistentStore)
    throws IOException
  {
  }

  private int setFilePos(CachedObject paramCachedObject)
    throws IOException
  {
    int i = paramCachedObject.getStorageSize();
    int j = this.freeBlocks == null ? -1 : this.freeBlocks.get(i);
    if (j == -1)
    {
      j = (int)(this.fileFreePosition / this.cacheFileScale);
      long l = this.fileFreePosition + i;
      if (l > this.maxDataFileSize)
        throw new IOException(Trace.getMessage(225));
      this.fileFreePosition = l;
    }
    paramCachedObject.setPos(j);
    return j;
  }

  public synchronized void add(CachedObject paramCachedObject)
    throws IOException
  {
    int i = paramCachedObject.getRealSize(this.rowOut);
    i = (i + this.cachedRowPadding - 1) / this.cachedRowPadding * this.cachedRowPadding;
    paramCachedObject.setStorageSize(i);
    int j = setFilePos(paramCachedObject);
    this.cache.put(j, paramCachedObject);
    if (this.storeOnInsert)
      saveRow(paramCachedObject);
  }

  public synchronized void restore(CachedObject paramCachedObject)
    throws IOException
  {
    int i = paramCachedObject.getPos();
    this.cache.put(i, paramCachedObject);
    if (this.storeOnInsert)
      saveRow(paramCachedObject);
  }

  public synchronized int getStorageSize(int paramInt)
    throws IOException
  {
    CachedObject localCachedObject = this.cache.get(paramInt);
    if (localCachedObject != null)
      return localCachedObject.getStorageSize();
    return readSize(paramInt);
  }

  public synchronized CachedObject get(int paramInt, PersistentStore paramPersistentStore, boolean paramBoolean)
    throws HsqlException
  {
    // Byte code:
    //   0: iload_1
    //   1: ifge +5 -> 6
    //   4: aconst_null
    //   5: areturn
    //   6: aload_0
    //   7: getfield 7	org/hsqldb/persist/DataFileCache:cache	Lorg/hsqldb/persist/Cache;
    //   10: iload_1
    //   11: invokevirtual 151	org/hsqldb/persist/Cache:get	(I)Lorg/hsqldb/persist/CachedObject;
    //   14: astore 4
    //   16: aload 4
    //   18: ifnonnull +45 -> 63
    //   21: aload_0
    //   22: iload_1
    //   23: invokevirtual 153	org/hsqldb/persist/DataFileCache:readObject	(I)Lorg/hsqldb/rowio/RowInputInterface;
    //   26: astore 5
    //   28: aload 5
    //   30: ifnonnull +5 -> 35
    //   33: aconst_null
    //   34: areturn
    //   35: aload_2
    //   36: aload 5
    //   38: invokeinterface 154 2 0
    //   43: astore 4
    //   45: aload 4
    //   47: invokeinterface 150 1 0
    //   52: istore_1
    //   53: aload_0
    //   54: getfield 7	org/hsqldb/persist/DataFileCache:cache	Lorg/hsqldb/persist/Cache;
    //   57: iload_1
    //   58: aload 4
    //   60: invokevirtual 147	org/hsqldb/persist/Cache:put	(ILorg/hsqldb/persist/CachedObject;)V
    //   63: iload_3
    //   64: ifeq +11 -> 75
    //   67: aload 4
    //   69: iconst_1
    //   70: invokeinterface 155 2 0
    //   75: aload 4
    //   77: areturn
    //   78: astore 4
    //   80: aload_0
    //   81: getfield 17	org/hsqldb/persist/DataFileCache:database	Lorg/hsqldb/Database;
    //   84: getfield 39	org/hsqldb/Database:logger	Lorg/hsqldb/persist/Logger;
    //   87: getfield 40	org/hsqldb/persist/Logger:appLog	Lorg/hsqldb/lib/SimpleLog;
    //   90: aload 4
    //   92: new 9	java/lang/StringBuffer
    //   95: dup
    //   96: invokespecial 10	java/lang/StringBuffer:<init>	()V
    //   99: aload_0
    //   100: getfield 14	org/hsqldb/persist/DataFileCache:fileName	Ljava/lang/String;
    //   103: invokevirtual 11	java/lang/StringBuffer:append	(Ljava/lang/String;)Ljava/lang/StringBuffer;
    //   106: ldc 156
    //   108: invokevirtual 11	java/lang/StringBuffer:append	(Ljava/lang/String;)Ljava/lang/StringBuffer;
    //   111: iload_1
    //   112: invokevirtual 157	java/lang/StringBuffer:append	(I)Ljava/lang/StringBuffer;
    //   115: invokevirtual 13	java/lang/StringBuffer:toString	()Ljava/lang/String;
    //   118: invokevirtual 85	org/hsqldb/lib/SimpleLog:logContext	(Ljava/lang/Throwable;Ljava/lang/String;)V
    //   121: sipush 129
    //   124: sipush 209
    //   127: iconst_2
    //   128: anewarray 87	java/lang/Object
    //   131: dup
    //   132: iconst_0
    //   133: aload 4
    //   135: aastore
    //   136: dup
    //   137: iconst_1
    //   138: aload_0
    //   139: getfield 14	org/hsqldb/persist/DataFileCache:fileName	Ljava/lang/String;
    //   142: aastore
    //   143: invokestatic 88	org/hsqldb/Trace:error	(II[Ljava/lang/Object;)Lorg/hsqldb/HsqlException;
    //   146: athrow
    //
    // Exception table:
    //   from	to	target	type
    //   6	34	78	java/io/IOException
    //   35	77	78	java/io/IOException
  }

  synchronized RowInputInterface getRaw(int paramInt)
    throws IOException
  {
    return readObject(paramInt);
  }

  protected synchronized int readSize(int paramInt)
    throws IOException
  {
    this.dataFile.seek(paramInt * this.cacheFileScale);
    return this.dataFile.readInt();
  }

  protected synchronized RowInputInterface readObject(int paramInt)
    throws IOException
  {
    this.dataFile.seek(paramInt * this.cacheFileScale);
    int i = this.dataFile.readInt();
    this.rowIn.resetRow(paramInt, i);
    this.dataFile.read(this.rowIn.getBuffer(), 4, i - 4);
    return this.rowIn;
  }

  public synchronized CachedObject release(int paramInt)
  {
    return this.cache.release(paramInt);
  }

  protected synchronized void saveRows(CachedObject[] paramArrayOfCachedObject, int paramInt1, int paramInt2)
    throws IOException
  {
    try
    {
      for (int i = paramInt1; i < paramInt1 + paramInt2; i++)
      {
        CachedObject localCachedObject = paramArrayOfCachedObject[i];
        saveRow(localCachedObject);
        paramArrayOfCachedObject[i] = null;
      }
    }
    catch (IOException localIOException)
    {
      this.database.logger.appLog.logContext(localIOException, null);
      throw localIOException;
    }
    catch (Throwable localThrowable)
    {
      this.database.logger.appLog.logContext(localThrowable, null);
      throw new IOException(localThrowable.toString());
    }
    finally
    {
      initBuffers();
    }
  }

  public synchronized void saveRow(CachedObject paramCachedObject)
    throws IOException
  {
    setFileModified();
    this.rowOut.reset();
    paramCachedObject.write(this.rowOut);
    this.dataFile.seek(paramCachedObject.getPos() * this.cacheFileScale);
    this.dataFile.write(this.rowOut.getOutputStream().getBuffer(), 0, this.rowOut.getOutputStream().size());
  }

  void backupFile()
    throws IOException
  {
    try
    {
      if (this.fa.isStreamElement(this.fileName))
        ZipUnzipFile.compressFile(this.fileName, this.backupFileName + ".new", this.database.getFileAccess());
    }
    catch (IOException localIOException)
    {
      this.database.logger.appLog.logContext(localIOException, null);
      throw localIOException;
    }
  }

  void renameBackupFile()
  {
    if (this.fa.isStreamElement(this.backupFileName + ".new"))
    {
      this.fa.removeElement(this.backupFileName);
      this.fa.renameElement(this.backupFileName + ".new", this.backupFileName);
    }
  }

  void renameDataFile()
  {
    if (this.fa.isStreamElement(this.fileName + ".new"))
    {
      this.fa.removeElement(this.fileName);
      this.fa.renameElement(this.fileName + ".new", this.fileName);
    }
  }

  void deleteFile(boolean paramBoolean)
  {
    this.fa.removeElement(this.fileName);
    if (this.fa.isStreamElement(this.fileName))
    {
      if (paramBoolean)
      {
        System.gc();
        this.fa.removeElement(this.fileName);
      }
      if (this.fa.isStreamElement(this.fileName))
      {
        this.fa.renameElement(this.fileName, this.fileName + ".old");
        File localFile = new File(this.fileName + ".old");
        FileUtil.deleteOnExit(localFile);
      }
    }
  }

  void deleteBackup()
  {
    this.fa.removeElement(this.backupFileName);
  }

  static void deleteOrResetFreePos(Database paramDatabase, String paramString)
  {
    ScaledRAFile localScaledRAFile = null;
    paramDatabase.getFileAccess().removeElement(paramString);
    if (paramDatabase.isStoredFileAccess())
      return;
    if (!paramDatabase.getFileAccess().isStreamElement(paramString))
      return;
    try
    {
      localScaledRAFile = new ScaledRAFile(paramDatabase, paramString, false);
      localScaledRAFile.seek(12L);
      localScaledRAFile.writeLong(32L);
    }
    catch (IOException localIOException1)
    {
      paramDatabase.logger.appLog.logContext(localIOException1, null);
    }
    finally
    {
      if (localScaledRAFile != null)
        try
        {
          localScaledRAFile.close();
        }
        catch (IOException localIOException2)
        {
          paramDatabase.logger.appLog.logContext(localIOException2, null);
        }
    }
  }

  public int capacity()
  {
    return this.maxCacheSize;
  }

  public long bytesCapacity()
  {
    return this.maxCacheBytes;
  }

  public long getTotalCachedBlockSize()
  {
    return this.cache.getTotalCachedBlockSize();
  }

  public int getFreeBlockCount()
  {
    return this.freeBlocks.size();
  }

  public int getTotalFreeBlockSize()
  {
    return 0;
  }

  public long getFileFreePos()
  {
    return this.fileFreePosition;
  }

  public int getCachedObjectCount()
  {
    return this.cache.size();
  }

  public String getFileName()
  {
    return this.fileName;
  }

  public boolean hasRowInfo()
  {
    return this.hasRowInfo;
  }

  public boolean isFileModified()
  {
    return this.fileModified;
  }

  protected synchronized void setFileModified()
    throws IOException
  {
    if (!this.fileModified)
    {
      this.dataFile.seek(28L);
      int i = BitMap.set(0, 2);
      if (this.hasRowInfo)
        i = BitMap.set(i, 3);
      this.dataFile.writeInt(i);
      this.fileModified = true;
    }
  }
}

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.hsqldb.persist.DataFileCache
 * JD-Core Version:    0.6.0
 */