package org.hsqldb;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.StringTokenizer;
import org.hsqldb.lib.ArrayUtil;
import org.hsqldb.lib.FileUtil;
import org.hsqldb.lib.HashSet;
import org.hsqldb.lib.Iterator;
import org.hsqldb.lib.StopWatch;
import org.hsqldb.lib.StringUtil;
import org.hsqldb.lib.WrapperIterator;
import org.hsqldb.lib.java.JavaSystem;
import org.hsqldb.persist.HsqlProperties;
import org.hsqldb.resources.BundleHandler;

public class Server
  implements HsqlSocketRequestHandler
{
  protected static final int serverBundleHandle = BundleHandler.getBundleHandle("org_hsqldb_Server_messages", null);
  HsqlProperties serverProperties;
  HashSet serverConnSet;
  private String[] dbAlias;
  private String[] dbType;
  private String[] dbPath;
  private HsqlProperties[] dbProps;
  private int[] dbID;
  private int maxConnections;
  protected String serverId;
  protected int serverProtocol;
  protected ThreadGroup serverConnectionThreadGroup;
  protected HsqlSocketFactory socketFactory;
  protected ServerSocket socket;
  private Thread serverThread;
  private Throwable serverError;
  private volatile int serverState;
  private volatile boolean isSilent;
  private volatile boolean isRemoteOpen;
  private PrintWriter logWriter;
  private PrintWriter errWriter;

  public Server()
  {
    this(1);
  }

  protected Server(int paramInt)
  {
    init(paramInt);
  }

  public static void main(String[] paramArrayOfString)
  {
    String str = FileUtil.canonicalOrAbsolutePath("server");
    HsqlProperties localHsqlProperties1 = ServerConfiguration.getPropertiesFromFile(str);
    HsqlProperties localHsqlProperties2 = localHsqlProperties1 == null ? new HsqlProperties() : localHsqlProperties1;
    HsqlProperties localHsqlProperties3 = null;
    try
    {
      localHsqlProperties3 = HsqlProperties.argArrayToProps(paramArrayOfString, "server");
    }
    catch (ArrayIndexOutOfBoundsException localArrayIndexOutOfBoundsException)
    {
      printHelp("server.help");
      return;
    }
    if (localHsqlProperties3 != null)
    {
      if (localHsqlProperties3.getErrorKeys().length != 0)
      {
        printHelp("server.help");
        return;
      }
      localHsqlProperties2.addProperties(localHsqlProperties3);
    }
    ServerConfiguration.translateDefaultDatabaseProperty(localHsqlProperties2);
    ServerConfiguration.translateDefaultNoSystemExitProperty(localHsqlProperties2);
    Server localServer = new Server();
    try
    {
      localServer.setProperties(localHsqlProperties2);
    }
    catch (Exception localException)
    {
      localServer.printError("Failed to set properties");
      localServer.printStackTrace(localException);
      return;
    }
    localServer.print("Startup sequence initiated from main() method");
    if (localHsqlProperties1 != null)
    {
      localServer.print("Loaded properties from [" + str + ".properties]");
    }
    else
    {
      localServer.print("Could not load properties from file");
      localServer.print("Using cli/default properties only");
    }
    localServer.start();
  }

  public void checkRunning(boolean paramBoolean)
    throws RuntimeException
  {
    printWithThread("checkRunning(" + paramBoolean + ") entered");
    int i = getState();
    int j = ((paramBoolean) && (i != 1)) || ((!paramBoolean) && (i != 16)) ? 1 : 0;
    if (j != 0)
    {
      String str = "server is " + (paramBoolean ? "not " : "") + "running";
      throw new RuntimeException(str);
    }
    printWithThread("checkRunning(" + paramBoolean + ") exited");
  }

  public synchronized void signalCloseAllServerConnections()
  {
    printWithThread("signalCloseAllServerConnections() entered");
    WrapperIterator localWrapperIterator;
    synchronized (this.serverConnSet)
    {
      localWrapperIterator = new WrapperIterator(this.serverConnSet.toArray(null));
    }
    while (localWrapperIterator.hasNext())
    {
      ??? = (ServerConnection)localWrapperIterator.next();
      printWithThread("Closing " + ???);
      ((ServerConnection)???).signalClose();
    }
    printWithThread("signalCloseAllServerConnections() exited");
  }

  protected void finalize()
    throws Throwable
  {
    if (this.serverThread != null)
      releaseServerSocket();
  }

  public String getAddress()
  {
    return this.socket == null ? this.serverProperties.getProperty("server.address") : this.socket.getInetAddress().getHostAddress();
  }

  public String getDatabaseName(int paramInt, boolean paramBoolean)
  {
    if (paramBoolean)
      return this.serverProperties.getProperty("server.dbname." + paramInt);
    if (getState() == 1)
      return (this.dbAlias == null) || (paramInt < 0) || (paramInt >= this.dbAlias.length) ? null : this.dbAlias[paramInt];
    return null;
  }

  public String getDatabasePath(int paramInt, boolean paramBoolean)
  {
    if (paramBoolean)
      return this.serverProperties.getProperty("server.database." + paramInt);
    if (getState() == 1)
      return (this.dbPath == null) || (paramInt < 0) || (paramInt >= this.dbPath.length) ? null : this.dbPath[paramInt];
    return null;
  }

  public String getDatabaseType(int paramInt)
  {
    return (this.dbType == null) || (paramInt < 0) || (paramInt >= this.dbType.length) ? null : this.dbType[paramInt];
  }

  public String getDefaultWebPage()
  {
    return "[IGNORED]";
  }

  public String getHelpString()
  {
    return BundleHandler.getString(serverBundleHandle, "server.help");
  }

  public PrintWriter getErrWriter()
  {
    return this.errWriter;
  }

  public PrintWriter getLogWriter()
  {
    return this.logWriter;
  }

  public int getPort()
  {
    return this.serverProperties.getIntegerProperty("server.port", ServerConfiguration.getDefaultPort(this.serverProtocol, isTls()));
  }

  public String getProductName()
  {
    return "HSQLDB server";
  }

  public String getProductVersion()
  {
    return "1.8.0";
  }

  public String getProtocol()
  {
    return isTls() ? "HSQLS" : "HSQL";
  }

  public Throwable getServerError()
  {
    return this.serverError;
  }

  public String getServerId()
  {
    return this.serverId;
  }

  public synchronized int getState()
  {
    return this.serverState;
  }

  public String getStateDescriptor()
  {
    Throwable localThrowable = getServerError();
    String str;
    switch (this.serverState)
    {
    case 16:
      str = "SHUTDOWN";
      break;
    case 4:
      str = "OPENING";
      break;
    case 8:
      str = "CLOSING";
      break;
    case 1:
      str = "ONLINE";
      break;
    default:
      str = "UNKNOWN";
    }
    return str;
  }

  public String getWebRoot()
  {
    return "[IGNORED]";
  }

  public void handleConnection(Socket paramSocket)
  {
    printWithThread("handleConnection(" + paramSocket + ") entered");
    if (!allowConnection(paramSocket))
    {
      try
      {
        paramSocket.close();
      }
      catch (Exception localException)
      {
      }
      printWithThread("allowConnection(): connection refused");
      printWithThread("handleConnection() exited");
      return;
    }
    if (this.socketFactory != null)
      this.socketFactory.configureSocket(paramSocket);
    if (this.serverProtocol == 1)
    {
      localObject1 = new ServerConnection(paramSocket, this);
      str = ((ServerConnection)localObject1).getConnectionThreadName();
      synchronized (this.serverConnSet)
      {
        this.serverConnSet.add(localObject1);
      }
    }
    Object localObject1 = new WebServerConnection(paramSocket, (WebServer)this);
    String str = ((WebServerConnection)localObject1).getConnectionThreadName();
    Thread localThread = new Thread(this.serverConnectionThreadGroup, (Runnable)localObject1, str);
    localThread.start();
    printWithThread("handleConnection() exited");
  }

  public boolean isNoSystemExit()
  {
    return this.serverProperties.isPropertyTrue("server.no_system_exit");
  }

  public boolean isRestartOnShutdown()
  {
    return this.serverProperties.isPropertyTrue("server.restart_on_shutdown");
  }

  public boolean isSilent()
  {
    return this.isSilent;
  }

  public boolean isTls()
  {
    return this.serverProperties.isPropertyTrue("server.tls");
  }

  public boolean isTrace()
  {
    return this.serverProperties.isPropertyTrue("server.trace");
  }

  public boolean putPropertiesFromFile(String paramString)
  {
    if (getState() != 16)
      throw new RuntimeException();
    paramString = FileUtil.canonicalOrAbsolutePath(paramString);
    HsqlProperties localHsqlProperties = ServerConfiguration.getPropertiesFromFile(paramString);
    if ((localHsqlProperties == null) || (localHsqlProperties.isEmpty()))
      return false;
    printWithThread("putPropertiesFromFile(): [" + paramString + ".properties]");
    try
    {
      setProperties(localHsqlProperties);
    }
    catch (Exception localException)
    {
      throw new RuntimeException("Failed to set properties: " + localException);
    }
    return true;
  }

  public void putPropertiesFromString(String paramString)
  {
    if (getState() != 16)
      throw new RuntimeException();
    if (StringUtil.isEmpty(paramString))
      return;
    printWithThread("putPropertiesFromString(): [" + paramString + "]");
    HsqlProperties localHsqlProperties = HsqlProperties.delimitedArgPairsToProps(paramString, "=", ";", "server");
    try
    {
      setProperties(localHsqlProperties);
    }
    catch (Exception localException)
    {
      throw new RuntimeException("Failed to set properties: " + localException);
    }
  }

  public void setAddress(String paramString)
    throws RuntimeException
  {
    checkRunning(false);
    if (StringUtil.isEmpty(paramString))
      paramString = "0.0.0.0";
    printWithThread("setAddress(" + paramString + ")");
    this.serverProperties.setProperty("server.address", paramString);
  }

  public void setDatabaseName(int paramInt, String paramString)
    throws RuntimeException
  {
    checkRunning(false);
    printWithThread("setDatabaseName(" + paramInt + "," + paramString + ")");
    this.serverProperties.setProperty("server.dbname." + paramInt, paramString);
  }

  public void setDatabasePath(int paramInt, String paramString)
    throws RuntimeException
  {
    checkRunning(false);
    printWithThread("setDatabasePath(" + paramInt + "," + paramString + ")");
    this.serverProperties.setProperty("server.database." + paramInt, paramString);
  }

  public void setDefaultWebPage(String paramString)
  {
    checkRunning(false);
    printWithThread("setDefaultWebPage(" + paramString + ")");
    if (this.serverProtocol != 0)
      return;
    this.serverProperties.setProperty("server.default_page", paramString);
  }

  public void setPort(int paramInt)
    throws RuntimeException
  {
    checkRunning(false);
    printWithThread("setPort(" + paramInt + ")");
    this.serverProperties.setProperty("server.port", paramInt);
  }

  public void setErrWriter(PrintWriter paramPrintWriter)
  {
    this.errWriter = paramPrintWriter;
  }

  public void setLogWriter(PrintWriter paramPrintWriter)
  {
    this.logWriter = paramPrintWriter;
  }

  public void setNoSystemExit(boolean paramBoolean)
  {
    printWithThread("setNoSystemExit(" + paramBoolean + ")");
    this.serverProperties.setProperty("server.no_system_exit", paramBoolean);
  }

  public void setRestartOnShutdown(boolean paramBoolean)
  {
    printWithThread("setRestartOnShutdown(" + paramBoolean + ")");
    this.serverProperties.setProperty("server.restart_on_shutdown", paramBoolean);
  }

  public void setSilent(boolean paramBoolean)
  {
    printWithThread("setSilent(" + paramBoolean + ")");
    this.serverProperties.setProperty("server.silent", paramBoolean);
    this.isSilent = paramBoolean;
  }

  public void setTls(boolean paramBoolean)
  {
    checkRunning(false);
    printWithThread("setTls(" + paramBoolean + ")");
    this.serverProperties.setProperty("server.tls", paramBoolean);
  }

  public void setTrace(boolean paramBoolean)
  {
    printWithThread("setTrace(" + paramBoolean + ")");
    this.serverProperties.setProperty("server.trace", paramBoolean);
    JavaSystem.setLogToSystem(paramBoolean);
  }

  public void setWebRoot(String paramString)
  {
    checkRunning(false);
    paramString = new File(paramString).getAbsolutePath();
    printWithThread("setWebRoot(" + paramString + ")");
    if (this.serverProtocol != 0)
      return;
    this.serverProperties.setProperty("server.root", paramString);
  }

  public void setProperties(HsqlProperties paramHsqlProperties)
  {
    checkRunning(false);
    if (paramHsqlProperties != null)
    {
      this.serverProperties.addProperties(paramHsqlProperties);
      ServerConfiguration.translateAddressProperty(this.serverProperties);
    }
    this.maxConnections = this.serverProperties.getIntegerProperty("server.maxconnections", 16);
    JavaSystem.setLogToSystem(isTrace());
    this.isSilent = this.serverProperties.isPropertyTrue("server.silent");
    this.isRemoteOpen = this.serverProperties.isPropertyTrue("server.remote_open");
  }

  public int start()
  {
    printWithThread("start() entered");
    int i = getState();
    if (this.serverThread != null)
    {
      printWithThread("start(): serverThread != null; no action taken");
      return i;
    }
    setState(4);
    this.serverThread = new ServerThread("HSQLDB Server ");
    this.serverThread.start();
    while (getState() == 4)
      try
      {
        Thread.sleep(100L);
      }
      catch (InterruptedException localInterruptedException)
      {
      }
    printWithThread("start() exiting");
    return i;
  }

  public int stop()
  {
    printWithThread("stop() entered");
    int i = getState();
    if (this.serverThread == null)
    {
      printWithThread("stop() serverThread is null; no action taken");
      return i;
    }
    releaseServerSocket();
    printWithThread("stop() exiting");
    return i;
  }

  protected boolean allowConnection(Socket paramSocket)
  {
    return true;
  }

  protected void init(int paramInt)
  {
    this.serverState = 16;
    this.serverConnSet = new HashSet();
    this.serverId = toString();
    this.serverId = this.serverId.substring(this.serverId.lastIndexOf('.') + 1);
    this.serverProtocol = paramInt;
    this.serverProperties = ServerConfiguration.newDefaultProperties(paramInt);
    this.logWriter = new PrintWriter(System.out);
    this.errWriter = new PrintWriter(System.err);
    JavaSystem.setLogToSystem(isTrace());
  }

  protected synchronized void setState(int paramInt)
  {
    this.serverState = paramInt;
  }

  final void notify(int paramInt1, int paramInt2)
  {
    printWithThread("notifiy(" + paramInt1 + "," + paramInt2 + ") entered");
    if (paramInt1 != 0)
      return;
    releaseDatabase(paramInt2);
    int i = 1;
    for (int j = 0; j < this.dbID.length; j++)
    {
      if (this.dbAlias[j] == null)
        continue;
      i = 0;
    }
    if ((!this.isRemoteOpen) && (i != 0))
      stop();
  }

  final synchronized void releaseDatabase(int paramInt)
  {
    int i = 0;
    printWithThread("releaseDatabase(" + paramInt + ") entered");
    for (int j = 0; j < this.dbID.length; j++)
    {
      if ((this.dbID[j] != paramInt) || (this.dbAlias[j] == null))
        continue;
      this.dbID[j] = 0;
      this.dbAlias[j] = null;
      this.dbPath[j] = null;
      this.dbType[j] = null;
      this.dbProps[j] = null;
    }
    WrapperIterator localWrapperIterator;
    synchronized (this.serverConnSet)
    {
      localWrapperIterator = new WrapperIterator(this.serverConnSet.toArray(null));
    }
    while (localWrapperIterator.hasNext())
    {
      ??? = (ServerConnection)localWrapperIterator.next();
      if (((ServerConnection)???).dbID != paramInt)
        continue;
      ((ServerConnection)???).signalClose();
      this.serverConnSet.remove(???);
    }
    printWithThread("releaseDatabase(" + paramInt + ") exiting");
  }

  protected synchronized void print(String paramString)
  {
    PrintWriter localPrintWriter = this.logWriter;
    if (localPrintWriter != null)
    {
      localPrintWriter.println("[" + this.serverId + "]: " + paramString);
      localPrintWriter.flush();
    }
  }

  final void printResource(String paramString)
  {
    if (serverBundleHandle < 0)
      return;
    String str = BundleHandler.getString(serverBundleHandle, paramString);
    if (str == null)
      return;
    StringTokenizer localStringTokenizer = new StringTokenizer(str, "\n\r");
    while (localStringTokenizer.hasMoreTokens())
      print(localStringTokenizer.nextToken());
  }

  protected synchronized void printStackTrace(Throwable paramThrowable)
  {
    if (this.errWriter != null)
    {
      paramThrowable.printStackTrace(this.errWriter);
      this.errWriter.flush();
    }
  }

  final void printWithTimestamp(String paramString)
  {
    print(HsqlDateTime.getSytemTimeString() + " " + paramString);
  }

  protected void printWithThread(String paramString)
  {
    if (!isSilent())
      print("[" + Thread.currentThread() + "]: " + paramString);
  }

  protected synchronized void printError(String paramString)
  {
    PrintWriter localPrintWriter = this.errWriter;
    if (localPrintWriter != null)
    {
      localPrintWriter.print("[" + this.serverId + "]: ");
      localPrintWriter.print("[" + Thread.currentThread() + "]: ");
      localPrintWriter.println(paramString);
      localPrintWriter.flush();
    }
  }

  final void printRequest(int paramInt, Result paramResult)
  {
    if (isSilent())
      return;
    StringBuffer localStringBuffer = new StringBuffer();
    localStringBuffer.append(paramInt);
    localStringBuffer.append(':');
    Iterator localIterator;
    switch (paramResult.mode)
    {
    case 65555:
      localStringBuffer.append("SQLCLI:SQLPREPARE ");
      localStringBuffer.append(paramResult.getMainString());
      break;
    case 65547:
      if (paramResult.getSize() < 2)
      {
        localStringBuffer.append(paramResult.getMainString());
      }
      else
      {
        localStringBuffer.append("SQLCLI:SQLEXECDIRECT:BATCHMODE\n");
        localIterator = paramResult.iterator();
      }
    case 65548:
    case 65552:
    case 7:
    case 6:
    case 66541:
    case 65610:
    case 65545:
    case 66552:
    default:
      while (localIterator.hasNext())
      {
        Object[] arrayOfObject = (Object[])localIterator.next();
        localStringBuffer.append(arrayOfObject[0]).append('\n');
        continue;
        localStringBuffer.append("SQLCLI:SQLEXECUTE:");
        if (paramResult.getSize() > 1)
          localStringBuffer.append("BATCHMODE:");
        localStringBuffer.append(paramResult.getStatementID());
        break;
        localStringBuffer.append("SQLCLI:SQLFREESTMT:");
        localStringBuffer.append(paramResult.getStatementID());
        break;
        localStringBuffer.append("HSQLCLI:GETSESSIONATTR");
        break;
        localStringBuffer.append("HSQLCLI:SETSESSIONATTR:");
        localStringBuffer.append("AUTOCOMMIT ");
        localStringBuffer.append(paramResult.rRoot.data[4]);
        localStringBuffer.append(" CONNECTION_READONLY ");
        localStringBuffer.append(paramResult.rRoot.data[6]);
        break;
        localStringBuffer.append("SQLCLI:SQLENDTRAN:");
        switch (paramResult.getEndTranType())
        {
        case 0:
          localStringBuffer.append("COMMIT");
          break;
        case 1:
          localStringBuffer.append("ROLLBACK");
          break;
        case 4:
          localStringBuffer.append("SAVEPOINT_NAME_RELEASE ");
          localStringBuffer.append(paramResult.getMainString());
          break;
        case 2:
          localStringBuffer.append("SAVEPOINT_NAME_ROLLBACK ");
          localStringBuffer.append(paramResult.getMainString());
          break;
        case 3:
        default:
          localStringBuffer.append(paramResult.getEndTranType());
          break;
          localStringBuffer.append("SQLCLI:SQLSTARTTRAN");
          break;
          localStringBuffer.append("SQLCLI:SQLDISCONNECT");
          break;
          localStringBuffer.append("SQLCLI:SQLSETCONNECTATTR:");
          switch (paramResult.getConnectionAttrType())
          {
          case 10027:
            localStringBuffer.append("SQL_ATTR_SAVEPOINT_NAME ");
            localStringBuffer.append(paramResult.getMainString());
            break;
          default:
            localStringBuffer.append(paramResult.getConnectionAttrType());
            break;
            localStringBuffer.append("SQLCLI:MODE:");
            localStringBuffer.append(paramResult.mode);
          }
        }
      }
    }
    print(localStringBuffer.toString());
  }

  final synchronized int getDBID(String paramString)
    throws HsqlException
  {
    int i = paramString.indexOf(';');
    String str1 = paramString;
    String str2 = null;
    if (i != -1)
    {
      str1 = paramString.substring(0, i);
      str2 = paramString.substring(i + 1);
    }
    int j = ArrayUtil.find(this.dbAlias, str1);
    if (j == -1)
    {
      if (str2 == null)
      {
        RuntimeException localRuntimeException = new RuntimeException("database alias does not exist");
        printError("database alias=" + str1 + " does not exist");
        setServerError(localRuntimeException);
        throw localRuntimeException;
      }
      return openDatabase(str1, str2);
    }
    return this.dbID[j];
  }

  final int openDatabase(String paramString1, String paramString2)
    throws HsqlException
  {
    if (!this.isRemoteOpen)
    {
      RuntimeException localRuntimeException = new RuntimeException("remote open not allowed");
      printError("Remote database open not allowed");
      setServerError(localRuntimeException);
      throw localRuntimeException;
    }
    int i = getFirstEmptyDatabaseIndex();
    if (i < -1)
    {
      localObject1 = new RuntimeException("limit of open databases reached");
      printError("limit of open databases reached");
      setServerError((Throwable)localObject1);
      throw ((Throwable)localObject1);
    }
    Object localObject1 = DatabaseURL.parseURL(paramString2, false);
    if (localObject1 == null)
    {
      localObject2 = new RuntimeException("invalid database path");
      printError("invalid database path");
      setServerError((Throwable)localObject2);
      throw ((Throwable)localObject2);
    }
    Object localObject2 = ((HsqlProperties)localObject1).getProperty("database");
    String str = ((HsqlProperties)localObject1).getProperty("connection_type");
    try
    {
      int j = DatabaseManager.getDatabase(str, (String)localObject2, this, (HsqlProperties)localObject1);
      this.dbID[i] = j;
      this.dbAlias[i] = paramString1;
      this.dbPath[i] = localObject2;
      this.dbType[i] = str;
      this.dbProps[i] = localObject1;
      return j;
    }
    catch (HsqlException localHsqlException)
    {
      printError("Database [index=" + i + "db=" + this.dbType[i] + this.dbPath[i] + ", alias=" + this.dbAlias[i] + "] did not open: " + localHsqlException.toString());
      setServerError(localHsqlException);
    }
    throw localHsqlException;
  }

  final int getFirstEmptyDatabaseIndex()
  {
    for (int i = 0; i < this.dbAlias.length; i++)
      if (this.dbAlias[i] == null)
        return i;
    return -1;
  }

  final boolean openDatabases()
  {
    printWithThread("openDatabases() entered");
    int i = 0;
    setDBInfoArrays();
    for (int j = 0; j < this.dbAlias.length; j++)
    {
      if (this.dbAlias[j] == null)
        continue;
      printWithThread("Opening database: [" + this.dbType[j] + this.dbPath[j] + "]");
      StopWatch localStopWatch = new StopWatch();
      int k;
      try
      {
        k = DatabaseManager.getDatabase(this.dbType[j], this.dbPath[j], this, this.dbProps[j]);
        this.dbID[j] = k;
        i = 1;
      }
      catch (HsqlException localHsqlException)
      {
        printError("Database [index=" + j + "db=" + this.dbType[j] + this.dbPath[j] + ", alias=" + this.dbAlias[j] + "] did not open: " + localHsqlException.toString());
        setServerError(localHsqlException);
        this.dbAlias[j] = null;
        this.dbPath[j] = null;
        this.dbType[j] = null;
        this.dbProps[j] = null;
        continue;
      }
      localStopWatch.stop();
      String str = "Database [index=" + j + ", id=" + k + ", " + "db=" + this.dbType[j] + this.dbPath[j] + ", alias=" + this.dbAlias[j] + "] opened sucessfully";
      print(localStopWatch.elapsedTimeToMessage(str));
    }
    printWithThread("openDatabases() exiting");
    if (this.isRemoteOpen)
      i = 1;
    if ((i == 0) && (getServerError() == null))
      setServerError(Trace.error(147));
    return i;
  }

  private void setDBInfoArrays()
  {
    this.dbAlias = getDBNameArray();
    this.dbPath = new String[this.dbAlias.length];
    this.dbType = new String[this.dbAlias.length];
    this.dbID = new int[this.dbAlias.length];
    this.dbProps = new HsqlProperties[this.dbAlias.length];
    for (int i = 0; i < this.dbAlias.length; i++)
    {
      if (this.dbAlias[i] == null)
        continue;
      String str = getDatabasePath(i, true);
      if (str == null)
      {
        this.dbAlias[i] = null;
      }
      else
      {
        HsqlProperties localHsqlProperties = DatabaseURL.parseURL(str, false);
        if (localHsqlProperties == null)
        {
          this.dbAlias[i] = null;
        }
        else
        {
          this.dbPath[i] = localHsqlProperties.getProperty("database");
          this.dbType[i] = localHsqlProperties.getProperty("connection_type");
          this.dbProps[i] = localHsqlProperties;
        }
      }
    }
  }

  private String[] getDBNameArray()
  {
    int i = "server.dbname.".length();
    String[] arrayOfString = new String[10];
    int j = 0;
    try
    {
      Enumeration localEnumeration = this.serverProperties.propertyNames();
      while (localEnumeration.hasMoreElements())
      {
        String str = (String)localEnumeration.nextElement();
        if (!str.startsWith("server.dbname."))
          continue;
        try
        {
          int k = Integer.parseInt(str.substring(i));
          j = k < j ? j : k;
          arrayOfString[k] = this.serverProperties.getProperty(str).toLowerCase();
        }
        catch (NumberFormatException localNumberFormatException)
        {
          printWithThread("dblist: " + localNumberFormatException.toString());
        }
      }
    }
    catch (ArrayIndexOutOfBoundsException localArrayIndexOutOfBoundsException)
    {
      printWithThread("dblist: " + localArrayIndexOutOfBoundsException.toString());
    }
    return arrayOfString;
  }

  private void openServerSocket()
    throws Exception
  {
    printWithThread("openServerSocket() entered");
    if (isTls())
      printWithThread("Requesting TLS/SSL-encrypted JDBC");
    StopWatch localStopWatch = new StopWatch();
    this.socketFactory = HsqlSocketFactory.getInstance(isTls());
    String str = getAddress();
    int i = getPort();
    if ((StringUtil.isEmpty(str)) || ("0.0.0.0".equalsIgnoreCase(str.trim())))
      this.socket = this.socketFactory.createServerSocket(i);
    else
      try
      {
        this.socket = this.socketFactory.createServerSocket(i, str);
      }
      catch (UnknownHostException localUnknownHostException)
      {
        String[] arrayOfString = ServerConfiguration.listLocalInetAddressNames();
        int j;
        Object[] arrayOfObject;
        if (arrayOfString.length > 0)
        {
          j = 148;
          arrayOfObject = new Object[] { str, arrayOfString };
        }
        else
        {
          j = 149;
          arrayOfObject = new Object[] { str };
        }
        throw new UnknownHostException(Trace.getMessage(j, true, arrayOfObject));
      }
    this.socket.setSoTimeout(1000);
    printWithThread("Got server socket: " + this.socket);
    print(localStopWatch.elapsedTimeToMessage("Server socket opened successfully"));
    if (this.socketFactory.isSecure())
      print("Using TLS/SSL-encrypted JDBC");
    printWithThread("openServerSocket() exiting");
  }

  private void printServerOnlineMessage()
  {
    String str = getProductName() + " " + getProductVersion() + " is online";
    printWithTimestamp(str);
    printResource("online.help");
  }

  protected void printProperties()
  {
    if (isSilent())
      return;
    Enumeration localEnumeration = this.serverProperties.propertyNames();
    while (localEnumeration.hasMoreElements())
    {
      String str1 = (String)localEnumeration.nextElement();
      String str2 = this.serverProperties.getProperty(str1);
      printWithThread(str1 + "=" + str2);
    }
  }

  private void releaseServerSocket()
  {
    printWithThread("releaseServerSocket() entered");
    if (this.socket != null)
    {
      printWithThread("Releasing server socket: [" + this.socket + "]");
      setState(8);
      try
      {
        this.socket.close();
      }
      catch (IOException localIOException)
      {
        printError("Exception closing server socket");
        printError("releaseServerSocket(): " + localIOException);
      }
      this.socket = null;
    }
    printWithThread("releaseServerSocket() exited");
  }

  // ERROR //
  private void run()
  {
    // Byte code:
    //   0: aload_0
    //   1: ldc_w 322
    //   4: invokevirtual 39	org/hsqldb/Server:printWithThread	(Ljava/lang/String;)V
    //   7: aload_0
    //   8: ldc_w 323
    //   11: invokevirtual 26	org/hsqldb/Server:print	(Ljava/lang/String;)V
    //   14: aload_0
    //   15: invokevirtual 324	org/hsqldb/Server:printProperties	()V
    //   18: new 266	org/hsqldb/lib/StopWatch
    //   21: dup
    //   22: invokespecial 267	org/hsqldb/lib/StopWatch:<init>	()V
    //   25: astore_1
    //   26: aload_0
    //   27: aconst_null
    //   28: invokevirtual 246	org/hsqldb/Server:setServerError	(Ljava/lang/Throwable;)V
    //   31: aload_0
    //   32: invokespecial 325	org/hsqldb/Server:openServerSocket	()V
    //   35: goto +30 -> 65
    //   38: astore 4
    //   40: aload_0
    //   41: aload 4
    //   43: invokevirtual 246	org/hsqldb/Server:setServerError	(Ljava/lang/Throwable;)V
    //   46: aload_0
    //   47: ldc_w 326
    //   50: invokevirtual 23	org/hsqldb/Server:printError	(Ljava/lang/String;)V
    //   53: aload_0
    //   54: aload 4
    //   56: invokevirtual 24	org/hsqldb/Server:printStackTrace	(Ljava/lang/Throwable;)V
    //   59: aload_0
    //   60: iconst_1
    //   61: invokevirtual 327	org/hsqldb/Server:shutdown	(Z)V
    //   64: return
    //   65: new 27	java/lang/StringBuffer
    //   68: dup
    //   69: invokespecial 28	java/lang/StringBuffer:<init>	()V
    //   72: ldc_w 328
    //   75: invokevirtual 30	java/lang/StringBuffer:append	(Ljava/lang/String;)Ljava/lang/StringBuffer;
    //   78: aload_0
    //   79: invokevirtual 329	java/lang/Object:hashCode	()I
    //   82: bipush 16
    //   84: invokestatic 330	java/lang/Integer:toString	(II)Ljava/lang/String;
    //   87: invokevirtual 30	java/lang/StringBuffer:append	(Ljava/lang/String;)Ljava/lang/StringBuffer;
    //   90: invokevirtual 32	java/lang/StringBuffer:toString	()Ljava/lang/String;
    //   93: astore_3
    //   94: new 331	java/lang/ThreadGroup
    //   97: dup
    //   98: aload_3
    //   99: invokespecial 332	java/lang/ThreadGroup:<init>	(Ljava/lang/String;)V
    //   102: astore_2
    //   103: aload_2
    //   104: iconst_0
    //   105: invokevirtual 333	java/lang/ThreadGroup:setDaemon	(Z)V
    //   108: aload_0
    //   109: aload_2
    //   110: putfield 112	org/hsqldb/Server:serverConnectionThreadGroup	Ljava/lang/ThreadGroup;
    //   113: aload_0
    //   114: invokevirtual 334	org/hsqldb/Server:openDatabases	()Z
    //   117: ifne +21 -> 138
    //   120: aload_0
    //   121: aconst_null
    //   122: invokevirtual 246	org/hsqldb/Server:setServerError	(Ljava/lang/Throwable;)V
    //   125: aload_0
    //   126: ldc_w 335
    //   129: invokevirtual 23	org/hsqldb/Server:printError	(Ljava/lang/String;)V
    //   132: aload_0
    //   133: iconst_1
    //   134: invokevirtual 327	org/hsqldb/Server:shutdown	(Z)V
    //   137: return
    //   138: aload_0
    //   139: iconst_1
    //   140: invokevirtual 164	org/hsqldb/Server:setState	(I)V
    //   143: aload_0
    //   144: aload_1
    //   145: ldc_w 336
    //   148: invokevirtual 272	org/hsqldb/lib/StopWatch:elapsedTimeToMessage	(Ljava/lang/String;)Ljava/lang/String;
    //   151: invokevirtual 26	org/hsqldb/Server:print	(Ljava/lang/String;)V
    //   154: aload_0
    //   155: invokespecial 337	org/hsqldb/Server:printServerOnlineMessage	()V
    //   158: aload_0
    //   159: aload_0
    //   160: getfield 62	org/hsqldb/Server:socket	Ljava/net/ServerSocket;
    //   163: invokevirtual 338	java/net/ServerSocket:accept	()Ljava/net/Socket;
    //   166: invokevirtual 339	org/hsqldb/Server:handleConnection	(Ljava/net/Socket;)V
    //   169: goto -11 -> 158
    //   172: astore 4
    //   174: goto -16 -> 158
    //   177: astore 4
    //   179: aload_0
    //   180: invokevirtual 40	org/hsqldb/Server:getState	()I
    //   183: iconst_1
    //   184: if_icmpne +39 -> 223
    //   187: aload_0
    //   188: aload 4
    //   190: invokevirtual 246	org/hsqldb/Server:setServerError	(Ljava/lang/Throwable;)V
    //   193: aload_0
    //   194: new 27	java/lang/StringBuffer
    //   197: dup
    //   198: invokespecial 28	java/lang/StringBuffer:<init>	()V
    //   201: aload_0
    //   202: invokevirtual 57	java/lang/StringBuffer:append	(Ljava/lang/Object;)Ljava/lang/StringBuffer;
    //   205: ldc_w 341
    //   208: invokevirtual 30	java/lang/StringBuffer:append	(Ljava/lang/String;)Ljava/lang/StringBuffer;
    //   211: invokevirtual 32	java/lang/StringBuffer:toString	()Ljava/lang/String;
    //   214: invokevirtual 23	org/hsqldb/Server:printError	(Ljava/lang/String;)V
    //   217: aload_0
    //   218: aload 4
    //   220: invokevirtual 24	org/hsqldb/Server:printStackTrace	(Ljava/lang/Throwable;)V
    //   223: aload_0
    //   224: iconst_0
    //   225: invokevirtual 327	org/hsqldb/Server:shutdown	(Z)V
    //   228: goto +32 -> 260
    //   231: astore 4
    //   233: aload_0
    //   234: aload 4
    //   236: invokevirtual 343	java/lang/Throwable:toString	()Ljava/lang/String;
    //   239: invokevirtual 39	org/hsqldb/Server:printWithThread	(Ljava/lang/String;)V
    //   242: aload_0
    //   243: iconst_0
    //   244: invokevirtual 327	org/hsqldb/Server:shutdown	(Z)V
    //   247: goto +13 -> 260
    //   250: astore 5
    //   252: aload_0
    //   253: iconst_0
    //   254: invokevirtual 327	org/hsqldb/Server:shutdown	(Z)V
    //   257: aload 5
    //   259: athrow
    //   260: return
    //
    // Exception table:
    //   from	to	target	type
    //   31	35	38	java/lang/Exception
    //   158	169	172	java/io/InterruptedIOException
    //   158	177	177	java/io/IOException
    //   158	177	231	java/lang/Throwable
    //   158	223	250	finally
    //   231	242	250	finally
    //   250	252	250	finally
  }

  protected void setServerError(Throwable paramThrowable)
  {
    this.serverError = paramThrowable;
  }

  public void shutdown()
  {
    shutdown(false);
  }

  protected synchronized void shutdown(boolean paramBoolean)
  {
    if (this.serverState == 16)
      return;
    printWithThread("shutdown() entered");
    StopWatch localStopWatch = new StopWatch();
    print("Initiating shutdown sequence...");
    releaseServerSocket();
    DatabaseManager.deRegisterServer(this);
    int i;
    if (this.dbPath != null)
      for (i = 0; i < this.dbPath.length; i++)
        releaseDatabase(this.dbID[i]);
    if (this.serverConnectionThreadGroup != null)
    {
      if (!this.serverConnectionThreadGroup.isDestroyed())
      {
        for (i = 0; this.serverConnectionThreadGroup.activeCount() > 0; i++)
          try
          {
            Thread.sleep(100L);
          }
          catch (Exception localException)
          {
          }
        try
        {
          this.serverConnectionThreadGroup.destroy();
          printWithThread(this.serverConnectionThreadGroup.getName() + " destroyed");
        }
        catch (Throwable localThrowable1)
        {
          printWithThread(this.serverConnectionThreadGroup.getName() + " not destroyed");
          printWithThread(localThrowable1.toString());
        }
      }
      this.serverConnectionThreadGroup = null;
    }
    this.serverThread = null;
    setState(16);
    print(localStopWatch.elapsedTimeToMessage("Shutdown sequence completed"));
    if (isNoSystemExit())
    {
      printWithTimestamp("SHUTDOWN : System.exit() was not called");
      printWithThread("shutdown() exited");
    }
    else
    {
      printWithTimestamp("SHUTDOWN : System.exit() is called next");
      printWithThread("shutdown() exiting...");
      try
      {
        System.exit(0);
      }
      catch (Throwable localThrowable2)
      {
        printWithThread(localThrowable2.toString());
      }
    }
  }

  protected static void printHelp(String paramString)
  {
    System.out.println(BundleHandler.getString(serverBundleHandle, paramString));
  }

  private class ServerThread extends Thread
  {
    ServerThread(String arg2)
    {
      super();
      setName(str + '@' + Integer.toString(Server.this.hashCode(), 16));
    }

    public void run()
    {
      Server.this.run();
      Server.this.printWithThread("ServerThread.run() exited");
    }
  }
}

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.hsqldb.Server
 * JD-Core Version:    0.6.0
 */