/*      */ package org.apache.commons.logging.impl;
/*      */ 
/*      */ import java.lang.reflect.Constructor;
/*      */ import java.lang.reflect.InvocationTargetException;
/*      */ import java.lang.reflect.Method;
/*      */ import java.net.URL;
/*      */ import java.util.Enumeration;
/*      */ import java.util.Hashtable;
/*      */ import java.util.Vector;
/*      */ import org.apache.commons.logging.Log;
/*      */ import org.apache.commons.logging.LogConfigurationException;
/*      */ import org.apache.commons.logging.LogFactory;
/*      */ 
/*      */ public class LogFactoryImpl extends LogFactory
/*      */ {
/*      */   private static final String LOGGING_IMPL_LOG4J_LOGGER = "org.apache.commons.logging.impl.Log4JLogger";
/*      */   private static final String LOGGING_IMPL_JDK14_LOGGER = "org.apache.commons.logging.impl.Jdk14Logger";
/*      */   private static final String LOGGING_IMPL_LUMBERJACK_LOGGER = "org.apache.commons.logging.impl.Jdk13LumberjackLogger";
/*      */   private static final String LOGGING_IMPL_SIMPLE_LOGGER = "org.apache.commons.logging.impl.SimpleLog";
/*      */   private static final String PKG_IMPL = "org.apache.commons.logging.impl.";
/*   81 */   private static final int PKG_LEN = "org.apache.commons.logging.impl.".length();
/*      */   public static final String LOG_PROPERTY = "org.apache.commons.logging.Log";
/*      */   protected static final String LOG_PROPERTY_OLD = "org.apache.commons.logging.log";
/*      */   public static final String ALLOW_FLAWED_CONTEXT_PROPERTY = "org.apache.commons.logging.Log.allowFlawedContext";
/*      */   public static final String ALLOW_FLAWED_DISCOVERY_PROPERTY = "org.apache.commons.logging.Log.allowFlawedDiscovery";
/*      */   public static final String ALLOW_FLAWED_HIERARCHY_PROPERTY = "org.apache.commons.logging.Log.allowFlawedHierarchy";
/*  173 */   private static final String[] classesToDiscover = { "org.apache.commons.logging.impl.Log4JLogger", "org.apache.commons.logging.impl.Jdk14Logger", "org.apache.commons.logging.impl.Jdk13LumberjackLogger", "org.apache.commons.logging.impl.SimpleLog" };
/*      */ 
/*  187 */   private boolean useTCCL = true;
/*      */   private String diagnosticPrefix;
/*  198 */   protected Hashtable attributes = new Hashtable();
/*      */ 
/*  205 */   protected Hashtable instances = new Hashtable();
/*      */   private String logClassName;
/*  221 */   protected Constructor logConstructor = null;
/*      */ 
/*  227 */   protected Class[] logConstructorSignature = { String.class };
/*      */ 
/*  235 */   protected Method logMethod = null;
/*      */ 
/*  241 */   protected Class[] logMethodSignature = { LogFactory.class };
/*      */   private boolean allowFlawedContext;
/*      */   private boolean allowFlawedDiscovery;
/*      */   private boolean allowFlawedHierarchy;
/*      */ 
/*      */   public LogFactoryImpl()
/*      */   {
/*   92 */     initDiagnostics();
/*   93 */     if (isDiagnosticsEnabled())
/*   94 */       logDiagnostic("Instance created.");
/*      */   }
/*      */ 
/*      */   public Object getAttribute(String name)
/*      */   {
/*  270 */     return this.attributes.get(name);
/*      */   }
/*      */ 
/*      */   public String[] getAttributeNames()
/*      */   {
/*  282 */     Vector names = new Vector();
/*  283 */     Enumeration keys = this.attributes.keys();
/*  284 */     while (keys.hasMoreElements()) {
/*  285 */       names.addElement((String)keys.nextElement());
/*      */     }
/*  287 */     String[] results = new String[names.size()];
/*  288 */     for (int i = 0; i < results.length; i++) {
/*  289 */       results[i] = ((String)names.elementAt(i));
/*      */     }
/*  291 */     return results;
/*      */   }
/*      */ 
/*      */   public Log getInstance(Class clazz)
/*      */     throws LogConfigurationException
/*      */   {
/*  307 */     return getInstance(clazz.getName());
/*      */   }
/*      */ 
/*      */   public Log getInstance(String name)
/*      */     throws LogConfigurationException
/*      */   {
/*  331 */     Log instance = (Log)this.instances.get(name);
/*  332 */     if (instance == null) {
/*  333 */       instance = newInstance(name);
/*  334 */       this.instances.put(name, instance);
/*      */     }
/*  336 */     return instance;
/*      */   }
/*      */ 
/*      */   public void release()
/*      */   {
/*  351 */     logDiagnostic("Releasing all known loggers");
/*  352 */     this.instances.clear();
/*      */   }
/*      */ 
/*      */   public void removeAttribute(String name)
/*      */   {
/*  364 */     this.attributes.remove(name);
/*      */   }
/*      */ 
/*      */   public void setAttribute(String name, Object value)
/*      */   {
/*  395 */     if (this.logConstructor != null) {
/*  396 */       logDiagnostic("setAttribute: call too late; configuration already performed.");
/*      */     }
/*      */ 
/*  399 */     if (value == null)
/*  400 */       this.attributes.remove(name);
/*      */     else {
/*  402 */       this.attributes.put(name, value);
/*      */     }
/*      */ 
/*  405 */     if (name.equals("use_tccl"))
/*  406 */       this.useTCCL = Boolean.valueOf(value.toString()).booleanValue();
/*      */   }
/*      */ 
/*      */   protected static ClassLoader getContextClassLoader()
/*      */     throws LogConfigurationException
/*      */   {
/*  425 */     return LogFactory.getContextClassLoader();
/*      */   }
/*      */ 
/*      */   protected static boolean isDiagnosticsEnabled()
/*      */   {
/*  434 */     return LogFactory.isDiagnosticsEnabled();
/*      */   }
/*      */ 
/*      */   protected static ClassLoader getClassLoader(Class clazz)
/*      */   {
/*  444 */     return LogFactory.getClassLoader(clazz);
/*      */   }
/*      */ 
/*      */   private void initDiagnostics()
/*      */   {
/*  472 */     Class clazz = getClass();
/*  473 */     ClassLoader classLoader = getClassLoader(clazz);
/*      */     String classLoaderName;
/*      */     try
/*      */     {
/*      */       String classLoaderName;
/*  476 */       if (classLoader == null)
/*  477 */         classLoaderName = "BOOTLOADER";
/*      */       else
/*  479 */         classLoaderName = LogFactory.objectId(classLoader);
/*      */     }
/*      */     catch (SecurityException e) {
/*  482 */       classLoaderName = "UNKNOWN";
/*      */     }
/*  484 */     this.diagnosticPrefix = ("[LogFactoryImpl@" + System.identityHashCode(this) + " from " + classLoaderName + "] ");
/*      */   }
/*      */ 
/*      */   protected void logDiagnostic(String msg)
/*      */   {
/*  496 */     if (isDiagnosticsEnabled())
/*  497 */       LogFactory.logRawDiagnostic(this.diagnosticPrefix + msg);
/*      */   }
/*      */ 
/*      */   /** @deprecated */
/*      */   protected String getLogClassName()
/*      */   {
/*  510 */     if (this.logClassName == null) {
/*  511 */       discoverLogImplementation(getClass().getName());
/*      */     }
/*      */ 
/*  514 */     return this.logClassName;
/*      */   }
/*      */ 
/*      */   /** @deprecated */
/*      */   protected Constructor getLogConstructor()
/*      */     throws LogConfigurationException
/*      */   {
/*  537 */     if (this.logConstructor == null) {
/*  538 */       discoverLogImplementation(getClass().getName());
/*      */     }
/*      */ 
/*  541 */     return this.logConstructor;
/*      */   }
/*      */ 
/*      */   /** @deprecated */
/*      */   protected boolean isJdk13LumberjackAvailable()
/*      */   {
/*  552 */     return isLogLibraryAvailable("Jdk13Lumberjack", "org.apache.commons.logging.impl.Jdk13LumberjackLogger");
/*      */   }
/*      */ 
/*      */   /** @deprecated */
/*      */   protected boolean isJdk14Available()
/*      */   {
/*  568 */     return isLogLibraryAvailable("Jdk14", "org.apache.commons.logging.impl.Jdk14Logger");
/*      */   }
/*      */ 
/*      */   /** @deprecated */
/*      */   protected boolean isLog4JAvailable()
/*      */   {
/*  581 */     return isLogLibraryAvailable("Log4J", "org.apache.commons.logging.impl.Log4JLogger");
/*      */   }
/*      */ 
/*      */   protected Log newInstance(String name)
/*      */     throws LogConfigurationException
/*      */   {
/*  598 */     Log instance = null;
/*      */     try {
/*  600 */       if (this.logConstructor == null) {
/*  601 */         instance = discoverLogImplementation(name);
/*      */       }
/*      */       else {
/*  604 */         Object[] params = { name };
/*  605 */         instance = (Log)this.logConstructor.newInstance(params);
/*      */       }
/*      */ 
/*  608 */       if (this.logMethod != null) {
/*  609 */         Object[] params = { this };
/*  610 */         this.logMethod.invoke(instance, params);
/*      */       }
/*      */ 
/*  613 */       return instance;
/*      */     }
/*      */     catch (LogConfigurationException lce)
/*      */     {
/*  620 */       throw lce;
/*      */     }
/*      */     catch (InvocationTargetException e)
/*      */     {
/*  625 */       Throwable c = e.getTargetException();
/*  626 */       if (c != null) {
/*  627 */         throw new LogConfigurationException(c);
/*      */       }
/*  629 */       throw new LogConfigurationException(e);
/*      */     }
/*      */     catch (Throwable t)
/*      */     {
/*      */     }
/*  634 */     throw new LogConfigurationException(t);
/*      */   }
/*      */ 
/*      */   private boolean isLogLibraryAvailable(String name, String classname)
/*      */   {
/*  647 */     if (isDiagnosticsEnabled())
/*  648 */       logDiagnostic("Checking for '" + name + "'.");
/*      */     try
/*      */     {
/*  651 */       Log log = createLogFromClass(classname, getClass().getName(), false);
/*      */ 
/*  656 */       if (log == null) {
/*  657 */         if (isDiagnosticsEnabled()) {
/*  658 */           logDiagnostic("Did not find '" + name + "'.");
/*      */         }
/*  660 */         return false;
/*      */       }
/*  662 */       if (isDiagnosticsEnabled()) {
/*  663 */         logDiagnostic("Found '" + name + "'.");
/*      */       }
/*  665 */       return true;
/*      */     }
/*      */     catch (LogConfigurationException e) {
/*  668 */       if (isDiagnosticsEnabled())
/*  669 */         logDiagnostic("Logging system '" + name + "' is available but not useable.");
/*      */     }
/*  671 */     return false;
/*      */   }
/*      */ 
/*      */   private String getConfigurationValue(String property)
/*      */   {
/*  687 */     if (isDiagnosticsEnabled()) {
/*  688 */       logDiagnostic("[ENV] Trying to get configuration for item " + property);
/*      */     }
/*      */ 
/*  691 */     Object valueObj = getAttribute(property);
/*  692 */     if (valueObj != null) {
/*  693 */       if (isDiagnosticsEnabled()) {
/*  694 */         logDiagnostic("[ENV] Found LogFactory attribute [" + valueObj + "] for " + property);
/*      */       }
/*  696 */       return valueObj.toString();
/*      */     }
/*      */ 
/*  699 */     if (isDiagnosticsEnabled()) {
/*  700 */       logDiagnostic("[ENV] No LogFactory attribute found for " + property);
/*      */     }
/*      */     try
/*      */     {
/*  704 */       String value = System.getProperty(property);
/*  705 */       if (value != null) {
/*  706 */         if (isDiagnosticsEnabled()) {
/*  707 */           logDiagnostic("[ENV] Found system property [" + value + "] for " + property);
/*      */         }
/*  709 */         return value;
/*      */       }
/*      */ 
/*  712 */       if (isDiagnosticsEnabled())
/*  713 */         logDiagnostic("[ENV] No system property found for property " + property);
/*      */     }
/*      */     catch (SecurityException e) {
/*  716 */       if (isDiagnosticsEnabled()) {
/*  717 */         logDiagnostic("[ENV] Security prevented reading system property " + property);
/*      */       }
/*      */     }
/*      */ 
/*  721 */     if (isDiagnosticsEnabled()) {
/*  722 */       logDiagnostic("[ENV] No configuration defined for item " + property);
/*      */     }
/*      */ 
/*  725 */     return null;
/*      */   }
/*      */ 
/*      */   private boolean getBooleanConfiguration(String key, boolean dflt)
/*      */   {
/*  733 */     String val = getConfigurationValue(key);
/*  734 */     if (val == null)
/*  735 */       return dflt;
/*  736 */     return Boolean.valueOf(val).booleanValue();
/*      */   }
/*      */ 
/*      */   private void initConfiguration()
/*      */   {
/*  747 */     this.allowFlawedContext = getBooleanConfiguration("org.apache.commons.logging.Log.allowFlawedContext", true);
/*  748 */     this.allowFlawedDiscovery = getBooleanConfiguration("org.apache.commons.logging.Log.allowFlawedDiscovery", true);
/*  749 */     this.allowFlawedHierarchy = getBooleanConfiguration("org.apache.commons.logging.Log.allowFlawedHierarchy", true);
/*      */   }
/*      */ 
/*      */   private Log discoverLogImplementation(String logCategory)
/*      */     throws LogConfigurationException
/*      */   {
/*  765 */     if (isDiagnosticsEnabled()) {
/*  766 */       logDiagnostic("Discovering a Log implementation...");
/*      */     }
/*      */ 
/*  769 */     initConfiguration();
/*      */ 
/*  771 */     Log result = null;
/*      */ 
/*  774 */     String specifiedLogClassName = findUserSpecifiedLogClassName();
/*      */ 
/*  776 */     if (specifiedLogClassName != null) {
/*  777 */       if (isDiagnosticsEnabled()) {
/*  778 */         logDiagnostic("Attempting to load user-specified log class '" + specifiedLogClassName + "'...");
/*      */       }
/*      */ 
/*  782 */       result = createLogFromClass(specifiedLogClassName, logCategory, true);
/*      */ 
/*  785 */       if (result == null) {
/*  786 */         StringBuffer messageBuffer = new StringBuffer("User-specified log class '");
/*  787 */         messageBuffer.append(specifiedLogClassName);
/*  788 */         messageBuffer.append("' cannot be found or is not useable.");
/*      */ 
/*  792 */         if (specifiedLogClassName != null) {
/*  793 */           informUponSimilarName(messageBuffer, specifiedLogClassName, "org.apache.commons.logging.impl.Log4JLogger");
/*  794 */           informUponSimilarName(messageBuffer, specifiedLogClassName, "org.apache.commons.logging.impl.Jdk14Logger");
/*  795 */           informUponSimilarName(messageBuffer, specifiedLogClassName, "org.apache.commons.logging.impl.Jdk13LumberjackLogger");
/*  796 */           informUponSimilarName(messageBuffer, specifiedLogClassName, "org.apache.commons.logging.impl.SimpleLog");
/*      */         }
/*  798 */         throw new LogConfigurationException(messageBuffer.toString());
/*      */       }
/*      */ 
/*  801 */       return result;
/*      */     }
/*      */ 
/*  832 */     if (isDiagnosticsEnabled()) {
/*  833 */       logDiagnostic("No user-specified Log implementation; performing discovery using the standard supported logging implementations...");
/*      */     }
/*      */ 
/*  837 */     for (int i = 0; (i < classesToDiscover.length) && (result == null); i++) {
/*  838 */       result = createLogFromClass(classesToDiscover[i], logCategory, true);
/*      */     }
/*      */ 
/*  841 */     if (result == null) {
/*  842 */       throw new LogConfigurationException("No suitable Log implementation");
/*      */     }
/*      */ 
/*  846 */     return result;
/*      */   }
/*      */ 
/*      */   private void informUponSimilarName(StringBuffer messageBuffer, String name, String candidate)
/*      */   {
/*  859 */     if (name.equals(candidate))
/*      */     {
/*  862 */       return;
/*      */     }
/*      */ 
/*  868 */     if (name.regionMatches(true, 0, candidate, 0, PKG_LEN + 5)) {
/*  869 */       messageBuffer.append(" Did you mean '");
/*  870 */       messageBuffer.append(candidate);
/*  871 */       messageBuffer.append("'?");
/*      */     }
/*      */   }
/*      */ 
/*      */   private String findUserSpecifiedLogClassName()
/*      */   {
/*  885 */     if (isDiagnosticsEnabled()) {
/*  886 */       logDiagnostic("Trying to get log class from attribute 'org.apache.commons.logging.Log'");
/*      */     }
/*  888 */     String specifiedClass = (String)getAttribute("org.apache.commons.logging.Log");
/*      */ 
/*  890 */     if (specifiedClass == null) {
/*  891 */       if (isDiagnosticsEnabled()) {
/*  892 */         logDiagnostic("Trying to get log class from attribute 'org.apache.commons.logging.log'");
/*      */       }
/*      */ 
/*  895 */       specifiedClass = (String)getAttribute("org.apache.commons.logging.log");
/*      */     }
/*      */ 
/*  898 */     if (specifiedClass == null) {
/*  899 */       if (isDiagnosticsEnabled()) {
/*  900 */         logDiagnostic("Trying to get log class from system property 'org.apache.commons.logging.Log'");
/*      */       }
/*      */       try
/*      */       {
/*  904 */         specifiedClass = System.getProperty("org.apache.commons.logging.Log");
/*      */       } catch (SecurityException e) {
/*  906 */         if (isDiagnosticsEnabled()) {
/*  907 */           logDiagnostic("No access allowed to system property 'org.apache.commons.logging.Log' - " + e.getMessage());
/*      */         }
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/*  913 */     if (specifiedClass == null) {
/*  914 */       if (isDiagnosticsEnabled()) {
/*  915 */         logDiagnostic("Trying to get log class from system property 'org.apache.commons.logging.log'");
/*      */       }
/*      */       try
/*      */       {
/*  919 */         specifiedClass = System.getProperty("org.apache.commons.logging.log");
/*      */       } catch (SecurityException e) {
/*  921 */         if (isDiagnosticsEnabled()) {
/*  922 */           logDiagnostic("No access allowed to system property 'org.apache.commons.logging.log' - " + e.getMessage());
/*      */         }
/*      */ 
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/*  931 */     if (specifiedClass != null) {
/*  932 */       specifiedClass = specifiedClass.trim();
/*      */     }
/*      */ 
/*  935 */     return specifiedClass;
/*      */   }
/*      */ 
/*      */   private Log createLogFromClass(String logAdapterClassName, String logCategory, boolean affectState)
/*      */     throws LogConfigurationException
/*      */   {
/*  963 */     if (isDiagnosticsEnabled()) {
/*  964 */       logDiagnostic("Attempting to instantiate '" + logAdapterClassName + "'");
/*      */     }
/*      */ 
/*  967 */     Object[] params = { logCategory };
/*  968 */     Log logAdapter = null;
/*  969 */     Constructor constructor = null;
/*      */ 
/*  971 */     Class logAdapterClass = null;
/*  972 */     ClassLoader currentCL = getBaseClassLoader();
/*      */     while (true)
/*      */     {
/*  977 */       logDiagnostic("Trying to load '" + logAdapterClassName + "' from classloader " + LogFactory.objectId(currentCL));
/*      */       try
/*      */       {
/*  983 */         if (isDiagnosticsEnabled())
/*      */         {
/*  989 */           String resourceName = logAdapterClassName.replace('.', '/') + ".class";
/*      */           URL url;
/*      */           URL url;
/*  990 */           if (currentCL != null)
/*  991 */             url = currentCL.getResource(resourceName);
/*      */           else {
/*  993 */             url = ClassLoader.getSystemResource(resourceName + ".class");
/*      */           }
/*      */ 
/*  996 */           if (url == null)
/*  997 */             logDiagnostic("Class '" + logAdapterClassName + "' [" + resourceName + "] cannot be found.");
/*      */           else {
/*  999 */             logDiagnostic("Class '" + logAdapterClassName + "' was found at '" + url + "'");
/*      */           }
/*      */         }
/*      */ 
/* 1003 */         Class c = null;
/*      */         try {
/* 1005 */           c = Class.forName(logAdapterClassName, true, currentCL);
/*      */         }
/*      */         catch (ClassNotFoundException originalClassNotFoundException)
/*      */         {
/* 1010 */           String msg = "" + originalClassNotFoundException.getMessage();
/* 1011 */           logDiagnostic("The log adapter '" + logAdapterClassName + "' is not available via classloader " + LogFactory.objectId(currentCL) + ": " + msg.trim());
/*      */           try
/*      */           {
/* 1026 */             c = Class.forName(logAdapterClassName);
/*      */           }
/*      */           catch (ClassNotFoundException secondaryClassNotFoundException) {
/* 1029 */             msg = "" + secondaryClassNotFoundException.getMessage();
/* 1030 */             logDiagnostic("The log adapter '" + logAdapterClassName + "' is not available via the LogFactoryImpl class classloader: " + msg.trim());
/*      */ 
/* 1035 */             break;
/*      */           }
/*      */         }
/*      */ 
/* 1039 */         constructor = c.getConstructor(this.logConstructorSignature);
/* 1040 */         Object o = constructor.newInstance(params);
/*      */ 
/* 1046 */         if ((o instanceof Log)) {
/* 1047 */           logAdapterClass = c;
/* 1048 */           logAdapter = (Log)o;
/* 1049 */           break;
/*      */         }
/*      */ 
/* 1062 */         handleFlawedHierarchy(currentCL, c);
/*      */       }
/*      */       catch (NoClassDefFoundError e)
/*      */       {
/* 1069 */         String msg = "" + e.getMessage();
/* 1070 */         logDiagnostic("The log adapter '" + logAdapterClassName + "' is missing dependencies when loaded via classloader " + LogFactory.objectId(currentCL) + ": " + msg.trim());
/*      */ 
/* 1077 */         break;
/*      */       }
/*      */       catch (ExceptionInInitializerError e)
/*      */       {
/* 1085 */         String msg = "" + e.getMessage();
/* 1086 */         logDiagnostic("The log adapter '" + logAdapterClassName + "' is unable to initialize itself when loaded via classloader " + LogFactory.objectId(currentCL) + ": " + msg.trim());
/*      */ 
/* 1093 */         break;
/*      */       }
/*      */       catch (LogConfigurationException e)
/*      */       {
/* 1097 */         throw e;
/*      */       }
/*      */       catch (Throwable t)
/*      */       {
/* 1102 */         handleFlawedDiscovery(logAdapterClassName, currentCL, t);
/*      */       }
/*      */ 
/* 1105 */       if (currentCL == null)
/*      */       {
/*      */         break;
/*      */       }
/*      */ 
/* 1110 */       currentCL = currentCL.getParent();
/*      */     }
/*      */ 
/* 1113 */     if ((logAdapter != null) && (affectState))
/*      */     {
/* 1115 */       this.logClassName = logAdapterClassName;
/* 1116 */       this.logConstructor = constructor;
/*      */       try
/*      */       {
/* 1120 */         this.logMethod = logAdapterClass.getMethod("setLogFactory", this.logMethodSignature);
/*      */ 
/* 1122 */         logDiagnostic("Found method setLogFactory(LogFactory) in '" + logAdapterClassName + "'");
/*      */       }
/*      */       catch (Throwable t) {
/* 1125 */         this.logMethod = null;
/* 1126 */         logDiagnostic("[INFO] '" + logAdapterClassName + "' from classloader " + LogFactory.objectId(currentCL) + " does not declare optional method " + "setLogFactory(LogFactory)");
/*      */       }
/*      */ 
/* 1133 */       logDiagnostic("Log adapter '" + logAdapterClassName + "' from classloader " + LogFactory.objectId(logAdapterClass.getClassLoader()) + " has been selected for use.");
/*      */     }
/*      */ 
/* 1139 */     return logAdapter;
/*      */   }
/*      */ 
/*      */   private ClassLoader getBaseClassLoader()
/*      */     throws LogConfigurationException
/*      */   {
/* 1162 */     ClassLoader thisClassLoader = getClassLoader(LogFactoryImpl.class);
/*      */ 
/* 1164 */     if (!this.useTCCL) {
/* 1165 */       return thisClassLoader;
/*      */     }
/*      */ 
/* 1168 */     ClassLoader contextClassLoader = getContextClassLoader();
/*      */ 
/* 1170 */     ClassLoader baseClassLoader = getLowestClassLoader(contextClassLoader, thisClassLoader);
/*      */ 
/* 1173 */     if (baseClassLoader == null)
/*      */     {
/* 1178 */       if (this.allowFlawedContext) {
/* 1179 */         if (isDiagnosticsEnabled()) {
/* 1180 */           logDiagnostic("[WARNING] the context classloader is not part of a parent-child relationship with the classloader that loaded LogFactoryImpl.");
/*      */         }
/*      */ 
/* 1188 */         return contextClassLoader;
/*      */       }
/*      */ 
/* 1191 */       throw new LogConfigurationException("Bad classloader hierarchy; LogFactoryImpl was loaded via a classloader that is not related to the current context classloader.");
/*      */     }
/*      */ 
/* 1198 */     if (baseClassLoader != contextClassLoader)
/*      */     {
/* 1204 */       if (this.allowFlawedContext) {
/* 1205 */         if (isDiagnosticsEnabled()) {
/* 1206 */           logDiagnostic("Warning: the context classloader is an ancestor of the classloader that loaded LogFactoryImpl; it should be the same or a descendant. The application using commons-logging should ensure the context classloader is used correctly.");
/*      */         }
/*      */ 
/*      */       }
/*      */       else
/*      */       {
/* 1214 */         throw new LogConfigurationException("Bad classloader hierarchy; LogFactoryImpl was loaded via a classloader that is not related to the current context classloader.");
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/* 1221 */     return baseClassLoader;
/*      */   }
/*      */ 
/*      */   private ClassLoader getLowestClassLoader(ClassLoader c1, ClassLoader c2)
/*      */   {
/* 1237 */     if (c1 == null) {
/* 1238 */       return c2;
/*      */     }
/* 1240 */     if (c2 == null) {
/* 1241 */       return c1;
/*      */     }
/*      */ 
/* 1246 */     ClassLoader current = c1;
/* 1247 */     while (current != null) {
/* 1248 */       if (current == c2)
/* 1249 */         return c1;
/* 1250 */       current = current.getParent();
/*      */     }
/*      */ 
/* 1254 */     current = c2;
/* 1255 */     while (current != null) {
/* 1256 */       if (current == c1)
/* 1257 */         return c2;
/* 1258 */       current = current.getParent();
/*      */     }
/*      */ 
/* 1261 */     return null;
/*      */   }
/*      */ 
/*      */   private void handleFlawedDiscovery(String logAdapterClassName, ClassLoader classLoader, Throwable discoveryFlaw)
/*      */   {
/* 1283 */     if (isDiagnosticsEnabled()) {
/* 1284 */       LogFactory.logRawDiagnostic("Could not instantiate Log '" + logAdapterClassName + "' -- " + discoveryFlaw.getClass().getName() + ": " + discoveryFlaw.getLocalizedMessage(), discoveryFlaw);
/*      */     }
/*      */ 
/* 1290 */     if (!this.allowFlawedDiscovery)
/* 1291 */       throw new LogConfigurationException(discoveryFlaw);
/*      */   }
/*      */ 
/*      */   private void handleFlawedHierarchy(ClassLoader badClassLoader, Class badClass)
/*      */     throws LogConfigurationException
/*      */   {
/* 1325 */     boolean implementsLog = false;
/* 1326 */     String logInterfaceName = Log.class.getName();
/* 1327 */     Class[] interfaces = badClass.getInterfaces();
/* 1328 */     for (int i = 0; i < interfaces.length; i++) {
/* 1329 */       if (logInterfaceName.equals(interfaces[i].getName())) {
/* 1330 */         implementsLog = true;
/* 1331 */         break;
/*      */       }
/*      */     }
/*      */ 
/* 1335 */     if (implementsLog)
/*      */     {
/* 1338 */       if (isDiagnosticsEnabled()) {
/*      */         try {
/* 1340 */           ClassLoader logInterfaceClassLoader = getClassLoader(Log.class);
/* 1341 */           logDiagnostic("Class '" + badClass.getName() + "' was found in classloader " + LogFactory.objectId(badClassLoader) + ". It is bound to a Log interface which is not" + " the one loaded from classloader " + LogFactory.objectId(logInterfaceClassLoader));
/*      */         }
/*      */         catch (Throwable t)
/*      */         {
/* 1349 */           logDiagnostic("Error while trying to output diagnostics about bad class '" + badClass + "'");
/*      */         }
/*      */ 
/*      */       }
/*      */ 
/* 1355 */       if (!this.allowFlawedHierarchy) {
/* 1356 */         StringBuffer msg = new StringBuffer();
/* 1357 */         msg.append("Terminating logging for this context ");
/* 1358 */         msg.append("due to bad log hierarchy. ");
/* 1359 */         msg.append("You have more than one version of '");
/* 1360 */         msg.append(Log.class.getName());
/* 1361 */         msg.append("' visible.");
/* 1362 */         if (isDiagnosticsEnabled()) {
/* 1363 */           logDiagnostic(msg.toString());
/*      */         }
/* 1365 */         throw new LogConfigurationException(msg.toString());
/*      */       }
/*      */ 
/* 1368 */       if (isDiagnosticsEnabled()) {
/* 1369 */         StringBuffer msg = new StringBuffer();
/* 1370 */         msg.append("Warning: bad log hierarchy. ");
/* 1371 */         msg.append("You have more than one version of '");
/* 1372 */         msg.append(Log.class.getName());
/* 1373 */         msg.append("' visible.");
/* 1374 */         logDiagnostic(msg.toString());
/*      */       }
/*      */     }
/*      */     else {
/* 1378 */       if (!this.allowFlawedDiscovery) {
/* 1379 */         StringBuffer msg = new StringBuffer();
/* 1380 */         msg.append("Terminating logging for this context. ");
/* 1381 */         msg.append("Log class '");
/* 1382 */         msg.append(badClass.getName());
/* 1383 */         msg.append("' does not implement the Log interface.");
/* 1384 */         if (isDiagnosticsEnabled()) {
/* 1385 */           logDiagnostic(msg.toString());
/*      */         }
/*      */ 
/* 1388 */         throw new LogConfigurationException(msg.toString());
/*      */       }
/*      */ 
/* 1391 */       if (isDiagnosticsEnabled()) {
/* 1392 */         StringBuffer msg = new StringBuffer();
/* 1393 */         msg.append("[WARNING] Log class '");
/* 1394 */         msg.append(badClass.getName());
/* 1395 */         msg.append("' does not implement the Log interface.");
/* 1396 */         logDiagnostic(msg.toString());
/*      */       }
/*      */     }
/*      */   }
/*      */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.apache.commons.logging.impl.LogFactoryImpl
 * JD-Core Version:    0.6.0
 */