/*     */ package gnu.trove;
/*     */ 
/*     */ import java.io.Serializable;
/*     */ import java.util.Arrays;
/*     */ import java.util.Random;
/*     */ 
/*     */ public class TDoubleArrayList
/*     */   implements Serializable, Cloneable
/*     */ {
/*     */   protected double[] _data;
/*     */   protected int _pos;
/*     */   protected static final int DEFAULT_CAPACITY = 10;
/*     */ 
/*     */   public TDoubleArrayList()
/*     */   {
/*  50 */     this(10);
/*     */   }
/*     */ 
/*     */   public TDoubleArrayList(int capacity)
/*     */   {
/*  60 */     this._data = new double[capacity];
/*  61 */     this._pos = 0;
/*     */   }
/*     */ 
/*     */   public TDoubleArrayList(double[] values)
/*     */   {
/*  73 */     this(Math.max(values.length, 10));
/*  74 */     add(values);
/*     */   }
/*     */ 
/*     */   public void ensureCapacity(int capacity)
/*     */   {
/*  88 */     if (capacity > this._data.length) {
/*  89 */       int newCap = Math.max(this._data.length << 1, capacity);
/*  90 */       double[] tmp = new double[newCap];
/*  91 */       System.arraycopy(this._data, 0, tmp, 0, this._data.length);
/*  92 */       this._data = tmp;
/*     */     }
/*     */   }
/*     */ 
/*     */   public int size()
/*     */   {
/* 102 */     return this._pos;
/*     */   }
/*     */ 
/*     */   public boolean isEmpty()
/*     */   {
/* 111 */     return this._pos == 0;
/*     */   }
/*     */ 
/*     */   public void trimToSize()
/*     */   {
/* 119 */     if (this._data.length > size()) {
/* 120 */       double[] tmp = new double[size()];
/* 121 */       toNativeArray(tmp, 0, tmp.length);
/* 122 */       this._data = tmp;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void add(double val)
/*     */   {
/* 134 */     ensureCapacity(this._pos + 1);
/* 135 */     this._data[(this._pos++)] = val;
/*     */   }
/*     */ 
/*     */   public void add(double[] vals)
/*     */   {
/* 145 */     add(vals, 0, vals.length);
/*     */   }
/*     */ 
/*     */   public void add(double[] vals, int offset, int length)
/*     */   {
/* 157 */     ensureCapacity(this._pos + length);
/* 158 */     System.arraycopy(vals, offset, this._data, this._pos, length);
/* 159 */     this._pos += length;
/*     */   }
/*     */ 
/*     */   public void insert(int offset, double value)
/*     */   {
/* 171 */     if (offset == this._pos) {
/* 172 */       add(value);
/* 173 */       return;
/*     */     }
/* 175 */     ensureCapacity(this._pos + 1);
/*     */ 
/* 177 */     System.arraycopy(this._data, offset, this._data, offset + 1, this._pos - offset);
/*     */ 
/* 179 */     this._data[offset] = value;
/* 180 */     this._pos += 1;
/*     */   }
/*     */ 
/*     */   public void insert(int offset, double[] values)
/*     */   {
/* 192 */     insert(offset, values, 0, values.length);
/*     */   }
/*     */ 
/*     */   public void insert(int offset, double[] values, int valOffset, int len)
/*     */   {
/* 207 */     if (offset == this._pos) {
/* 208 */       add(values, valOffset, len);
/* 209 */       return;
/*     */     }
/*     */ 
/* 212 */     ensureCapacity(this._pos + len);
/*     */ 
/* 214 */     System.arraycopy(this._data, offset, this._data, offset + len, this._pos - offset);
/*     */ 
/* 216 */     System.arraycopy(values, valOffset, this._data, offset, len);
/* 217 */     this._pos += len;
/*     */   }
/*     */ 
/*     */   public double get(int offset)
/*     */   {
/* 227 */     if (offset >= this._pos) {
/* 228 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 230 */     return this._data[offset];
/*     */   }
/*     */ 
/*     */   public double getQuick(int offset)
/*     */   {
/* 241 */     return this._data[offset];
/*     */   }
/*     */ 
/*     */   public void set(int offset, double val)
/*     */   {
/* 251 */     if (offset >= this._pos) {
/* 252 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 254 */     this._data[offset] = val;
/*     */   }
/*     */ 
/*     */   public double getSet(int offset, double val)
/*     */   {
/* 266 */     if (offset >= this._pos) {
/* 267 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 269 */     double old = this._data[offset];
/* 270 */     this._data[offset] = val;
/* 271 */     return old;
/*     */   }
/*     */ 
/*     */   public void set(int offset, double[] values)
/*     */   {
/* 282 */     set(offset, values, 0, values.length);
/*     */   }
/*     */ 
/*     */   public void set(int offset, double[] values, int valOffset, int length)
/*     */   {
/* 296 */     if ((offset < 0) || (offset + length >= this._pos)) {
/* 297 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 299 */     System.arraycopy(this._data, offset, values, valOffset, length);
/*     */   }
/*     */ 
/*     */   public void setQuick(int offset, double val)
/*     */   {
/* 310 */     this._data[offset] = val;
/*     */   }
/*     */ 
/*     */   public void clear()
/*     */   {
/* 318 */     clear(10);
/*     */   }
/*     */ 
/*     */   public void clear(int capacity)
/*     */   {
/* 328 */     this._data = new double[capacity];
/* 329 */     this._pos = 0;
/*     */   }
/*     */ 
/*     */   public void reset()
/*     */   {
/* 341 */     this._pos = 0;
/* 342 */     fill(0.0D);
/*     */   }
/*     */ 
/*     */   public void resetQuick()
/*     */   {
/* 360 */     this._pos = 0;
/*     */   }
/*     */ 
/*     */   public double remove(int offset)
/*     */   {
/* 370 */     double old = get(offset);
/* 371 */     remove(offset, 1);
/* 372 */     return old;
/*     */   }
/*     */ 
/*     */   public void remove(int offset, int length)
/*     */   {
/* 383 */     if ((offset < 0) || (offset >= this._pos)) {
/* 384 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/*     */ 
/* 387 */     if (offset == 0)
/*     */     {
/* 389 */       System.arraycopy(this._data, length, this._data, 0, this._pos - length);
/* 390 */     } else if (this._pos - length != offset)
/*     */     {
/* 395 */       System.arraycopy(this._data, offset + length, this._data, offset, this._pos - (offset + length));
/*     */     }
/*     */ 
/* 398 */     this._pos -= length;
/*     */   }
/*     */ 
/*     */   public void transformValues(TDoubleFunction function)
/*     */   {
/* 410 */     for (int i = this._pos; i-- > 0; )
/* 411 */       this._data[i] = function.execute(this._data[i]);
/*     */   }
/*     */ 
/*     */   public void reverse()
/*     */   {
/* 419 */     reverse(0, this._pos);
/*     */   }
/*     */ 
/*     */   public void reverse(int from, int to)
/*     */   {
/* 429 */     if (from == to) {
/* 430 */       return;
/*     */     }
/* 432 */     if (from > to) {
/* 433 */       throw new IllegalArgumentException("from cannot be greater than to");
/*     */     }
/* 435 */     int i = from; for (int j = to - 1; i < j; j--) {
/* 436 */       swap(i, j);
/*     */ 
/* 435 */       i++;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void shuffle(Random rand)
/*     */   {
/* 447 */     for (int i = this._pos; i-- > 1; )
/* 448 */       swap(i, rand.nextInt(i));
/*     */   }
/*     */ 
/*     */   private final void swap(int i, int j)
/*     */   {
/* 459 */     double tmp = this._data[i];
/* 460 */     this._data[i] = this._data[j];
/* 461 */     this._data[j] = tmp;
/*     */   }
/*     */ 
/*     */   public Object clone()
/*     */   {
/* 473 */     Object o = null;
/*     */     try {
/* 475 */       o = super.clone();
/*     */     }
/*     */     catch (CloneNotSupportedException e) {
/*     */     }
/* 479 */     return o;
/*     */   }
/*     */ 
/*     */   public double[] toNativeArray()
/*     */   {
/* 488 */     return toNativeArray(0, this._pos);
/*     */   }
/*     */ 
/*     */   public double[] toNativeArray(int offset, int len)
/*     */   {
/* 499 */     double[] rv = new double[len];
/* 500 */     toNativeArray(rv, offset, len);
/* 501 */     return rv;
/*     */   }
/*     */ 
/*     */   public void toNativeArray(double[] dest, int offset, int len)
/*     */   {
/* 512 */     if (len == 0) {
/* 513 */       return;
/*     */     }
/* 515 */     if ((offset < 0) || (offset >= this._pos)) {
/* 516 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 518 */     System.arraycopy(this._data, 0, dest, offset, len);
/*     */   }
/*     */ 
/*     */   public boolean equals(Object other)
/*     */   {
/* 531 */     if (other == this)
/* 532 */       return true;
/* 533 */     if ((other instanceof TDoubleArrayList)) {
/* 534 */       TDoubleArrayList that = (TDoubleArrayList)other;
/* 535 */       if (that.size() != size()) {
/* 536 */         return false;
/*     */       }
/* 538 */       for (int i = this._pos; i-- > 0; ) {
/* 539 */         if (this._data[i] != that._data[i]) {
/* 540 */           return false;
/*     */         }
/*     */       }
/* 543 */       return true;
/*     */     }
/*     */ 
/* 546 */     return false;
/*     */   }
/*     */ 
/*     */   public int hashCode()
/*     */   {
/* 551 */     int h = 0;
/* 552 */     for (int i = this._pos; i-- > 0; ) {
/* 553 */       h += HashFunctions.hash(this._data[i]);
/*     */     }
/* 555 */     return h;
/*     */   }
/*     */ 
/*     */   public boolean forEach(TDoubleProcedure procedure)
/*     */   {
/* 568 */     for (int i = 0; i < this._pos; i++) {
/* 569 */       if (!procedure.execute(this._data[i])) {
/* 570 */         return false;
/*     */       }
/*     */     }
/* 573 */     return true;
/*     */   }
/*     */ 
/*     */   public boolean forEachDescending(TDoubleProcedure procedure)
/*     */   {
/* 584 */     for (int i = this._pos; i-- > 0; ) {
/* 585 */       if (!procedure.execute(this._data[i])) {
/* 586 */         return false;
/*     */       }
/*     */     }
/* 589 */     return true;
/*     */   }
/*     */ 
/*     */   public void sort()
/*     */   {
/* 601 */     Arrays.sort(this._data, 0, this._pos);
/*     */   }
/*     */ 
/*     */   public void sort(int fromIndex, int toIndex)
/*     */   {
/* 613 */     Arrays.sort(this._data, fromIndex, toIndex);
/*     */   }
/*     */ 
/*     */   public void fill(double val)
/*     */   {
/* 624 */     Arrays.fill(this._data, 0, this._pos, val);
/*     */   }
/*     */ 
/*     */   public void fill(int fromIndex, int toIndex, double val)
/*     */   {
/* 635 */     if (toIndex > this._pos) {
/* 636 */       ensureCapacity(toIndex);
/* 637 */       this._pos = toIndex;
/*     */     }
/* 639 */     Arrays.fill(this._data, fromIndex, toIndex, val);
/*     */   }
/*     */ 
/*     */   public int binarySearch(double value)
/*     */   {
/* 654 */     return binarySearch(value, 0, this._pos);
/*     */   }
/*     */ 
/*     */   public int binarySearch(double value, int fromIndex, int toIndex)
/*     */   {
/* 669 */     if (fromIndex < 0) {
/* 670 */       throw new ArrayIndexOutOfBoundsException(fromIndex);
/*     */     }
/* 672 */     if (toIndex > this._pos) {
/* 673 */       throw new ArrayIndexOutOfBoundsException(toIndex);
/*     */     }
/*     */ 
/* 676 */     int low = fromIndex;
/* 677 */     int high = toIndex - 1;
/*     */ 
/* 679 */     while (low <= high) {
/* 680 */       int mid = low + high >> 1;
/* 681 */       double midVal = this._data[mid];
/*     */ 
/* 683 */       if (midVal < value)
/* 684 */         low = mid + 1;
/* 685 */       else if (midVal > value)
/* 686 */         high = mid - 1;
/*     */       else {
/* 688 */         return mid;
/*     */       }
/*     */     }
/* 691 */     return -(low + 1);
/*     */   }
/*     */ 
/*     */   public int indexOf(double value)
/*     */   {
/* 704 */     return indexOf(0, value);
/*     */   }
/*     */ 
/*     */   public int indexOf(int offset, double value)
/*     */   {
/* 719 */     for (int i = offset; i < this._pos; i++) {
/* 720 */       if (this._data[i] == value) {
/* 721 */         return i;
/*     */       }
/*     */     }
/* 724 */     return -1;
/*     */   }
/*     */ 
/*     */   public int lastIndexOf(double value)
/*     */   {
/* 737 */     return lastIndexOf(this._pos, value);
/*     */   }
/*     */ 
/*     */   public int lastIndexOf(int offset, double value)
/*     */   {
/* 752 */     for (int i = offset; i-- > 0; ) {
/* 753 */       if (this._data[i] == value) {
/* 754 */         return i;
/*     */       }
/*     */     }
/* 757 */     return -1;
/*     */   }
/*     */ 
/*     */   public boolean contains(double value)
/*     */   {
/* 767 */     return lastIndexOf(value) >= 0;
/*     */   }
/*     */ 
/*     */   public TDoubleArrayList grep(TDoubleProcedure condition)
/*     */   {
/* 778 */     TDoubleArrayList list = new TDoubleArrayList();
/* 779 */     for (int i = 0; i < this._pos; i++) {
/* 780 */       if (condition.execute(this._data[i])) {
/* 781 */         list.add(this._data[i]);
/*     */       }
/*     */     }
/* 784 */     return list;
/*     */   }
/*     */ 
/*     */   public TDoubleArrayList inverseGrep(TDoubleProcedure condition)
/*     */   {
/* 795 */     TDoubleArrayList list = new TDoubleArrayList();
/* 796 */     for (int i = 0; i < this._pos; i++) {
/* 797 */       if (!condition.execute(this._data[i])) {
/* 798 */         list.add(this._data[i]);
/*     */       }
/*     */     }
/* 801 */     return list;
/*     */   }
/*     */ 
/*     */   public double max()
/*     */   {
/* 811 */     if (size() == 0) {
/* 812 */       throw new IllegalStateException("cannot find maximum of an empty list");
/*     */     }
/* 814 */     double max = this._data[(this._pos - 1)];
/* 815 */     for (int i = this._pos - 1; i-- > 0; ) {
/* 816 */       max = Math.max(max, this._data[this._pos]);
/*     */     }
/* 818 */     return max;
/*     */   }
/*     */ 
/*     */   public double min()
/*     */   {
/* 828 */     if (size() == 0) {
/* 829 */       throw new IllegalStateException("cannot find minimum of an empty list");
/*     */     }
/* 831 */     double min = this._data[(this._pos - 1)];
/* 832 */     for (int i = this._pos - 1; i-- > 0; ) {
/* 833 */       min = Math.min(min, this._data[this._pos]);
/*     */     }
/* 835 */     return min;
/*     */   }
/*     */ 
/*     */   public String toString()
/*     */   {
/* 846 */     StringBuffer buf = new StringBuffer("{");
/* 847 */     forEach(new TDoubleProcedure(buf) { private final StringBuffer val$buf;
/*     */ 
/* 849 */       public boolean execute(double val) { this.val$buf.append(val);
/* 850 */         this.val$buf.append(", ");
/* 851 */         return true;
/*     */       }
/*     */     });
/* 854 */     buf.append("}");
/* 855 */     return buf.toString();
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     gnu.trove.TDoubleArrayList
 * JD-Core Version:    0.6.0
 */