/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.aspects.currentinvocation;

import java.util.EmptyStackException;
import java.util.Stack;

import org.jboss.aop.joinpoint.Invocation;
import org.jboss.logging.Logger;

/**
 * CurrentInvocation
 * 
 * Holds the Invocation associated with the Current Thread for
 * access from any class within the associated Thread's context 
 * (scope).  Internally contains a central registry backed by 
 * a ThreadLocal to contain the Invocation.
 * 
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class CurrentInvocation
{

   // -------------------------------------------------------------------------------------------------||
   // Class Members -----------------------------------------------------------------------------------||
   // -------------------------------------------------------------------------------------------------||

   /*
    * JBoss Logger
    */
   private static final Logger logger = Logger.getLogger(CurrentInvocation.class);

   // -------------------------------------------------------------------------------------------------||
   // Instance Members --------------------------------------------------------------------------------||
   // -------------------------------------------------------------------------------------------------||

   /*
    * Registry for the current Invocation
    */
   private static final StackThreadLocal<Invocation> registry = new StackThreadLocal<Invocation>();

   // -------------------------------------------------------------------------------------------------||
   // Functional Methods ------------------------------------------------------------------------------||
   // -------------------------------------------------------------------------------------------------||

   /**
    * Retrieves the Invocation associated with the current
    * Thread / request if one is associated.  Returns null
    * otherwise
    * 
    *  @return
    */
   public static Invocation getCurrentInvocation()
   {
      try
      {
         return CurrentInvocation.getRegistry().peek();
      }
      // In case there's no invocation on the stack
      catch (EmptyStackException ese)
      {
         return null;
      }
   }

   /**
    * Associates the specified invocation with the current
    * Thread / request and sets it as the current invocation
    * 
    * @param invocation
    */
   static void registerInvocation(Invocation invocation)
   {
      // Log
      logger.trace("Placing invocation " + invocation + " into scope for Thread " + Thread.currentThread().toString());

      // Push
      CurrentInvocation.getRegistry().push(invocation);
   }

   /**
    * Deregisters the current invocation 
    */
   static void deregisterCurrentInvocation()
   {
      // Pop
      Invocation current = CurrentInvocation.getRegistry().pop();

      // Log and pop
      logger
            .trace("Cleared Thread " + Thread.currentThread().toString() + " of association with Invocation " + current);
   }

   // -------------------------------------------------------------------------------------------------||
   // Accessors ---------------------------------------------------------------------------------------||
   // -------------------------------------------------------------------------------------------------||

   private static StackThreadLocal<Invocation> getRegistry()
   {
      return CurrentInvocation.registry;
   }

   // -------------------------------------------------------------------------------------------------||
   // Inner Classes  ----------------------------------------------------------------------------------||
   // -------------------------------------------------------------------------------------------------||

   /**
    * StackThreadLocal
    * 
    * An Enhanced ThreadLocal implementation using a stack-based 
    * backend to push/pop Invocations 
    */
   private static class StackThreadLocal<T extends Invocation> extends ThreadLocal<Stack<T>>
   {

      public T peek()
      {
         return this.get().peek();
      }

      public void push(T invocation)
      {
         this.get().push(invocation);
      }

      public T pop()
      {
         return this.get().pop();
      }

      @Override
      protected Stack<T> initialValue()
      {
         // Initialize to a new Stack
         return new Stack<T>();
      }
   }

}
