/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.poll;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ValidationException;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.Retain;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.TextField;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.chenillekit.tapestry.core.components.DateTimeField;
import org.chorem.pollen.business.ServiceComment;
import org.chorem.pollen.business.ServicePoll;
import org.chorem.pollen.business.ServicePollAccount;
import org.chorem.pollen.business.ServiceResults;
import org.chorem.pollen.business.ServiceVote;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.CommentDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.dto.ResultDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.dto.VoteDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.utils.MD5;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.data.EvenOdd;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.data.uio.DateChoiceUIO;
import org.chorem.pollen.ui.data.uio.ImageChoiceUIO;
import org.chorem.pollen.ui.services.Configuration;
import org.chorem.pollen.ui.utils.ImageUtil;
import org.slf4j.Logger;

/**
 * Classe de la page de vote.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: VoteForPoll.java 2619 2009-07-02 13:33:10Z nrannou $
 */
@IncludeStylesheet("context:css/vote.css")
public class VoteForPoll {

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SessionState
    private UserDTO user;
    @Property
    private boolean userExists;

    /**
     * Objet de session représentant l'url du site.
     */
    @SessionState
    @Property
    private String siteURL;

    @Component
    private Form voteForm;

    @Component(id = "pollAccountName")
    private TextField nameField;

    /** Composant DateTimeField pour les choix du sondage */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField dateDTF;

    @Inject
    private Logger logger;

    @Inject
    private Messages messages;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    /**
     * Sondage pour lequel l'utilisateur vote
     */
    @Property
    @Persist
    private PollDTO poll;

    /**
     * Vote courant de l'utilisateur
     */
    @Property
    private VoteDTO vote;

    /** Choix courant du sondage */
    @Property
    private ChoiceDTO choiceOfPoll;
    /** Choix courant du nouveau vote */
    @Property
    private ChoiceDTO choiceOfVote;

    /** Résultats du sondage */
    @Persist
    private List<ResultDTO> results;

    /**
     * Modèle pour l'affichage de la liste des sondages
     */
    @SuppressWarnings( { "unchecked", "unused" })
    @Property
    @Retain
    private BeanModel voteModel;

    @InjectPage
    private ImageDisplay imageDisplay;

    @InjectComponent
    private Zone pollZone;

    @InjectComponent
    private Zone commentZone;

    /**
     * Objet servant à changer la couleur à chaque ligne de la liste des
     * sondages
     */
    @SuppressWarnings("unused")
    @Property
    private EvenOdd evenOdd = new EvenOdd();

    private boolean addChoice;

    private boolean alreadyVoted;

    /** Liste des choix */
    private List<ChoiceDTO> voteChoices = new ArrayList<ChoiceDTO>();

    @Property
    @Persist
    private PollAccountDTO pollAccount;

    /** Nouveau choix de type texte */
    @Property
    @Persist
    private ChoiceDTO newChoice;

    /** Nouveau choix de type date */
    @Property
    @Persist
    private DateChoiceUIO newDateChoice;

    /** Nouveau choix de type image */
    @Property
    @Persist
    private ImageChoiceUIO newImageChoice;

    /** Commentaire courant pour l'affichage des commentaires */
    @SuppressWarnings("unused")
    @Property
    @Persist
    private CommentDTO comment;

    /** Nouveau commentaire posté */
    @Property
    @Persist
    private CommentDTO newComment;

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "yourName")
    private String yourName;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServiceVote serviceVote;
    @Inject
    private ServiceComment serviceComment;
    @Inject
    private ServicePollAccount servicePollAccount;
    @Inject
    private ServiceResults serviceResults;

    /**
     * Met à jour le sondage
     *
     * @param id l'identifiant haché du sondage
     */
    void updatePoll(String id) {
        if (id != null && !id.equals("")) {
            poll = servicePoll.findPollByPollId(id);
            //if (poll != null) {
            // TODO conversions choice -> choiceUIO pour date et image
            //}
        }
    }

    /** Méthode appelée après la soumission du vote. */
    Object onSuccessFromVoteForm() {
        if (initPollAccount()) {
            //if (poll.getPollType() == PollType.FREE) {
            List<ChoiceDTO> choiceDTOs = new ArrayList<ChoiceDTO>();
            if (poll.getVoteCounting() == VoteCountingType.NORMAL) {
                int nbChoices = 0;
                for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                    if (choice.getValue() == 1) {
                        nbChoices++;
                    }
                    choiceDTOs.add(choice);
                }
                if (nbChoices > poll.getMaxChoiceNb()) {
                    voteForm.recordError(messages.format(("tooManyChoices"),
                            poll.getMaxChoiceNb()));
                    return pollZone.getBody();
                }
            } else if (poll.getVoteCounting() == VoteCountingType.PERCENTAGE) {
                int total = 0;
                for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                    total = total + choice.getValue();
                    choiceDTOs.add(choice);
                }
                if (total != 100) {
                    voteForm.recordError(messages.get("not100percent"));
                    return pollZone.getBody();
                }
            } else if (poll.getVoteCounting() == VoteCountingType.CONDORCET) {
                for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                    if (choice.getValue() == 0) {
                        choice.setValue(100);
                    }
                    choiceDTOs.add(choice);
                }
            }

            VoteDTO vote = new VoteDTO(null, poll.getId(), null);
            vote.setChoiceDTOs(choiceDTOs);
            vote.setWeight(pollAccount.getWeight());

            // mise à jour du vote ou création d'un nouveau vote
            if (alreadyVoted) {
                for (VoteDTO v : poll.getVoteDTOs()) {
                    PollAccountDTO voteAccount = servicePollAccount
                            .findPollAccountById(v.getPollAccountId());
                    if (voteAccount.getVotingId().equals(
                            pollAccount.getVotingId())) {
                        vote.setId(v.getId());
                        serviceVote.deleteVote(vote);
                        serviceVote.createVote(vote, pollAccount);
                    }
                }
            } else {
                serviceVote.createVote(vote, pollAccount);
            }

            updatePoll(poll.getPollId());
            if (poll.isContinuousResults()) {
                results = serviceResults.getAllResults(poll.getPollId());
            }

            sendMailNotification();
            //}
        }
        voteChoices.clear();
        return pollZone.getBody();
    }

    /** Méthode appelée à l'édition d'un vote. */
    Object onActionFromEditVote(String votingId) {
        pollAccount.setVotingId(votingId);
        return pollZone.getBody();
    }

    /** Initialisation du pollAccount et contrôle du nom. */
    private boolean initPollAccount() {

        //// Contrôle et définition du votingId
        alreadyVoted = false;
        boolean userIdMatching = false;
        boolean anonymousForbidden = false;
        boolean restrictedListsForbidden = false;

        // Contrôle de la présence du votant dans les listes de votants
        // du sondage (si le sondage n'est pas libre)
        if (poll.getPollType() != PollType.FREE) {
            restrictedListsForbidden = true;
            for (VotingListDTO list : poll.getVotingListDTOs()) {
                for (PollAccountDTO account : list.getPollAccountDTOs()) {
                    if (pollAccount.getVotingId().equals(account.getVotingId())) {
                        restrictedListsForbidden = false;
                        pollAccount = servicePollAccount
                                .findPollAccountById(account.getId());
                        pollAccount.setVotingListId(list.getId());
                        pollAccount.setWeight(account.getWeight());
                        logger.debug("Compte \"" + account.getVotingId()
                                + "\" présent dans la liste \""
                                + list.getName() + "\" (poids="
                                + account.getWeight() + ")");
                    }
                }
            }
            if (restrictedListsForbidden) {
                voteForm.recordError(nameField, messages.format(
                        "restrictedListsForbidden", pollAccount.getVotingId()));
            }
        }

        //// Définition de l'userId
        if (userExists) {
            pollAccount.setUserId(user.getId());
        } else {
            pollAccount.setUserId("");
        }

        // Contrôle si le votant n'a pas déjà voté et si le vote est anonyme
        if (isVoteAnonymous()) { // l'utilisateur à soumis un vote anonyme
            if (poll.isAnonymousVoteAllowed()) {
                pollAccount.setVotingId("anonymous"
                        + UUID.randomUUID().toString().replaceAll("-", ""));
            } else {
                anonymousForbidden = true;
                voteForm.recordError(nameField, messages
                        .get("anonymousForbidden"));
            }
        } else { // l'utilisateur à soumis un vote avec un nom
            alreadyVoted = hasAlreadyVoted(pollAccount.getVotingId());
            userIdMatching = isUserIdMatching(pollAccount.getVotingId());
            if (alreadyVoted && !userIdMatching) {
                voteForm.recordError(nameField, messages.format("alreadyVoted",
                        pollAccount.getVotingId()));
            }
        }

        logger.debug("alreadyVoted: " + alreadyVoted);
        logger.debug("userIdMatching: " + userIdMatching);
        logger.debug("anonymousForbidden: " + anonymousForbidden);
        logger.debug("restrictedListsForbidden: " + restrictedListsForbidden);
        return ((!alreadyVoted || userIdMatching) && !anonymousForbidden && !restrictedListsForbidden);
    }

    /**
     * Retourne vrai si le votant a déjà voté.
     *
     * @param votingId le votant a rechercher
     * @return vrai si le votant a déjà voté
     */
    private boolean hasAlreadyVoted(String votingId) {

        // comparaison du votant (pollAccount) avec les votants des votes existants
        for (VoteDTO vote : poll.getVoteDTOs()) {
            PollAccountDTO account = servicePollAccount
                    .findPollAccountById(vote.getPollAccountId());

            if (account.getVotingId().equals(votingId)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Retourne vrai si l'utilisateur connecté est l'auteur du vote.
     *
     * @param votingId le votant a rechercher
     * @return vrai si l'utilisateur est l'auteur du vote
     */
    public boolean isUserIdMatching(String votingId) {
        boolean userIdMatching = false;
        int i = 0;

        // parcours des votes pour trouver celui correspondant au votingId
        // et controle du userId
        while (i < poll.getVoteDTOs().size() && !userIdMatching) {

            // account : compte associé au vote courant
            String id = poll.getVoteDTOs().get(i).getPollAccountId();
            PollAccountDTO account = servicePollAccount.findPollAccountById(id);

            // si le votant du vote correspond au votingId
            if (account.getVotingId().equals(votingId)) {

                // si l'utilisateur du vote correspond au userId
                if (userExists && user.getId().length() > 0) {
                    userIdMatching = user.getId().equals(account.getUserId());
                }
            }

            i++;
        }

        return userIdMatching;
    }

    /** Envoi du mail de notification */
    private void sendMailNotification() {
        String voteURL = siteURL + "poll/VoteFor/" + poll.getPollId();
        String modifURL = siteURL + "poll/Modification/" + poll.getPollId()
                + ":" + MD5.encode(poll.getCreatorId());
        Map<String, String> data = new HashMap<String, String>();
        data.put("host", conf.getProperty("email_host"));
        data.put("port", conf.getProperty("email_port"));
        data.put("from", conf.getProperty("email_from"));

        // Mail au créateur
        data.put("to", poll.getCreatorEmail());
        data
                .put("title", messages.format("voteEmail_subject", poll
                        .getTitle()));
        data.put("msg", messages.format("voteEmail_msg", poll.getTitle(), poll
                .getVoteDTOs().size(), voteURL, modifURL));

        for (PreventRuleDTO rule : poll.getPreventRuleDTOs()) {
            PreventRuleManager manager = new PreventRuleManager(rule);
            manager.execute("vote", poll.getVoteDTOs().size(), data);
        }
    }

    public boolean isAddChoice() {
        return addChoice;
    }

    public void setAddChoice(boolean addChoice) {
        choiceOfVote.setValue((addChoice) ? 1 : 0);
    }

    /**
     * Retourne si le choix fait partie du vote (s'il a été renseigné par le
     * votant).
     *
     * @param choice le choix concerné
     * @return true si le choix est dans le vote
     */
    public Boolean isChoiceInVote(ChoiceDTO choice) {
        if (choice != null) {
            switch (poll.getVoteCounting()) {
            case NORMAL:
                return choice.getValue() > 0;
            case PERCENTAGE:
                return true;
            case CONDORCET:
                return choice.getValue() < 100;
            }
        }
        return null;
    }

    /**
     * Retourne le choix de vote correspondant au choix de sondage courant.
     *
     * @return un choix de vote
     */
    public ChoiceDTO getCurrentVoteChoice() {
        for (ChoiceDTO choice : vote.getChoiceDTOs()) {
            if (choice.getId().equals(choiceOfPoll.getId())) {
                return choice;
            }
        }
        return null;
    }

    /**
     * Retourne le résultat correspondant au choix de sondage courant.
     *
     * @return le résultat du choix
     */
    public String getCurrentChoiceResult() {
        String val = "";
        for (ResultDTO result : results) {
            if (result.getName().equals(choiceOfPoll.getName())) {
                val = result.getValue();

                // le résultat peut-être un double : 1,0 -> 1 et 1,2 -> 1,2
                if (val.endsWith(".0")) {
                    val = val.substring(0, val.indexOf('.'));
                }
            }
        }
        return val;
    }

    /**
     * Retourne true si le sondage est anonyme ou si le vote soumis est anonyme.
     * Utilisé lors de la soumission d'un vote.
     *
     * @return true si le vote est anonyme
     */
    public boolean isVoteAnonymous() {
        return poll.isAnonymous()
                || pollAccount.getVotingId() == null
                || pollAccount.getVotingId().equals("")
                || pollAccount.getVotingId().equalsIgnoreCase(
                        messages.get("anonymous"));
    }

    /**
     * Retourne true si le sondage est anonyme ou si le vote courant est
     * anonyme. Utilisé dans la boucle de parcours des votes.
     *
     * @return true si le vote est anonyme
     */
    public boolean isCurrentVoteAnonymous() {
        return poll.isAnonymous()
                || getCurrentVotingId().startsWith("anonymous");
    }

    /** Retourne l'identifiant du votant du vote courant */
    public String getCurrentVotingId() {
        String id = vote.getPollAccountId();
        String votingId = servicePollAccount.findPollAccountById(id)
                .getVotingId();
        return votingId;
    }

    /** Validation du champs de saisie du choix (texte). */
    void onValidateFromTextName(String value) throws ValidationException {
        validateNewChoice(value);
    }

    /** Validation du champs de saisie du choix (date). */
    void onValidateFromDateDTF(Date value) throws ValidationException {
        validateNewChoice(String.valueOf(value.getTime()));
    }

    /** Validation du champs de saisie du choix (image). */
    void onValidateFromImgFile(UploadedFile value) throws ValidationException {
        validateNewChoice(value.getFileName().replace(' ', '_'));
    }

    /** Validation du champs de saisie du choix. */
    private void validateNewChoice(String value) throws ValidationException {
        for (ChoiceDTO choice : poll.getChoiceDTOs()) {
            if (value.equals(choice.getName())) {
                throw new ValidationException(messages.format("choiceExists",
                        value));
            }
        }
    }

    Object onSuccessFromChoiceForm() {
        if (isTextType()) {
            if (newChoice.getName() != null) {
                newChoice.setValidate(true);
                newChoice.setPollId(poll.getId());
                poll.getChoiceDTOs().add(newChoice);
            }
        } else if (isDateType()) {
            if (newDateChoice.getDate() != null) {
                newDateChoice.setValidate(true);
                newDateChoice.setPollId(poll.getId());
                newDateChoice.setName(String.valueOf(newDateChoice.getDate()
                        .getTime()));
                poll.getChoiceDTOs().add(newDateChoice);
            }
        } else if (isImageType()) {
            if (newImageChoice.getImg() != null) {
                newImageChoice.setValidate(true);
                newImageChoice.setPollId(poll.getId());
                newImageChoice.setName(newImageChoice.getImg().getFileName()
                        .replace(' ', '_'));
                poll.getChoiceDTOs().add(newImageChoice);
                ImageUtil.saveImage(newImageChoice, new File(conf
                        .getProperty("upImgDir"), poll.getPollId()));
            }
        }

        servicePoll.updatePoll(poll);
        updatePoll(poll.getPollId());
        newChoice = new ChoiceDTO();
        newDateChoice = new DateChoiceUIO();
        newImageChoice = new ImageChoiceUIO();
        return this;
    }

    Object onSuccessFromCommentForm() {
        newComment.setPollId(poll.getId());
        newComment.setPostDate(new Date());
        serviceComment.createComment(newComment);
        updatePoll(poll.getPollId());
        newComment = new CommentDTO();
        return commentZone.getBody();
    }

    public boolean isPollNull() {
        return poll == null;
    }

    public boolean isPollChoiceStarted() {
        Date now = new Date();
        return poll.getBeginChoiceDate() == null
                || poll.getBeginChoiceDate().before(now);
    }

    public boolean isPollStarted() {
        Date now = new Date();
        return poll.getBeginDate() == null || poll.getBeginDate().before(now);
    }

    public boolean isPollFinished() {
        Date now = new Date();
        return poll.getEndDate() != null && poll.getEndDate().before(now);
    }

    public boolean isPollChoiceOrVoteStarted() {
        return isPollChoiceStarted() || isPollStarted();
    }

    public boolean isPollChoiceRunning() {
        return poll.isChoiceAddAllowed() && isPollChoiceStarted()
                && !isPollStarted();
    }

    public boolean isPollRunning() {
        return isPollStarted() && !isPollFinished() && !poll.isClosed();
    }

    public boolean isDescNull() {
        return choiceOfPoll.getDescription() == null;
    }

    /** Retourne vrai si le champs pollAccount doit apparaître. */
    public boolean isAccountFieldDisplayed() {
        return !poll.isAnonymous() || isRestrictedPoll() || isGroupPoll();
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public boolean isNormalVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.NORMAL;
    }

    public boolean isPercentageVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.PERCENTAGE;
    }

    public boolean isCondorcetVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.CONDORCET;
    }

    /**
     * Récupération du créateur du sondage.
     *
     * @return le nom du créateur du sondage.
     */
    public String getCreatorName() {
        PollAccountDTO pollAccount = servicePollAccount
                .findPollAccountById(poll.getCreatorId());
        return pollAccount.getVotingId();
    }

    /** Retourne le message d'aide correspondant au type de sondage. */
    public String getHelpMessage() {
        switch (poll.getVoteCounting()) {
        case NORMAL:
            return messages.get("normalVote-help");
        case PERCENTAGE:
            return messages.get("percentageVote-help");
        case CONDORCET:
            return messages.get("condorcetVote-help");
        default:
            return "";
        }
    }

    /** Retourne la date correspondant au choix courant */
    public Date getChoiceNameAsDate() {
        return new Date(Long.valueOf(choiceOfPoll.getName()));
    }

    /** Retourne le chemin de l'image correspondant au choix courant */
    public String getChoiceImageSrc() {
        File dir = new File(conf.getProperty("upImgDir"), poll.getPollId());
        File file = new File(dir, choiceOfPoll.getName());
        return file.getAbsolutePath();
    }

    /** Retourne le chemin de la miniature correspondant au choix courant */
    public String getChoiceThumbSrc() {
        File dir = new File(conf.getProperty("upImgDir"), poll.getPollId());
        File file = new File(dir, "thumb_" + choiceOfPoll.getName());
        return file.getAbsolutePath();
    }

    /** Action réalisée lorsqu'on clique sur l'image */
    Object onDisplayImage(String choiceId) {
        imageDisplay.setPoll(poll);
        imageDisplay.setChoiceId(choiceId);
        imageDisplay.setPageStyle("Vote");
        return imageDisplay;
    }

    /** Affichage de "" plutôt que 0 pour le Condorcet */
    String onToClientFromCondorcetInput() {
        if (choiceOfVote.getValue() == 0) {
            return "";
        }
        return null;
    }

    /** Récupération d'un entier plutôt que "" pour le Condorcet */
    /*Object onParseClientFromCondorcetInput(String input) {
        if ("".equals(input)) return 0; //ne fonctionne pas
        return null;
    }*/

    /**
     * Méthode appelée au moment de l'activation de la page
     *
     * @param id l'identifiant du sondage pour lequel vote l'utilisateur
     */
    void onActivate(String id) {
        pollAccount = new PollAccountDTO();
        if (userExists) {
            pollAccount.setVotingId(user.getFirstName() + " "
                    + user.getLastName());
        } else {
            pollAccount.setVotingId(yourName);
        }
        updatePoll(id);

        if (poll != null) {
            if (poll.isContinuousResults()) {
                results = serviceResults.getAllResults(poll.getPollId());
            }

            newChoice = new ChoiceDTO();
            newDateChoice = new DateChoiceUIO();
            newImageChoice = new ImageChoiceUIO();
            newComment = new CommentDTO();

            newComment.setPollId(poll.getId());
        }

        // Affichage des erreurs
        if (poll == null) {
            feedback.addError(messages.get("pollNotFound"));
        } else if (poll.isClosed()) {
            feedback.addInfo(messages.get("pollClosed"));
        } else if (!isPollStarted()) {
            feedback.addInfo(messages.get("pollNotStarted"));
        } else if (isPollFinished()) {
            feedback.addInfo(messages.get("pollFinished"));
        }
        if (isPollChoiceRunning()) {
            feedback.addInfo(messages.get("pollChoiceRunning"));
        }
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };
    }
}
