package org.chorem.pollen.entities.migration;

/*
 * #%L
 * Pollen :: Services
 * $Id: PollenMigrationCallbackV1_5.java 3712 2012-09-30 12:57:25Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5/pollen-services/src/main/java/org/chorem/pollen/entities/migration/PollenMigrationCallbackV1_5.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.opensymphony.xwork2.ActionContext;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenApplicationContext;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.entities.PollenDAOHelper;
import org.chorem.pollen.services.DefaultPollenServiceContext;
import org.chorem.pollen.services.PollenServiceContext;
import org.chorem.pollen.services.PollenServiceFactory;
import org.chorem.pollen.services.impl.VoteService;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaSQLQuery;
import org.nuiton.topia.migration.TopiaMigrationCallbackByClassNG;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * Migration for version {@code 1.4.5.2}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5.2
 */
public class PollenMigrationCallbackV1_5 extends TopiaMigrationCallbackByClassNG.MigrationCallBackForVersion {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(PollenMigrationCallbackV1_5.class);

    @Override
    public Version getVersion() {
        return VersionUtil.valueOf("1.5");
    }

    @Override
    protected void prepareMigrationScript(TopiaContextImplementor tx,
                                          List<String> queries,
                                          boolean showSql,
                                          boolean showProgression) throws TopiaException {

        PollenApplicationContext applicationContext = PollenApplicationContext.get(
                ActionContext.getContext());

        PollenServiceContext sContext = DefaultPollenServiceContext.newContext(
                Locale.getDefault(),
                tx,
                applicationContext.getConfiguration(),
                new PollenServiceFactory(),
                applicationContext.getVoteCountingFactory()
        );

        // clean pollAccount for votes
        // see http://chorem.org/issues/804
        removeDuplicateVoteWithSameUserAccount(tx, sContext, queries);

        // generate missing accountId
        // see http://chorem.org/issues/814
        generateMissingAccountId(tx, sContext, queries);

        // remove result from persistance
        // see http://chorem.org/issues/815
        removeResults(tx, queries);

        // remove hidden choices
        // see http://chorem.org/issues/816
        removeHiddenChoices(tx, queries);
    }

    private void removeResults(TopiaContextImplementor tx, List<String> queries) {
        queries.add("DROP TABLE result;");
        queries.add("ALTER TABLE poll DROP COLUMN resultUptodate");
    }

    private void removeHiddenChoices(TopiaContextImplementor tx, List<String> queries) {

        queries.add("DELETE FROM votetochoice WHERE choice in (SELECT topiaid FROM choice WHERE name LIKE 'HIDDEN_%');");
        queries.add("DELETE FROM choice WHERE name LIKE 'HIDDEN_%';");
    }

    private void removeDuplicateVoteWithSameUserAccount(TopiaContextImplementor tx,
                                                        PollenServiceContext sContext,
                                                        List<String> queries) throws TopiaException {

        // get all poll with several votes with same userAccount
        TopiaSQLQuery<String> getAllPollIdsQuery = new TopiaSQLQuery<String>() {

            @Override
            protected PreparedStatement prepareQuery(Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement(
                        "SELECT p.pollid FROM poll p WHERE " +
                        "(SELECT COUNT(pa) > COUNT(DISTINCT(pa.useraccount)) " +
                        "FROM vote v,pollaccount pa WHERE " +
                        "v.poll = p.topiaid AND v.pollaccount = pa.topiaid " +
                        "AND pa.useraccount IS NOT NULL);");
                return ps;
            }

            @Override
            protected String prepareResult(ResultSet set) throws SQLException {
                String pollId = set.getString(1);
                return pollId;
            }
        };

        List<String> pollIds = getAllPollIdsQuery.findMultipleResult(tx);

        PollDAO pollDAO = PollenDAOHelper.getPollDAO(tx);

        VoteService voteService = sContext.newService(VoteService.class);

        Multimap<Poll, PollAccount> accountToAnonymous = ArrayListMultimap.create();
        Multimap<Poll, Vote> voteToFixByPoll = ArrayListMultimap.create();

        long nbPolls = pollIds.size();
        int currentPollIndex = 0;

        for (String pollId : pollIds) {

            if (log.isInfoEnabled()) {
                log.info("Treat bad poll [" + (currentPollIndex++) + "/" +
                         nbPolls + "]" + pollId);
            }

            Poll poll = pollDAO.findByPollId(pollId);
            boolean pollAnonymous = poll.isAnonymous();

            List<Vote> votes = voteService.getAllVotes(poll);

            Multimap<UserAccount, Vote> voteByUserAccount = ArrayListMultimap.create();
            for (Vote vote : votes) {
                UserAccount useraccount = voteToUserAccount.apply(vote);
                if (useraccount != null) {
                    voteByUserAccount.put(useraccount, vote);
                }
            }

            for (UserAccount userAccount : voteByUserAccount.keySet()) {

                if (userAccount != null) {

                    Collection<Vote> votesForUser = voteByUserAccount.get(userAccount);
                    if (votesForUser.size() > 1) {

                        // ok found a userAccount
                        if (log.isWarnEnabled()) {
                            log.warn("Poll" + (pollAnonymous ? "(anonymous)" : "") + " [" + poll.getPollId() + "] with multi vote for the same userAccount : " +
                                     userAccount.getEmail());
                        }
                        if (pollAnonymous) {
                            for (Vote vote : votesForUser) {
                                accountToAnonymous.put(poll, vote.getPollAccount());
                            }
                        } else {
                            voteToFixByPoll.putAll(poll, votesForUser);
                        }
                    }
                }
            }
        }

        // treat anonymous vote
        Set<Poll> anonymousPollToFix = accountToAnonymous.keySet();

        if (CollectionUtils.isNotEmpty(anonymousPollToFix)) {
            if (log.isWarnEnabled()) {
                log.warn("There is " + anonymousPollToFix.size() + " anonymous " +
                         "poll to fix, will remove the userAccount from " +
                         "anonymous votes");
            }
            String request = "UPDATE pollaccount set useraccount = NULL WHERE topiaid='%s';";
            for (PollAccount account : accountToAnonymous.values()) {
                queries.add(String.format(request, account.getTopiaId()));
            }
        }

        // treat other votes
        Set<Poll> pollToFix = voteToFixByPoll.keySet();
        if (CollectionUtils.isNotEmpty(pollToFix)) {
            if (log.isWarnEnabled()) {
                log.warn("There is " + pollToFix.size() + " other polls " +
                         "to fix, will remove the userAccount from any votes");
            }
            String request = "UPDATE pollaccount set useraccount = NULL WHERE topiaid='%s';";
            for (Vote account : voteToFixByPoll.values()) {
                queries.add(String.format(request, account.getPollAccount().getTopiaId()));
            }
        }

    }

    private void generateMissingAccountId(TopiaContextImplementor tx,
                                          PollenServiceContext sContext,
                                          List<String> queries) throws TopiaException {
        // get all pollAccount with no accountId
        TopiaSQLQuery<String> sqlQuery = new TopiaSQLQuery<String>() {

            @Override
            protected PreparedStatement prepareQuery(Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement(
                        "SELECT p.topiaId FROM pollAccount p, vote v WHERE " +
                        "p.accountId = '' AND v.pollaccount = p.topiaid");
                return ps;
            }

            @Override
            protected String prepareResult(ResultSet set) throws SQLException {
                String pollAccountId = set.getString(1);
                return pollAccountId;
            }
        };

        List<String> accountIds = sqlQuery.findMultipleResult(tx);

        String request =
                "UPDATE pollaccount set accountId = '%s' WHERE topiaid='%s';";
        for (String id : accountIds) {
            queries.add(String.format(request, sContext.generateId(), id));
        }
    }

    final Function<Vote, UserAccount> voteToUserAccount = new Function<Vote, UserAccount>() {
        @Override
        public UserAccount apply(Vote input) {
            PollAccount pollAccount = input.getPollAccount();
            return pollAccount == null ? null : pollAccount.getUserAccount();
        }
    };

}