/*
 * #%L
 * Pollen :: Services
 * $Id: PollenNotifierWorker.java 3761 2012-12-01 12:34:54Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.2/pollen-services/src/main/java/org/chorem/pollen/services/PollenNotifierWorker.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services;

import com.google.common.base.Preconditions;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.services.impl.PollService;
import org.chorem.pollen.services.impl.PreventRuleService;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

import java.io.Closeable;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

/**
 * This service run as a daemon seek to send some notification from running poll
 * store in {@link PreventRule}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenNotifierWorker implements Runnable, Closeable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(PollenNotifierWorker.class);

    public static final int ONE_MINUTE = 60 * 1000;

    public static final int ONE_HOUR = 60 * ONE_MINUTE;

    /** Service context used to create other services. */
    protected final PollenServiceContext serviceContext;

    /** Timer gérant les threads. */
    private final Timer timer;

    /** Topia root context to pollen db. */
    private final TopiaContext rootContext;

    public PollenNotifierWorker(PollenServiceContext serviceContext,
                                TopiaContext rootContext) {
        Preconditions.checkNotNull(serviceContext);
        Preconditions.checkNotNull(rootContext);
        Preconditions.checkState(!rootContext.isClosed());
        this.rootContext = rootContext;
        this.serviceContext = serviceContext;
        timer = new Timer();
    }

    @Override
    public void run() {

        // starts it
        timer.schedule(new TimerTask() {

            @Override
            public void run() {
                try {
                    if (log.isDebugEnabled()) {
                        log.debug("New round of " + PollenNotifierWorker.this);
                    }
                    TopiaContext transaction = rootContext.beginTransaction();
                    try {
                        serviceContext.setTransaction(transaction);

                        PollService pollService =
                                serviceContext.newService(PollService.class);
                        PreventRuleService preventRuleService =
                                serviceContext.newService(PreventRuleService.class);

                        // Get running polls with filled end date
                        Date now = serviceContext.getCurrentTime();
                        List<Poll> polls =
                                pollService.getRunningPollsWithEndTime(now);

                        if (log.isDebugEnabled()) {
                            log.debug("Found " + polls.size() +
                                      " running polls with end date to scan");
                        }
                        for (Poll poll : polls) {

                            if (log.isDebugEnabled()) {
                                log.debug("Tests for poll " + poll.getTitle());
                            }
                            preventRuleService.onPollToRemind(poll, now);

                        }

                    } finally {

                        serviceContext.setTransaction(null);

                        transaction.closeContext();
                    }
                } catch (TopiaException e) {
                    throw new PollenTechnicalException("Topia problem", e);
                }
            }
        }, 0, 10 * ONE_MINUTE); // repeat at ten minutes
    }

    @Override
    public void close() throws IOException {
        timer.cancel();
    }

}
