/*
 * #%L
 * Pollen :: Services
 * $Id: UserService.java 3667 2012-08-29 15:28:07Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5/pollen-services/src/main/java/org/chorem/pollen/services/impl/UserService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.entities.PollenBinderHelper;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.InvalidEmailException;
import org.chorem.pollen.services.exceptions.UserEmailAlreadyUsedException;
import org.chorem.pollen.services.exceptions.UserInvalidPasswordException;
import org.chorem.pollen.services.exceptions.UserLoginAlreadyUsedException;
import org.chorem.pollen.services.exceptions.UserNotFoundException;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaFilterPagerUtil;
import org.nuiton.topia.persistence.util.TopiaEntityBinder;
import org.nuiton.util.StringUtil;
import org.nuiton.util.beans.Binder;

import java.util.List;

import static org.nuiton.i18n.I18n._;

public class UserService extends PollenServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(UserService.class);

    public UserAccount connect(String login, String password) throws UserNotFoundException, UserInvalidPasswordException {

        Preconditions.checkNotNull(login);
        Preconditions.checkNotNull(password);

        UserAccountDAO dao = getDAO(UserAccount.class);

        try {

            UserAccount user = dao.findByLogin(login);

            if (user == null) {
                throw new UserNotFoundException();
            }

            String encodedPassword = encodePassword(password);
            if (!encodedPassword.equals(user.getPassword())) {
                throw new UserInvalidPasswordException();
            }

            return user;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public UserAccount createUser(UserAccount user,
                                  boolean byAdmin) throws UserLoginAlreadyUsedException, UserEmailAlreadyUsedException {

        Preconditions.checkNotNull(user);

        String password = user.getPassword(); // le mot de passe en clair

        UserAccountDAO dao = getDAO(UserAccount.class);

        UserAccount userAccount;
        try {

            UserAccount userByLogin = dao.findByLogin(user.getLogin());
            if (userByLogin != null) {
                throw new UserLoginAlreadyUsedException();
            }
            UserAccount userByEmail = dao.findByEmail(user.getEmail());
            if (userByEmail != null) {
                throw new UserEmailAlreadyUsedException();
            }
            if (byAdmin) {

                // let's generate the new password
                password = generatePassword();
                user.setPassword(password);
            }

            String encodedPassword = encodePassword(password);
            userAccount = dao.create(
                    UserAccount.PROPERTY_LOGIN, user.getLogin(),
                    UserAccount.PROPERTY_PASSWORD, encodedPassword
            );
            user.setTopiaId(userAccount.getTopiaId());
            copyUserAccount(user, userAccount, true);

            commitTransaction("Could not create user");

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }

        // send a email to user
        EmailService emailService = newService(EmailService.class);
        emailService.onUserCreated(user);

//        URL url = serviceContext.getApplicationURL();
//
//        if (url != null && StringUtil.isEmail(user.getEmail())) {
//
//            Locale locale = getLocale();
//            String subject = l_(locale, "pollen.email.userRegister.subject",
//                                user.getLogin());
//            String content = l_(locale, "pollen.email.userRegister.content",
//                                user.getDisplayName(),
//                                user.getLogin(), password, url);
//
//            EmailService emailService = newService(EmailService.class);
//
//            emailService.sendEmail(
//                    user.getEmail(),
//                    subject,
//                    content);
//        }

        return userAccount;
    }

    public UserAccount updateUser(UserAccount user,
                                  String newPassword,
                                  boolean byAdmin) throws UserEmailAlreadyUsedException, UserInvalidPasswordException {

        Preconditions.checkNotNull(user);

        UserAccountDAO dao = getDAO(UserAccount.class);

        try {

            UserAccount userToUpdate = dao.findByTopiaId(user.getTopiaId());

            // Do not manage password for an admin update
            if (!byAdmin) {

                // Check original password
                String encodedPassword = encodePassword(user.getPassword());
                if (!encodedPassword.equals(userToUpdate.getPassword())) {
                    throw new UserInvalidPasswordException();
                }

                if (StringUtils.isNotEmpty(newPassword)) {

                    // encode new password and store it in user account
                    String newEncodedPassword = encodePassword(newPassword);
                    userToUpdate.setPassword(newEncodedPassword);
                }
            }

            // In case of email change, check if an other user has not already
            // the new email existing user found
            //FIXME-tchemit-2012-08_29 Voir http://chorem.org/issues/796
            String oldEmail = userToUpdate.getEmail();
            if (StringUtils.isNotBlank(user.getEmail()) &&
                ObjectUtils.notEqual(user.getEmail(), oldEmail)
                && dao.isUserExist(user)) {
                throw new UserEmailAlreadyUsedException();
            }

            //FIXME-tchemit-2012-08_29 Voir http://chorem.org/issues/796
            copyUserAccount(user, userToUpdate, false);

            commitTransaction("Could not update user");
            return userToUpdate;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    /**
     * This service is used to retrieve the userAccount associated to the given
     * {@code email}. If the user is found, an email will be send with a new
     * password. Otherwise an error will occured.
     *
     * @param email User email
     * @throws InvalidEmailException if the email is not valid
     * @throws UserNotFoundException if no user matches the given email
     */
    public void lostPassword(String email) throws UserNotFoundException, InvalidEmailException {

        Preconditions.checkNotNull(email);

        if (!StringUtil.isEmail(email)) {
            throw new InvalidEmailException();
        }

        UserAccount user;
        UserAccountDAO dao = getDAO(UserAccount.class);
        try {
            user = dao.findByEmail(email);
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }

        if (user == null) {
            throw new UserNotFoundException();
        }

        String newPassword = generatePassword();
        String encodedPassword = encodePassword(newPassword);

        user.setPassword(encodedPassword);

        // send the email
        EmailService emailService = newService(EmailService.class);
        emailService.onLostPassword(user, newPassword);

        commitTransaction("Can't update lost user password");
    }

    public UserAccount getNewUser() {
        UserAccountDAO dao = getDAO(UserAccount.class);
        UserAccount result = newInstance(dao);
        return result;
    }

    public void deleteUser(String id) throws UserNotFoundException {

        Preconditions.checkNotNull(id);

        UserAccountDAO dao = getDAO(UserAccount.class);

        try {
            UserAccount user = dao.findByTopiaId(id);

            if (user == null) {
                throw new UserNotFoundException();
            }
            dao.delete(user);

            commitTransaction("Could not delete user " + id);
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public List<UserAccount> getUsers(TopiaFilterPagerUtil.FilterPagerBean pager) {

        Preconditions.checkNotNull(pager);

        try {
            UserAccountDAO dao = getDAO(UserAccount.class);
            List<UserAccount> result = dao.findUsers(pager);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public int getNbUsers() {

        UserAccountDAO dao = getDAO(UserAccount.class);

        try {
            Long result = dao.count();
            return result.intValue();
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void createDefaultUsers() {

        UserAccount user = getNewUser();

        user.setAdministrator(true);
        String login = getConfiguration().getAdminLogin();
        user.setLogin(login);
        user.setEmail(getConfiguration().getAdminEmail());
        user.setPassword(getConfiguration().getAdminPassword());
        try {
            createUser(user, false);
            if (log.isInfoEnabled()) {
                log.info(_("pollen.info.admin.created", login));
            }
        } catch (Exception eee) {
            if (log.isInfoEnabled()) {
                log.info(_("pollen.info.admin.exists") + " : " +
                         _(eee.getMessage()));
            }
        }
    }

    protected String generatePassword() {
        return RandomStringUtils.randomAlphanumeric(8);
    }

    protected String encodePassword(String password) {
        return StringUtil.encodeMD5(password);
    }

    /**
     * Copy {@code source} user account to {@code destination} one.
     * The email is lower cased in the {@code destination} user account.
     *
     * @param source      user account to copy
     * @param destination which receive the copy
     * @see Binder#copy(Object, Object, String...)
     */
    protected void copyUserAccount(UserAccount source,
                                   UserAccount destination,
                                   boolean copyEmail) {
        TopiaEntityBinder<UserAccount> binder =
                PollenBinderHelper.getSimpleTopiaBinder(UserAccount.class);


        binder.copy(source, destination,
                    UserAccount.PROPERTY_ADMINISTRATOR,
                    UserAccount.PROPERTY_FIRST_NAME,
                    UserAccount.PROPERTY_LAST_NAME);

        if (copyEmail) {
            // Don't keep case for email
            destination.setEmail(StringUtils.lowerCase(source.getEmail()));
        }
    }

}
