/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.migration;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.dialect.Dialect;
import org.hibernate.dialect.H2Dialect;
import org.hibernate.dialect.PostgreSQLDialect;
import org.nuiton.topia.migration.DatabaseManager;
import org.nuiton.topia.migration.callback.MigrationCallbackHandler;
import org.nuiton.util.VersionUtil;

/**
 * Classe d'interaction avant la migration (MigrationCallbackHandler).
 *
 * @author rannou
 * @version $Id: PollenMigrationCallbackHandler.java 2615 2009-07-01 13:47:45Z
 *          nrannou $
 */
public class PollenMigrationCallbackHandler implements MigrationCallbackHandler {

    /** log. */
    private static final Log log = LogFactory
            .getLog(PollenMigrationCallbackHandler.class);

    @Override
    public MigrationChoice doMigration(DatabaseManager dbManager,
            String dbVersion, String applicationVersion) {

        if (log.isInfoEnabled()) {
            log.info("Migration : version de la base : " + dbVersion
                    + ", version de l'application : " + applicationVersion);
        }

        Dialect dialect = dbManager.getDbConfiguration().buildSettings()
                .getDialect();
        MigrationChoice result = MigrationChoice.NO_MIGRATION;

        // ouverture d'une connexion direct JDBC sur la base
        try {
            Connection conn = dbManager.getConnection();
            try {
                conn.setAutoCommit(false);
                if (VersionUtil.valueOf(dbVersion).before(
                        VersionUtil.valueOf("1.1"))) {
                    migrateTo1_1(conn, dialect);
                }
                conn.commit();
                result = MigrationChoice.CUSTOM_MIGRATION;
            } catch (Exception eee) {
                log.error("Migration impossible de la base", eee);
                conn.rollback();
            } finally {
                conn.close();
            }
        } catch (Exception eee) {
            log.error("Erreur lors de la tentative de migration", eee);
        }
        return result;
    }

    /** Migration de la version 0 à 1.1 */
    private void migrateTo1_1(Connection conn, Dialect dialect)
            throws SQLException {
        log.info("Migrate to version 1_1");
        String[] sqls = null;

        // Script de migration
        if (dialect instanceof PostgreSQLDialect) {
            sqls = new String[] {
                    "alter table useraccount rename langage to language;",
                    "alter table pollaccount add accountid varchar;",
                    "update pollaccount set accountid=md5(topiaid);",
                    "alter table persontolist drop column hasvoted;",
                    "alter table persontolist add hasvoted boolean default false;",
                    "alter table vote add anonymous boolean default false;",
                    "alter table preventrule add active boolean default true;",
                    "alter table preventrule add onetime boolean default false;", };
        } else if (dialect instanceof H2Dialect) {
            sqls = new String[] {
                    "alter table useraccount alter column langage rename to language;",
                    "alter table pollaccount add accountid varchar;",
                    "update pollaccount set accountid=hash('sha256', stringtoutf8(topiaid), 1000);",
                    "alter table persontolist drop column hasvoted;",
                    "alter table persontolist add hasvoted boolean default false;",
                    "alter table vote add anonymous boolean default false;",
                    "alter table preventrule add active boolean default true;",
                    "alter table preventrule add onetime boolean default false;", };
        } else {
            if (log.isErrorEnabled()) {
                log
                        .error("Migration non prise en charge pour ce type de dialect : "
                                + dialect);
            }
        }

        // Exécution de la migration
        for (String sql : sqls) {
            log.info("try " + sql);
            PreparedStatement sta = conn.prepareStatement(sql);
            sta.executeUpdate();
            log.info("done " + sql);
        }
    }
}
