package io.quarkus.registry;

import java.util.Objects;

/**
 * Provides a low-level API for registering and accessing runtime values generated by Quarkus or Quarkus
 * extensions. Typically, such values describe a fully configured application, known only at application startup,
 * such as the HTTP port. Once a value is added to the {@link ValueRegistry}, it can't be changed.
 * <p>
 * These values should not be exposed by the Config mechanism directly, as that would require mutating the Config
 * system, which should be immutable, and cause hard-to-debug issues. Instead, {@link ValueRegistry} should be
 * preferred for exposing such values.
 * <p>
 * The {@link ValueRegistry} should also be the preferred solution when a service requires information from another
 * service, to avoid dependencies cycles and exposure of internal APIs.
 * <p>
 * While primarily targeting Quarkus and Quarkus Extensions' internal uses, general Quarkus Applications can also use
 * The {@link ValueRegistry}.
 */
public interface ValueRegistry {
    /**
     * Registers a value with a specified key.
     *
     * @param key key with which the specified value is to be associated
     * @param value value to be associated with the specified key
     * @throws IllegalArgumentException if the specified key already has an associated value
     */
    <T> void register(RuntimeKey<T> key, T value);

    /**
     * Registers a {@link RuntimeInfo} with a specified key.
     *
     * @param key key with which the specified value is to be associated
     * @param runtimeInfo value to be associated with the specified key
     * @throws IllegalArgumentException if the specified key already has an associated value
     * @see ValueRegistry.RuntimeInfo
     */
    <T> void registerInfo(RuntimeKey<T> key, RuntimeInfo<T> runtimeInfo);

    /**
     * Returns the value to which the specified key is mapped.
     *
     * @param key the key whose associated value is to be returned
     * @return the value to which the specified key is mapped
     * @throws java.lang.IllegalArgumentException if the specified key has no associated value
     */
    <T> T get(RuntimeKey<T> key);

    /**
     * Returns the value to which the specified key is mapped.
     *
     * @param key the key whose associated value is to be returned
     * @param defaultValue the default mapping of the key
     * @return the value to which the specified key is mapped, or {@code defaultValue} if the key has no value
     */
    <T> T getOrDefault(RuntimeKey<T> key, T defaultValue);

    /**
     * Returns {@code true} if {@link ValueRegistry} contains a mapping for the specified key.
     *
     * @param key key whose presence in {@link ValueRegistry} is to be tested
     * @return {@code true} if {@link ValueRegistry} contains a mapping for the specified key
     */
    <T> boolean containsKey(RuntimeKey<T> key);

    /**
     * Allows to defer and retrieve instances of an object that represent information only available at
     * runtime with {@link ValueRegistry}.
     */
    interface RuntimeInfo<T> {
        /**
         * Construct a new instance of {@code T}.
         *
         * @param valueRegistry which carries information to construct a new instance of {@code T}.
         * @return an instance of {@code T}.
         */
        T get(ValueRegistry valueRegistry);

        class SimpleRuntimeInfo<T> implements RuntimeInfo<T> {
            private final T value;

            SimpleRuntimeInfo(T value) {
                this.value = value;
            }

            @Override
            public T get(ValueRegistry valueRegistry) {
                return value;
            }

            public static <T> SimpleRuntimeInfo<T> of(final T value) {
                return new SimpleRuntimeInfo<>(value);
            }
        }
    }

    /**
     * A typed key.
     */
    final class RuntimeKey<T> {
        private final String key;
        private final Class<T> type;

        @SuppressWarnings("unchecked")
        RuntimeKey(String key) {
            this(key, (Class<T>) String.class);
        }

        RuntimeKey(String key, Class<T> type) {
            this.key = key;
            this.type = type;
        }

        public String key() {
            return key;
        }

        public Class<T> type() {
            return type;
        }

        @Override
        public boolean equals(Object o) {
            if (!(o instanceof RuntimeKey<?> that))
                return false;
            return Objects.equals(key, that.key);
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(key);
        }

        public static <T> RuntimeKey<T> key(final String key) {
            return new RuntimeKey<>(key);
        }

        public static <T> RuntimeKey<T> key(final String key, final Class<T> type) {
            return new RuntimeKey<>(key, type);
        }

        public static <T> RuntimeKey<T> key(final Class<T> type) {
            return new RuntimeKey<>(type.getName(), type);
        }

        public static RuntimeKey<Integer> intKey(final String key) {
            return new RuntimeKey<>(key, Integer.class);
        }

        public static RuntimeKey<Boolean> booleanKey(final String key) {
            return new RuntimeKey<>(key, Boolean.class);
        }

        // Add static constructors as needed
    }

    /**
     * Returns the value to which the specified key is mapped.
     * <p>
     * To avoid allocating a {@link RuntimeKey} for the Config bridge look up.
     *
     * @param key the key whose associated value is to be returned
     * @return the value to which the specified key is mapped
     * @see "io.quarkus.runtime.ValueRegistryConfigSource"
     */
    @Deprecated(forRemoval = true)
    RuntimeInfo<?> get(String key);
}
