package fr.ifremer.tutti.ui.swing.util.editor;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiComputedOrNotDataTableCell.java 694 2013-03-27 21:56:27Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.4/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/editor/TuttiComputedOrNotDataTableCell.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.ui.swing.util.TuttiComputedOrNotData;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.JAXXUtil;

import javax.swing.AbstractCellEditor;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.text.DecimalFormat;

/**
 * Editor for TuttiComputedOrNotData
 *
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0
 */
public class TuttiComputedOrNotDataTableCell extends DefaultTableCellRenderer {

    public static TableCellRenderer newRender(TableCellRenderer renderer,
                                              boolean useFloat,
                                              Integer decimalNumber,
                                              Color computedDataColor) {

        return new TuttiComputedOrNotDataTableCellRenderer(renderer,
                                                           useFloat,
                                                           decimalNumber,
                                                           computedDataColor);
    }

    public static TableCellEditor newEditor(Class type,
                                            boolean useSign,
                                            boolean useFloat,
                                            Integer decimalNumber,
                                            Color computedDataColor) {

        return new TuttiComputedOrNotDataTableCellEditor(type, useSign,
                                                         useFloat, decimalNumber,
                                                         computedDataColor);
    }

    public static class TuttiComputedOrNotDataTableCellEditor
            extends AbstractCellEditor
            implements TableCellEditor, FocusListener, AncestorListener {

        private static final long serialVersionUID = 1L;

        protected final TuttiComputedOrNotDataEditor numberEditor;

        protected TuttiComputedOrNotData data;

        /** constructor */
        public TuttiComputedOrNotDataTableCellEditor(Class type,
                                                     boolean useSign,
                                                     boolean useFloat,
                                                     Integer decimalNumber,
                                                     Color computedDataColor) {

            numberEditor = new TuttiComputedOrNotDataEditor();
            numberEditor.setComputedDataColor(computedDataColor);
            numberEditor.getTextField().setHorizontalAlignment(SwingConstants.RIGHT);
            numberEditor.getTextField().addFocusListener(this);
            numberEditor.getTextField().addAncestorListener(this);
            numberEditor.getTextField().setBorder(new LineBorder(Color.GRAY, 2));
            numberEditor.setSelectAllTextOnError(true);
            numberEditor.setNumberPattern(TuttiUI.DECIMAL3_PATTERN);

            numberEditor.setModelType(type);
            numberEditor.setUseSign(useSign);
            numberEditor.setUseFloat(useFloat);
            numberEditor.setDecimalNumber(decimalNumber);
            numberEditor.init();
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value,
                                                     boolean isSelected, int row, int column) {

            data = (TuttiComputedOrNotData) value;
            numberEditor.setModel(data.getData());

            // Check nullity and set the text that will be selected with the current value
            if (data.getData() != null) {
                numberEditor.setModelText(String.valueOf(data.getData()));
            }
            return numberEditor;
        }

        public TuttiComputedOrNotDataEditor getNumberEditor() {
            return numberEditor;
        }

        @Override
        public TuttiComputedOrNotData getCellEditorValue() {
            return data;
        }

        @Override
        public void focusGained(FocusEvent e) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    numberEditor.getTextField().requestFocus();
                    numberEditor.getTextField().selectAll();
                }
            });
        }

        @Override
        public void focusLost(FocusEvent e) {
        }

        @Override
        public void ancestorAdded(AncestorEvent event) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    numberEditor.getTextField().requestFocus();
                    numberEditor.getTextField().selectAll();
                }
            });
        }

        @Override
        public void ancestorRemoved(AncestorEvent event) {
        }

        @Override
        public void ancestorMoved(AncestorEvent event) {
        }

        @Override
        public boolean stopCellEditing() {
            boolean result = super.stopCellEditing();
            // Reset previous data to avoid keeping it on other cell edition
            if (result) {
                data.setData(numberEditor.getModel());

                numberEditor.setBean(null);

                data = null;
            }
            return result;
        }
    }

    public static class TuttiComputedOrNotDataTableCellRenderer<E>
            implements TableCellRenderer {

        protected final TableCellRenderer delegate;

        protected Integer decimalNumber;

        protected Color computedDataColor;

        protected boolean useFloat;

        public TuttiComputedOrNotDataTableCellRenderer(
                TableCellRenderer delegate,
                boolean useFloat,
                Integer decimalNumber,
                Color computedDataColor) {

            this.delegate = delegate;
            this.useFloat = useFloat;
            this.decimalNumber = decimalNumber;
            this.computedDataColor = computedDataColor;
        }

        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row,
                                                       int column) {

            TuttiComputedOrNotData data = (TuttiComputedOrNotData) value;
            Number dataValue = data.getData();
            Font font;
            Color foreground;
            String text;
            if (dataValue == null) {
                dataValue = data.getComputedData();
                font = TuttiUI.TEXTFIELD_COMPUTED_FONT;
                foreground = computedDataColor;

                if (useFloat && decimalNumber != null && dataValue != null) {
                    DecimalFormat decimalFormat = TuttiUIUtil.getDecimalFormat(1, decimalNumber);
                    text = JAXXUtil.getStringValue(decimalFormat.format(dataValue));
                } else {
                    text = JAXXUtil.getStringValue(dataValue);
                }

            } else {
                font = TuttiUI.TEXTFIELD_NORMAL_FONT;
                foreground = Color.BLACK;
                text = JAXXUtil.getStringValue(dataValue);
            }

            Component component = delegate.getTableCellRendererComponent(table,
                                                                         text,
                                                                         isSelected,
                                                                         hasFocus,
                                                                         row,
                                                                         column);
            component.setFont(font);
            component.setForeground(foreground);

            return component;
        }

    }
}
