package fr.ifremer.tutti.ui.swing.content.operation.catches.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesBatchUIHandler.java 581 2013-03-12 09:10:25Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/SpeciesBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.BatchContainer;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatchFrequency;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.FrequencyCellComponent;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.CreateSpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.CreateSpeciesBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.editor.LongTextCellComponent;
import fr.ifremer.tutti.ui.swing.util.editor.TuttiComputedOrNotDataTableCell;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.JOptionPane;
import javax.swing.RowFilter;
import javax.swing.UIManager;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class SpeciesBatchUIHandler extends AbstractTuttiBatchTableUIHandler<SpeciesBatchRowModel, SpeciesBatchUIModel, SpeciesBatchUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SpeciesBatchUIHandler.class);

    public static JAXXContextEntryDef<List<Caracteristic>> FREQUENCY_LENGTH_CONTEXT_ENTRY =
            JAXXUtil.newListContextEntryDef("frequencyLength");

    public static JAXXContextEntryDef<List<Species>> SPECIES_REFERENT_CONTEXT_ENTRY =
            JAXXUtil.newListContextEntryDef("speciesReferent");

    public static final Set<String> SAMPLING_PROPERTIES = Sets.newHashSet(
            SpeciesBatchRowModel.PROPERTY_SAMPLE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SPECIES,
            SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT);

    private final EnumMap<TableViewMode, RowFilter<SpeciesBatchTableModel, Integer>> tableFilters;

    public SpeciesBatchUIHandler(TuttiUI<?, ?> parentUi,
                                 SpeciesBatchUI ui) {
        super(parentUi, ui,
              SpeciesBatchRowModel.PROPERTY_SPECIES,
              SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_NUMBER,
              SpeciesBatchRowModel.PROPERTY_COMMENT,
              SpeciesBatchRowModel.PROPERTY_ATTACHMENT,
              SpeciesBatchRowModel.PROPERTY_FREQUENCY,
              SpeciesBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM);
        tableFilters = new EnumMap<TableViewMode, RowFilter<SpeciesBatchTableModel, Integer>>(TableViewMode.class);

        tableFilters.put(TableViewMode.ALL, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                return true;
            }
        });

        tableFilters.put(TableViewMode.ROOT, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                boolean result = false;
                Integer rowIndex = entry.getIdentifier();
                if (rowIndex != null) {
                    SpeciesBatchTableModel model = entry.getModel();
                    SpeciesBatchRowModel row = model.getEntry(rowIndex);
                    result = row != null && row.isBatchRoot();
                }
                return result;
            }
        });

        tableFilters.put(TableViewMode.LEAF, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                boolean result = false;
                Integer rowIndex = entry.getIdentifier();
                if (rowIndex != null) {
                    SpeciesBatchTableModel model = entry.getModel();
                    SpeciesBatchRowModel row = model.getEntry(rowIndex);
                    result = row != null && row.isBatchLeaf();
                }
                return result;
            }
        });
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        SpeciesBatchUIModel model = getModel();

        List<SpeciesBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isInfoEnabled()) {
                log.info("Get species batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            model.removeAllAttachment(model.getAttachment());

            if (!TuttiEntities.isNew(bean)) {

                // get all batch species root (says the one with only a species sample category)
                BatchContainer<SpeciesBatch> rootSpeciesBatch =
                        persistenceService.getRootSpeciesBatch(bean.getId());

                model.setRootBatchId(rootSpeciesBatch.getId());

                List<Attachment> attachments =
                        persistenceService.getAllAttachments(Integer.valueOf(model.getObjectId()));
                model.addAllAttachment(attachments);

                if (log.isInfoEnabled()) {
                    log.info("species root batch id: " + model.getRootBatchId());
                }

                List<SpeciesBatch> catches = rootSpeciesBatch.getChildren();

                for (SpeciesBatch aBatch : catches) {

                    // root batch sample categroy is species
                    Preconditions.checkState(
                            aBatch.getSampleCategoryType() == SampleCategoryEnum.sortedUnsorted,
                            "Root species batch must be a sortedUnsorted sample " +
                            "category but was:" + aBatch.getSampleCategoryType());

                    SpeciesBatchRowModel rootRow =
                            loadSpeciesBatch(aBatch, null, rows);

                    if (log.isDebugEnabled()) {
                        log.debug("Loaded root batch " +
                                  decorate(rootRow.getSpecies()) + " - " +
                                  decorate(rootRow.getSortedUnsortedCategoryValue()));
                    }
                }
            }
        }

        model.setRows(rows);
        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public SpeciesBatchTableModel getTableModel() {
        return (SpeciesBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SpeciesBatchRowModel row) {

        // a row is valid if species category is not empty and valid
        // then if any of none empty category is valid
        boolean result = row.getSpecies() != null;
//                &&!row.getSortedUnsortedCategory().isEmpty() &&
//                         row.getSortedUnsortedCategory().isValid();

//        result &= row.getSizeCategory().isEmptyOrValid();
//        result &= row.getSexCategory().isEmptyOrValid();
//        result &= row.getMaturityCategory().isEmptyOrValid();
//        result &= row.getAgeCategory().isEmptyOrValid();
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SpeciesBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        if (SAMPLING_PROPERTIES.contains(propertyName)) {

            // species has changed, recompute valid property
            recomputeRowValidState(row);

            // recompute the totalUnsorted weight
//            recomputeTotalUnsortedWeight();
        }

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SpeciesBatchRowModel> rowMonitor,
                                             SpeciesBatchRowModel row) {
        if (row.isValid()) {
            // there is a valid bean attached to the monitor
            if (rowMonitor.wasModified()) {

                // monitored bean was modified, save it
                if (log.isInfoEnabled()) {
                    log.info("Row " + row + " was modified, will save it");
                }

                showInformationMessage(
                        "[ Captures - Espèces ] " +
                        "Sauvegarde des modifications de " + row + '.');

                saveRow(row);

                // clear modified flag on the monitor
                rowMonitor.clearModified();
            }
        } else {

            //FIXME See how to delete rows ? Or moreover how to save tehem...
            if (log.isWarnEnabled()) {
                log.warn("Will not remove not valid speciesBatch: " + row.getId());
            }

//            // row is not valid can not save it
//
//            SpeciesBatch catchBean = row.toBean();
//
//            if (!TuttiEntities.isNew(catchBean)) {
//
//                // remove this
//                persistenceService.deleteSpeciesBatch(catchBean.getId());
//                row.setId(null);
//            }
        }
    }

    @Override
    protected void onModelRowsChanged(List<SpeciesBatchRowModel> rows) {
        super.onModelRowsChanged(rows);

        // clear speciesUsed
        SpeciesBatchUIModel model = getModel();
        model.getSpeciesUsed().clear();
        model.setRootNumber(0);

        for (SpeciesBatchRowModel row : rows) {
            updateTotalFromFrequencies(row);
            if (row.isBatchRoot()) {

                // update speciesUsed
                addToSpeciesUsed(row);
            }
        }
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          SpeciesBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             SpeciesBatchRowModel oldRow,
                                             int newRowIndex,
                                             SpeciesBatchRowModel newRow) {
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);

        // when selected row has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<SpeciesBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("beforeInit: " + ui);
        }

        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);

        SpeciesBatchUIModel model = new SpeciesBatchUIModel(catchesUIModel);
        model.setTableViewMode(TableViewMode.ALL);
        ui.setContextValue(model);

        List<Caracteristic> lengthCaracterics;

        TuttiProtocol protocol;

        if (context.isProtocolFilled()) {

            // get loaded protocol

            protocol = getDataContext().getProtocol();

            lengthCaracterics = Lists.newArrayListWithCapacity(
                    protocol.sizeLengthClassesPmfmId());

            Map<String, Caracteristic> allCaractericsById =
                    TuttiEntities.splitById(getDataContext().getCaracteristics());

            if (!protocol.isLengthClassesPmfmIdEmpty()) {
                for (String id : protocol.getLengthClassesPmfmId()) {
                    lengthCaracterics.add(allCaractericsById.get(id));
                }
            }

        } else {

            // use all caracteristics
            lengthCaracterics = Lists.newArrayList(
                    getDataContext().getCaracteristics());
        }

        // use only the referent here, if the user needs a synonym, he will
        // click on the button to select another species
        List<Species> speciesList = getDataContext().getReferentSpecies();

        SPECIES_REFERENT_CONTEXT_ENTRY.setContextValue(ui, speciesList);
        FREQUENCY_LENGTH_CONTEXT_ENTRY.setContextValue(ui, lengthCaracterics);
    }

    @Override
    public void afterInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("afterInit: " + ui);
        }

        initUI(ui);

        List<SampleCategoryType> samplingOrder;

        List<Species> speciesUniverse =
                SPECIES_REFERENT_CONTEXT_ENTRY.getContextValue(ui);

        List<Species> allSpecies;

        Multimap<Species, SampleCategoryType> speciesSampleCategories =
                HashMultimap.create();

        boolean protocolFilled = context.isProtocolFilled();
        if (protocolFilled) {

            // get loaded protocol

            TuttiProtocol protocol = getDataContext().getProtocol();
            Preconditions.checkNotNull(protocol,
                                       "Could not find protocol in ui context");

            // fill sampling order from protocol

            samplingOrder = Lists.newArrayList(
                    SampleCategoryType.sortedUnsorted,
                    SampleCategoryType.size,
                    SampleCategoryType.sex,
                    SampleCategoryType.maturity,
                    SampleCategoryType.age);

            // fill available species from protocol

            allSpecies = Lists.newArrayList();
            if (!protocol.isSpeciesEmpty()) {

                // split by taxonId
                Map<String, Species> map = TuttiEntities.splitByTaxonId(speciesUniverse);

                for (SpeciesProtocol protocolSpecy : protocol.getSpecies()) {
                    String taxonId = String.valueOf(protocolSpecy.getSpeciesReferenceTaxonId());
                    Species species = map.get(taxonId);
                    allSpecies.add(species);

                    speciesSampleCategories.put(species, SampleCategoryType.sortedUnsorted);
                    speciesSampleCategories.put(species, SampleCategoryType.size);
                    speciesSampleCategories.put(species, SampleCategoryType.sex);
                    speciesSampleCategories.put(species, SampleCategoryType.maturity);
                    speciesSampleCategories.put(species, SampleCategoryType.age);
                }
            }

        } else {

            // no protocol, use default values

            samplingOrder = Lists.newArrayList(
                    SampleCategoryType.sortedUnsorted,
                    SampleCategoryType.size,
                    SampleCategoryType.sex,
                    SampleCategoryType.maturity,
                    SampleCategoryType.age);

            allSpecies = Lists.newArrayList(speciesUniverse);

            // each species can use any category
            for (Species species : allSpecies) {
                speciesSampleCategories.put(species, SampleCategoryType.sortedUnsorted);
                speciesSampleCategories.put(species, SampleCategoryType.size);
                speciesSampleCategories.put(species, SampleCategoryType.sex);
                speciesSampleCategories.put(species, SampleCategoryType.maturity);
                speciesSampleCategories.put(species, SampleCategoryType.age);
            }
        }

        if (log.isInfoEnabled()) {
            log.info("Will use sampling order: " + samplingOrder);
            log.info("Will use " + allSpecies.size() + " species.");
        }

        getModel().setSamplingOrder(samplingOrder);
        getModel().setAllSpecies(allSpecies);

        JXTable table = getTable();

        // can show / hide some columns in model
        table.setColumnControlVisible(true);

        // create table column model
        TableCellRenderer defaultRenderer =
                table.getDefaultRenderer(Object.class);

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        Decorator<CaracteristicQualitativeValue> caracteristicDecorator =
                getDecorator(CaracteristicQualitativeValue.class, null);

        Color computedDataColor = getConfig().getColorComputedWeights();

        { // Species column

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Species.class),
                             SpeciesBatchTableModel.SPECIES);
        }

        { // SortedUnsortedCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.SORTED_UNSORTED_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        { // SizeCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.SIZE_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        { // SexCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.SEX_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        { // MaturityCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.MATURITY_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        { // AgeCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.AGE_CATEGORY,
                                           getDecorator(Float.class, null),
                                           defaultRenderer);
        }

        { // Weight column

            addColumnToModel(columnModel,
                             TuttiComputedOrNotDataTableCell.newEditor(
                                     Float.class, false, true, 3, computedDataColor),
                             TuttiComputedOrNotDataTableCell.newRender(
                                     defaultRenderer, true, 3, computedDataColor),
                             SpeciesBatchTableModel.WEIGHT);
        }

        { // Number column (from frequencies)

            addColumnToModel(columnModel,
                             FrequencyCellComponent.newEditor(ui, computedDataColor),
                             FrequencyCellComponent.newRender(computedDataColor),
                             SpeciesBatchTableModel.COMPUTED_NUMBER);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             LongTextCellComponent.newEditor(ui.getLongTextEditor()),
                             LongTextCellComponent.newRender(n_("tutti.tooltip.comment.none")),
                             SpeciesBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui
                             ),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             SpeciesBatchTableModel.ATTACHMENT);
        }

        { // Species to confirm column

            addBooleanColumnToModel(columnModel,
                                    SpeciesBatchTableModel.SPECIES_TO_CONFIRM,
                                    getTable());
        }

        // create table model
        SpeciesBatchTableModel tableModel =
                new SpeciesBatchTableModel(columnModel, speciesSampleCategories);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        Color toConfirmColor = getConfig().getColorRowToConfirm();
        // paint the cell in orange if the row is to confirm
        Highlighter attachmentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate() {

                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                        SpeciesBatchRowModel row = getTableModel().getEntry(adapter.row);
                        return row.getSpeciesToConfirm();
                    }

                }, toConfirmColor);
        table.addHighlighter(attachmentHighlighter);

        initBatchTable(table, columnModel, tableModel);

        // highlight only the species column if the row is selected
        Highlighter selectedHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.IS_SELECTED,
                        new HighlightPredicate.IdentifierHighlightPredicate(SpeciesBatchTableModel.SPECIES)),
                UIManager.getColor("Table[Enabled+Selected].textBackground"));

        table.addHighlighter(selectedHighlighter);

        // paint the cell in dark orange if the row is to confirm and the cell is not editable
        attachmentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate() {

                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                        SpeciesBatchRowModel row = getTableModel().getEntry(adapter.row);
                        return row.getSpeciesToConfirm() && !adapter.isEditable();
                    }

                }, toConfirmColor.darker());
        table.addHighlighter(attachmentHighlighter);

        getModel().addPropertyChangeListener(SpeciesBatchUIModel.PROPERTY_TABLE_VIEW_MODE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                TableViewMode tableViewMode = (TableViewMode) evt.getNewValue();

                if (tableViewMode == null) {
                    tableViewMode = TableViewMode.ALL;
                }

                if (log.isDebugEnabled()) {
                    log.debug("Will use rowfilter for viewMode: " + tableViewMode);
                }
                RowFilter<SpeciesBatchTableModel, Integer> filter = tableFilters.get(tableViewMode);
                getTable().setRowFilter(filter);
            }
        });

    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("Closing: " + ui);
        }
        ui.getSpeciesBatchAttachmentsButton().onCloseUI();
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void createSpeciesBatch() {

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        CreateSpeciesBatchUI createBatchEditor = parent.getSpeciesTabCreateBatch();

        createBatchEditor.getHandler().openUI(getModel());
        parent.getHandler().setSpeciesSelectedCard(EditCatchesUIHandler.CREATE_BATCH_CARD);
    }

    public void addSpeciesBatch(CreateSpeciesBatchUIModel createModel) {
        if (createModel.isValid()) {

            SpeciesBatchTableModel tableModel = getTableModel();

            SpeciesBatchRowModel newRow = tableModel.createNewRow();
            Species species = createModel.getSpecies();
            newRow.setSpecies(species);

            CaracteristicQualitativeValue sortedUnsortedCategory = createModel.getSortedUnsortedCategory();
            SampleCategory<CaracteristicQualitativeValue> category = newRow.getSortedUnsortedCategory();
            category.setCategoryValue(sortedUnsortedCategory);
            category.setCategoryWeight(createModel.getBatchWeight());
            newRow.setSampleCategory(category);

            recomputeRowValidState(newRow);

            tableModel.addNewRow(newRow);
            AbstractSelectTableAction.doSelectCell(getTable(), tableModel.getRowCount() - 1, 0);

            saveRow(newRow);

            // update speciesUsed
            addToSpeciesUsed(newRow);
        }

        recomputeBatchActionEnable();
    }

    public void splitSpeciesBatch() {

        JXTable table = getTable();

        // get selected row
        int rowIndex = table.getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant split batch if no batch selected");

        SpeciesBatchTableModel tableModel = getTableModel();

        SpeciesBatchRowModel parentBatch = tableModel.getEntry(rowIndex);

        boolean split = true;
        if (parentBatch.getWeight() != null) {
            int i = JOptionPane.showConfirmDialog(
                    ui,
                    _("tutti.dialog.catches.species.split.weightNotNull.message"),
                    _("tutti.dialog.catches.species.split.weightNotNull.title"),
                    JOptionPane.OK_CANCEL_OPTION);

            if (i == JOptionPane.OK_OPTION) {
                parentBatch.setWeight(null);

            } else {
                split = false;
            }
        }

        if (split) {
            if (log.isInfoEnabled()) {
                log.info("Open split batch ui for row [" + rowIndex + ']');
            }

            EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
            SplitSpeciesBatchUI splitBatchEditor = parent.getSpeciesTabSplitBatch();

            splitBatchEditor.getHandler().editBatch(parentBatch);
            parent.getHandler().setSpeciesSelectedCard(EditCatchesUIHandler.SPLIT_BATCH_CARD);
        }
    }

    public void splitBatch(SplitSpeciesBatchUIModel splitModel) {
        if (splitModel.isValid()) {

            JXTable table = getTable();

            // get selected row
            int insertRow = table.getSelectedRow();

            SpeciesBatchTableModel tableModel = getTableModel();
            SpeciesBatchRowModel parentBatch = tableModel.getEntry(insertRow);

            // create batch rows

            SampleCategoryType selectedCategory = splitModel.getSelectedCategory();
            SampleCategoryEnum sampleCategoryEnum = selectedCategory.getType();

            // Create rows in batch table model

            List<SpeciesBatchRowModel> newBatches = Lists.newArrayList();
            for (SplitSpeciesBatchRowModel row : splitModel.getRows()) {
                if (row.isValid()) {

                    // can keep this row
                    SpeciesBatchRowModel newBatch = tableModel.createNewRow();

                    loadBatchRow(parentBatch,
                                 newBatch,
                                 sampleCategoryEnum,
                                 row.getCategoryValue(),
                                 row.getWeight());

                    recomputeRowValidState(newBatch);
                    newBatches.add(newBatch);

                    tableModel.addNewRow(++insertRow, newBatch);
                }
            }

            // add new batches to his parent
            parentBatch.setChildBatch(newBatches);

            //TODO Should only save parentBatch (will persist all his childs)
            //saveRow(parentBatch);

            // save new batches
            saveRows(newBatches);

            SpeciesBatchUIModel model = getModel();
            model.setLeafNumber(model.getLeafNumber() + newBatches.size() - 1);
        }

        recomputeBatchActionEnable();

//        // reselect this cell
//        AbstractSelectTableAction.doSelectCell(table, rowIndex, 0);
//        table.requestFocus();
    }

    public void updateTotalFromFrequencies(SpeciesBatchRowModel row) {
        List<SpeciesFrequencyRowModel> frequency = row.getFrequency();

        if (CollectionUtils.isNotEmpty(frequency)) {
            Integer totalNumber = 0;
            for (SpeciesFrequencyRowModel frequencyModel : frequency) {
                if (frequencyModel.getNumber() != null) {
                    totalNumber += frequencyModel.getNumber();
                }
            }
            row.setComputedNumber(totalNumber);
            row.getFinestCategory().setOnlyOneFrequency(frequency.size() == 1);
        }

    }

    public void saveRows(Iterable<SpeciesBatchRowModel> rows) {
        for (SpeciesBatchRowModel row : rows) {
            saveRow(row);
        }
    }

    protected void saveRow(SpeciesBatchRowModel row) {

        FishingOperation fishingOperation = getModel().getFishingOperation();
        Preconditions.checkNotNull(fishingOperation);

        Preconditions.checkNotNull(row.getSpecies());
        SampleCategory<?> sampleCategory = row.getSampleCategory();
        Preconditions.checkNotNull(sampleCategory);
        Preconditions.checkNotNull(sampleCategory.getCategoryType());
        Preconditions.checkNotNull(sampleCategory.getCategoryValue());

        SpeciesBatch catchBean = row.toBean();
        catchBean.setFishingOperation(fishingOperation);

        SpeciesBatchRowModel parent = row.getParentBatch();
        if (parent != null) {
            catchBean.setParentBatch(parent.toBean());
        }

        // apply sample category
        catchBean.setSampleCategoryType(sampleCategory.getCategoryType().getType());
        catchBean.setSampleCategoryValue(sampleCategory.getCategoryValue());
        catchBean.setSampleCategoryWeight(sampleCategory.getCategoryWeight());

        if (TuttiEntities.isNew(catchBean)) {

            SpeciesBatchRowModel batchParent = row.getParentBatch();
            String parentBatchId = null;

            if (batchParent != null) {
                parentBatchId = batchParent.getId();
            }

            if (log.isInfoEnabled()) {
                log.info("Persist new species batch with parentId: " +
                         parentBatchId);
            }
            catchBean = persistenceService.createSpeciesBatch(catchBean,
                                                              parentBatchId);
            row.setId(catchBean.getId());
        } else {
            if (log.isInfoEnabled()) {
                log.info("Persist existing species batch: " + catchBean.getId() + " (parent : " + catchBean.getParentBatch() + ")");
            }
            persistenceService.saveSpeciesBatch(catchBean);
        }

        List<SpeciesFrequencyRowModel> frequencyRows = row.getFrequency();

        List<SpeciesBatchFrequency> frequency =
                SpeciesFrequencyRowModel.toBeans(frequencyRows, catchBean);

        if (log.isInfoEnabled()) {
            log.info("Will save " + frequency.size() + " frequencies.");
        }
        frequency = persistenceService.saveSpeciesBatchFrequency(
                catchBean.getId(), frequency);

        // push it back to row model
        frequencyRows = SpeciesFrequencyRowModel.fromBeans(frequency);
        row.setFrequency(frequencyRows);
    }

    public String getFilterSpeciesBatchRootButtonText(int rootNumber) {
        return _("tutti.label.filterSpeciesBatchMode.mode.root", rootNumber);
    }

    protected void recomputeBatchActionEnable() {

        int rowIndex = getTable().getSelectedRow();

        //TODO Improve this test
        boolean enableAdd = true;
//                CollectionUtils.isNotEmpty(getModel().getAvailableSpecies());

        boolean enableRename = false;
        boolean enableSplit = false;
        boolean enableRemove = false;
        boolean enableRemoveSub = false;
        boolean enableCreateMelag = false;

        if (rowIndex != -1) {

            // there is a selected row

            //TODO If there is some sub-batch, can remove them
            //TODO If there is no sub-batch, can split current batch

            SpeciesBatchTableModel tableModel = getTableModel();
            SpeciesBatchRowModel row = tableModel.getEntry(rowIndex);
            int selectedRowCount = getTable().getSelectedRowCount();

            if (row.isValid()) {

                // must have at least species filled in row
                // otherwise nothing can be done

                enableSplit = true;
                enableRemove = true;
                enableRemoveSub = true;
                enableRename = true;
                enableCreateMelag = true;
            }

            if (enableSplit) {

                // can split if selected batch is a leaf
                enableSplit = row.isBatchLeaf()
                              && selectedRowCount == 1
                              && row.getComputedNumber() == null;
            }

            if (enableRename) {

                // can rename if selected batch is a parent
                enableRename = row.isBatchRoot()
                               && selectedRowCount == 1;
            }

            if (enableRemove) {

                // can always remove the batch
                enableRemove = selectedRowCount == 1;
            }

            if (enableRemoveSub) {

                // can remove sub batch if selected batch is not a leaf
                enableRemoveSub = !row.isBatchLeaf()
                                  && selectedRowCount == 1;
            }

            if (enableCreateMelag) {

                JXTable table = getTable();

                // can add species to a melag if several root are selected

                if (selectedRowCount < 2) {
                    enableCreateMelag = false;

                } else {
                    int[] selectedRows = table.getSelectedRows();
                    for (int selectedRowIndex : selectedRows) {
                        SpeciesBatchRowModel selectedRow =
                                tableModel.getEntry(selectedRowIndex);

                        if (!selectedRow.isBatchRoot()) {
                            enableCreateMelag = false;
                            break;
                        }
                    }
                }
            }
        }

        SpeciesBatchUIModel model = getModel();
        model.setCreateSpeciesBatchEnabled(enableAdd);
        model.setSplitSpeciesBatchEnabled(enableSplit);
        model.setRemoveSpeciesBatchEnabled(enableRemove);
        model.setRemoveSpeciesSubBatchEnabled(enableRemoveSub);
        model.setRenameSpeciesBatchEnabled(enableRename);
        model.setCreateMelagEnabled(enableCreateMelag);
    }

    public void collectChilds(SpeciesBatchRowModel row,
                              Set<SpeciesBatchRowModel> collectedRows) {

        if (!row.isBatchLeaf()) {

            for (SpeciesBatchRowModel batchChild : row.getChildBatch()) {
                collectedRows.add(batchChild);
                collectChilds(batchChild, collectedRows);
            }
        }
    }

    protected SpeciesBatchRowModel loadSpeciesBatch(SpeciesBatch aBatch,
                                                    SpeciesBatchRowModel parentRow,
                                                    List<SpeciesBatchRowModel> rows) {

        String id = aBatch.getId();

        List<SpeciesBatchFrequency> frequencies =
                persistenceService.getAllSpeciesBatchFrequency(id);

        List<Attachment> attachments =
                persistenceService.getAllAttachments(Integer.valueOf(id));

        SpeciesBatchRowModel newRow =
                new SpeciesBatchRowModel(aBatch, frequencies, attachments);

        SampleCategoryEnum sampleCategoryEnum =
                aBatch.getSampleCategoryType();

        Preconditions.checkNotNull(
                sampleCategoryEnum,
                "Can't have a batch with no sample category, but was: " + aBatch);

        loadBatchRow(parentRow,
                     newRow,
                     sampleCategoryEnum,
                     aBatch.getSampleCategoryValue(),
                     aBatch.getSampleCategoryWeight());

        rows.add(newRow);

        if (!aBatch.isChildBatchsEmpty()) {

            // create batch childs rows

            List<SpeciesBatchRowModel> batchChilds = Lists.
                    newArrayListWithCapacity(aBatch.sizeChildBatchs());

            for (SpeciesBatch childBatch : aBatch.getChildBatchs()) {
                SpeciesBatchRowModel childRow = loadSpeciesBatch(childBatch, newRow, rows);
                batchChilds.add(childRow);
            }
            newRow.setChildBatch(batchChilds);
        }

        return newRow;
    }

    protected void loadBatchRow(SpeciesBatchRowModel parentRow,
                                SpeciesBatchRowModel newRow,
                                SampleCategoryEnum sampleCategoryEnum,
                                Serializable categoryValue,
                                Float categoryWeight) {

        // get sample category from his type
        SampleCategory sampleCategory =
                newRow.getSampleCategory(sampleCategoryEnum);

        // fill it
        sampleCategory.setCategoryValue(categoryValue);
        sampleCategory.setCategoryWeight(categoryWeight);

        // push it back to row as his *main* sample category
        newRow.setSampleCategory(sampleCategory);

        if (parentRow != null) {

            // copy back parent data (mainly other sample categories)

            newRow.setSpecies(parentRow.getSpecies());
            newRow.setSpeciesToConfirm(parentRow.getSpeciesToConfirm());
            newRow.setParentBatch(parentRow);

            newRow.setSpecies(parentRow.getSpecies());
            if (sampleCategoryEnum != SampleCategoryEnum.sortedUnsorted) {
                newRow.setSortedUnsortedCategory(parentRow.getSortedUnsortedCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.size) {
                newRow.setSizeCategory(parentRow.getSizeCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.sex) {
                newRow.setSexCategory(parentRow.getSexCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.maturity) {
                newRow.setMaturityCategory(parentRow.getMaturityCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.age) {
                newRow.setAgeCategory(parentRow.getAgeCategory());
            }
        }
    }

    protected <C extends Serializable> void addSampleCategoryColumnToModel(TableColumnModel columnModel,
                                                                           ColumnIdentifier<SpeciesBatchRowModel> columnIdentifier,
                                                                           Decorator<C> decorator,
                                                                           TableCellRenderer defaultRenderer) {
        addColumnToModel(columnModel,
                         SampleCategoryComponent.newEditor(decorator),
                         SampleCategoryComponent.newRender(defaultRenderer,
                                                           decorator,
                                                           getConfig().getColorComputedWeights()),
                         columnIdentifier);
    }

    public void removeFromSpeciesUsed(SpeciesBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getSpecies());
        Preconditions.checkNotNull(row.getSortedUnsortedCategoryValue());
        if (log.isInfoEnabled()) {
            log.info("Remove from speciesUsed: " + decorate(row.getSortedUnsortedCategoryValue()) + " - " + decorate(row.getSpecies()));
        }
        SpeciesBatchUIModel model = getModel();
        model.getSpeciesUsed().remove(row.getSortedUnsortedCategoryValue(),
                                      row.getSpecies());

        if (row.isBatchRoot()) {
            model.setRootNumber(model.getRootNumber() - 1);
        }
    }

    protected void addToSpeciesUsed(SpeciesBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getSpecies());
        Preconditions.checkNotNull(row.getSortedUnsortedCategoryValue());
        if (log.isDebugEnabled()) {
            log.debug("Add to speciesUsed: " +
                      decorate(row.getSortedUnsortedCategoryValue()) +
                      " - " + decorate(row.getSpecies()));
        }
        SpeciesBatchUIModel model = getModel();
        model.getSpeciesUsed().put(row.getSortedUnsortedCategoryValue(),
                                   row.getSpecies());

        model.setRootNumber(model.getRootNumber() + 1);
    }

    public Species openAddSpeciesDialog(String title, List<Species> species) {
        SelectSpeciesUI dialogContent = new SelectSpeciesUI(ui);
        SelectSpeciesUIModel model = dialogContent.getModel();
        model.setSelectedSpecies(null);
        model.setSpecies(species);

        openDialog(dialogContent, title, new Dimension(400, 130));

        return model.getSelectedSpecies();
    }

}
