package fr.ifremer.tutti.ui.swing.content.operation;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiBatchTableUIHandler.java 77 2012-12-15 16:05:45Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/AbstractTuttiBatchTableUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.ezware.oxbow.swingbits.table.filter.DistinctColumnItem;
import com.ezware.oxbow.swingbits.table.filter.IFilterChangeListener;
import com.ezware.oxbow.swingbits.table.filter.ITableFilter;
import com.ezware.oxbow.swingbits.table.filter.JTableFilter;
import com.ezware.oxbow.swingbits.table.filter.TableRowFilterSupport;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableModel;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIModel;
import fr.ifremer.tutti.ui.swing.util.table.TableRowModificationListener;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.sort.TableSortController;

import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.FocusEvent;
import java.util.Collection;

/**
 * Abstract ui handler forbatch ui.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public abstract class AbstractTuttiBatchTableUIHandler<R extends AbstractTuttiBeanUIModel, M extends AbstractTuttiTableUIModel<?, R, M>> extends AbstractTuttiTableUIHandler<R, M> {


    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractTuttiBatchTableUIHandler.class);

    protected abstract void saveSelectedRowIfRequired();

    protected abstract boolean isRowValid(R row);

    protected abstract void saveRow(R row);

    protected abstract FishingOperation getFishingOperation();

    public abstract void selectFishingOperation(FishingOperation bean);

    /**
     * Persistence service.
     *
     * @since 0.2
     */
    protected final PersistenceService persistenceService;

    /**
     * To monitor changes on the fishing operation.
     *
     * @since 0.2
     */
    protected final TuttiBeanMonitor<M> fishingOperationMonitor;

    protected AbstractTuttiBatchTableUIHandler(FishingOperationsUI parentUi,
                                               String[] fishingProperties,
                                               String... properties) {
        super(parentUi.getHandler().getContext(), properties);
        this.persistenceService = context.getService(PersistenceService.class);
        this.fishingOperationMonitor = new TuttiBeanMonitor<M>(
                fishingProperties);
    }

    public final void saveSelectedRowIfRequired(FocusEvent event) {

        Component oppositeComponent = event.getOppositeComponent();

        JXTable parentContainer = null;

        if (oppositeComponent != null) {

            // check out if still on table
            parentContainer = SwingUtil.getParentContainer(
                    oppositeComponent, JXTable.class);
        }

        if (parentContainer == null) {

            // out of the table can save
            if (getFishingOperation() != null) {
                saveSelectedRowIfRequired();
            }
        }
    }

    protected void initBatchTable(JXTable table,
                                  TableColumnModel columnModel,
                                  AbstractTuttiTableModel<R> tableModel) {

        installTableKeyListener(columnModel, table);
        table.getTableHeader().setReorderingAllowed(false);

        JTableFilter tableFilter = new JTableFilter(table);
        TableRowFilterSupport.forFilter(tableFilter).searchable(true).useTableRenderers(true).actions(true).apply();

        tableFilter.addChangeListener(new IFilterChangeListener() {
            @Override
            public void filterChanged(ITableFilter<?> filter) {
                if (log.isInfoEnabled()) {
                    log.info("Filter changed " + filter);
                }
                TableModel model = filter.getTable().getModel();
                int columnCount = model.getColumnCount();
                for (int i = 0; i < columnCount; i++) {
                    if (filter.isFiltered(i)) {
                        Collection<DistinctColumnItem> filterState = filter.getFilterState(i);
                        if (log.isInfoEnabled()) {
                            log.info("Column " + i + " is filtered: " + filterState);
                        }
                    }
                }
            }
        });

        TableSortController<TableModel> sorter = new TableSortController<TableModel>(tableModel);
        sorter.setSortable(false);

        table.setRowSorter(sorter);

        Highlighter readOnlyHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                HighlightPredicate.READ_ONLY, Color.LIGHT_GRAY);

        table.addHighlighter(readOnlyHighlighter);
        Highlighter validHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(HighlightPredicate.EDITABLE, new HighlightPredicate() {
                    @Override
                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {

                        boolean result = false;
                        if (adapter.isEditable()) {
                            int rowIndex = adapter.convertRowIndexToModel(adapter.row);
                            R row = getTableModel().getEntry(rowIndex);
                            result = !row.isValid();
                        }
                        return result;
                    }
                }), Color.RED);
        table.addHighlighter(validHighlighter);

        // when model datas change let's propagate it table model
        listenRowsFromModel();

        // save when row chaged and was modified
        ListSelectionListener listener = new TableRowModificationListener<R>(
                tableModel, getRowMonitor()) {
            @Override
            protected void saveSelectedRow() {
                saveSelectedRowIfRequired();
            }
        };
        table.getSelectionModel().addListSelectionListener(listener);

        // always scroll to selected row
        SwingUtil.scrollToTableSelection(table);
    }

}
