/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font.freetype;

import java.io.IOException;
import java.io.InputStream;
import java.util.logging.Level;
import java.util.logging.Logger;

import de.intarsys.cwt.font.IFont;
import de.intarsys.cwt.freetype.CharMap;
import de.intarsys.cwt.freetype.Face;
import de.intarsys.cwt.freetype.Freetype;
import de.intarsys.cwt.freetype.Library;
import de.intarsys.pdf.content.CSException;
import de.intarsys.pdf.cos.COSRuntimeException;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontDescriptor;
import de.intarsys.pdf.font.PDFontTrueType;
import de.intarsys.pdf.font.PDFontType0;
import de.intarsys.pdf.font.PDFontType1;
import de.intarsys.pdf.font.PDFontType3;
import de.intarsys.pdf.platform.cwt.font.IPlatformFont;
import de.intarsys.pdf.platform.cwt.font.IPlatformFontFactory;
import de.intarsys.pdf.platform.cwt.font.NullPlatformFontFactory;
import de.intarsys.pdf.platform.cwt.font.PlatformFontException;
import de.intarsys.pdf.platform.cwt.font.PlatformFontTools;
import de.intarsys.pdf.platform.cwt.font.type3.Type3FontFactory;
import de.intarsys.tools.attribute.Attribute;
import de.intarsys.tools.attribute.IAttributeSupport;
import de.intarsys.tools.resourcetracker.ResourceTracker;
import de.intarsys.tools.stream.StreamTools;

public class FreetypeFontFactory implements IPlatformFontFactory {

	private static ResourceTracker tracker = new ResourceTracker(100) {
		@Override
		protected void basicDispose(Object resource) {
			((Face) resource).doneFace();
		}
	};

	private static final Logger Log = PACKAGE.Log;

	/** The freetype library object. */
	private static Library library;

	private static final Attribute ATTR_PLATFORMFONT = new Attribute(
			"platformFont"); //$NON-NLS-1$

	synchronized protected static Library getLibrary() {
		if (library == null) {
			library = Freetype.initFreeType();
		}
		return library;
	}

	protected static Face loadFace(byte[] fontdata, int index) {
		return getLibrary().newMemoryFace(fontdata, index);
	}

	static protected void registerPlatformFont(IAttributeSupport as,
			IPlatformFont object) {
		as.setAttribute(ATTR_PLATFORMFONT, object);
	}

	private Type3FontFactory type3FontFactory = new Type3FontFactory();

	private NullPlatformFontFactory nullFontFactory = new NullPlatformFontFactory();

	protected IPlatformFont basicCreate(PDFont font)
			throws PlatformFontException {
		if (font instanceof PDFontType3) {
			return type3FontFactory.createPlatformFont(font);
		}
		Face face = null;
		PDFontDescriptor pdDesc = null;
		try {
			pdDesc = font.getFontDescriptor();
		} catch (COSRuntimeException e) {
			// there are documents around that are invalid but should be
			// displayed
			// todo 1 @mit warning
		}
		if (pdDesc != null) {
			byte[] fontdata = null;
			fontdata = pdDesc.getFontFile();
			if (fontdata == null) {
				fontdata = pdDesc.getFontFile2();
			}
			if (fontdata == null) {
				fontdata = pdDesc.getFontFile3();
			}
			if (fontdata != null) {
				face = loadFace(fontdata, 0);
			}
		}
		if (face == null) {
			// no or invalid font data
			face = getExternalFont(font);
		}
		if (face == null) {
			// todo font replacement, error handling
			Log.log(Level.WARNING, "can't load font '" + font.getBaseFont()
					+ "'");
			return nullFontFactory.createPlatformFont(font);
		}
		selectCharacterMap(font, face);
		FreetypeFont result = new FreetypeFont(font, face);
		tracker.trackPhantom(font, face);
		return result;
	}

	/*
	 * this is synchronized as two threads (e.g. viewing and printing) may race
	 * for the platform font.
	 * 
	 * @see
	 * de.intarsys.pdf.platform.common.IPlatformFontFactory#createPlatformFont
	 * (de.intarsys.pdf.font.PDFont)
	 */
	synchronized public IPlatformFont createPlatformFont(PDFont font)
			throws PlatformFontException {
		getLibrary();
		IPlatformFont result = (IPlatformFont) font
				.getAttribute(ATTR_PLATFORMFONT);
		if (result == null) {
			result = basicCreate(font);
			registerPlatformFont(font, result);
		}
		return result;
	}

	protected Face getExternalFont(PDFont pdFont) throws CSException {
		IFont font = PlatformFontTools.getCWTFont(pdFont);
		if (font != null) {
			return loadFace(font);
		}
		return null;
	}

	protected Face loadFace(IFont font) {
		InputStream is = null;
		try {
			is = font.getFontProgram().getLocator().getInputStream();
			byte[] fontdata = StreamTools.toByteArray(is);
			return loadFace(fontdata, 0);
		} catch (IOException e) {
			StreamTools.close(is);
			return null;
		}
	}

	protected boolean selectCharacterMap(Face face, int platform, int encoding) {
		int num;

		if ((num = face.getNumCharMaps()) == 0) {
			return false;
		}
		for (int index = 0; index < num; index++) {
			CharMap map = face.getCharMap(index);
			if (map.getPlatformID() == platform) {
				if (encoding == -1 || map.getEncodingID() == encoding) {
					face.setCharMap(map);
					return true;
				}
			}
		}
		return false;
	}

	protected void selectCharacterMap(PDFont pdFont, Face face) {
		PDFontDescriptor fontDescriptor = null;
		try {
			fontDescriptor = pdFont.getFontDescriptor();
		} catch (RuntimeException e) {
			// better support invalid documents around...
		}
		// debuggin snippet
		// if ((face != null) && (face.getCharMaps() != null)) {
		// for (Iterator it = face.getCharMaps().iterator(); it.hasNext();) {
		// FTCharMap map = (FTCharMap) it.next();
		// String encoding = new String(
		// (byte[]) map.getEncoding().getValue()
		// );
		// // System.out.println(encoding);
		// }
		// }
		if (fontDescriptor == null) {
			if (!selectCharacterMap(face, 3, 1)) {
				selectCharacterMap(face, 1, 0);
			}
		} else {
			if (fontDescriptor.isSymbolic()) {
				if (pdFont instanceof PDFontType1) {
					selectCharacterMap(face, 7, -1);
				} else if (pdFont instanceof PDFontTrueType) {
					boolean selected = selectCharacterMap(face, 3, 0);
					if (!selected) {
						selectCharacterMap(face, 1, 0);
					}
				} else if (pdFont instanceof PDFontType0) {
					// no further mapping
				} else {
					selectCharacterMap(face, 1, 0);
				}
			} else {
				if (pdFont instanceof PDFontType1) {
					selectCharacterMap(face, 3, 1);
				} else if (pdFont instanceof PDFontTrueType) {
					if (!selectCharacterMap(face, 3, 1)) {
						selectCharacterMap(face, 1, 0);
					}
				} else if (pdFont instanceof PDFontType0) {
					// no further mapping
				} else {
					selectCharacterMap(face, 3, 1);
				}
			}
		}
	}
}
