/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font.freetype;

import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;

import de.intarsys.cwt.freetype.CharMap;
import de.intarsys.cwt.freetype.Face;
import de.intarsys.cwt.freetype.Freetype;
import de.intarsys.cwt.freetype.FreetypeException;
import de.intarsys.cwt.freetype.GlyphSlot;
import de.intarsys.cwt.freetype.nativec.FTEnum;
import de.intarsys.pdf.content.GraphicsState;
import de.intarsys.pdf.encoding.Encoding;
import de.intarsys.pdf.encoding.GlyphNameMap;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontDescriptor;
import de.intarsys.pdf.font.PDFontTrueType;
import de.intarsys.pdf.font.PDFontType0;
import de.intarsys.pdf.font.PDFontType1;
import de.intarsys.pdf.font.PDGlyphs;
import de.intarsys.pdf.platform.cwt.font.IPlatformFont;
import de.intarsys.pdf.platform.cwt.font.IPlatformGlyphs;
import de.intarsys.pdf.platform.cwt.font.PlatformFontException;
import de.intarsys.pdf.platform.cwt.rendering.ICSPlatformDevice;
import de.intarsys.tools.attribute.Attribute;

public class FreetypeFont implements IPlatformFont {

	private static final Attribute ATTR_PLATFORMGLYPHS = new Attribute(
			"platformGlyphs"); //$NON-NLS-1$

	protected static final Attribute ATTR_SCALE = new Attribute(
			"platformFontScale");

	protected static final Attribute ATTR_TRANSFORM = new Attribute(
			"platformFontTransform");

	final private PDFont font;

	final private Face face;

	private int highByte = -1;

	private double textSpaceFactor = 1;

	private PDGlyphs currentGlyphs = null;

	private boolean symbolic = false;

	protected FreetypeFont(PDFont font, Face face) {
		super();
		this.font = font;
		this.face = face;
		// the factor for converting from freetype to pdf text space
		textSpaceFactor = 1000.0 / getFace().getUnitsPerEM();
		PDFontDescriptor fontDescriptor = null;
		try {
			fontDescriptor = font.getFontDescriptor();
			symbolic = fontDescriptor.isSymbolic();
		} catch (Exception e) {
			// support invalid documents around
		}
	}

	protected IPlatformGlyphs basicCreate(PDGlyphs glyphs)
			throws PlatformFontException {
		int codepoint = glyphs.getCodepoint();
		if (codepoint < 0) {
			return null;
		}
		return new FreetypeGlyphs(this, glyphs);
	}

	synchronized public IPlatformGlyphs createPlatformGlyphs(PDGlyphs glyphs)
			throws PlatformFontException {
		IPlatformGlyphs result = (IPlatformGlyphs) glyphs
				.getAttribute(ATTR_PLATFORMGLYPHS);
		if (result == null) {
			result = basicCreate(glyphs);
			glyphs.setAttribute(ATTR_PLATFORMGLYPHS, result);
		}
		return result;
	}

	public int getAscent() {
		int result = getFace().getAscender();
		return (int) (result * getTextSpaceFactor());
	}

	public Rectangle2D getBBox() {
		Rectangle2D fontBox = getFace().getBBox();
		int x = (int) (fontBox.getMinX() * getTextSpaceFactor());
		int y = (int) (fontBox.getMinY() * getTextSpaceFactor());
		int width = (int) (fontBox.getWidth() * getTextSpaceFactor());
		int height = (int) (fontBox.getHeight() * getTextSpaceFactor());
		return new Rectangle2D.Float(x, y, width, height);
	}

	public int getCharIndex(char unicode) {
		CharMap oldMap = null;
		try {
			oldMap = face.getCharMap();
			face.selectCharMap(FTEnum.ENCODING_UNICODE);
			return face.getCharIndex(unicode);
		} catch (FreetypeException e) {
			face.setCharMap(oldMap);
			return unicode;
		}
	}

	public int getDescent() {
		int result = getFace().getDescender();
		return (int) (result * getTextSpaceFactor());
	}

	public Face getFace() {
		return face;
	}

	public int getHeight() {
		int result = getFace().getHeight();
		return (int) (result * getTextSpaceFactor());
	}

	public Object getImpl() {
		return getFace();
	}

	public PDFont getPDFont() {
		return font;
	}

	public double getTextSpaceFactor() {
		return textSpaceFactor;
	}

	public int getUnderlinePosition() {
		return face.getUnderlinePosition();
	}

	public int getUnderlineThickness() {
		return face.getUnderlineThickness();
	}

	public boolean isBoldStyle() {
		// todo derive value
		return false;
	}

	public boolean isItalicStyle() {
		// todo derive value
		return false;
	}

	public boolean isMonospaced() {
		// todo derive value
		return false;
	}

	public boolean isSymbolFont() {
		// todo derive value
		return false;
	}

	public boolean isSymbolic() {
		return symbolic;
	}

	protected GlyphSlot loadGlyphSlot(PDGlyphs glyphs)
			throws PlatformFontException {
		// check if this glyphs already loaded
		if (currentGlyphs == glyphs) {
			return face.getGlyphSlot();
		}
		currentGlyphs = glyphs;
		//
		// todo review this code cemetery
		//
		if (isSymbolic()) {
			if (font instanceof PDFontType1) {
				Encoding encoding = font.getEncoding();
				if (encoding.isFontSpecificEncoding()) {
					String glyphName = encoding.getDifferenceGlyphName(glyphs
							.getCodepoint());
					if (glyphName != null) {
						loadShapeFromName(face, glyphName);
					} else {
						loadShapeFromCode(face, glyphs.getCodepoint());
					}
				} else {
					String glyphName = encoding.getGlyphName(glyphs
							.getCodepoint());
					loadShapeFromName(face, glyphName);
				}
			} else if (font instanceof PDFontTrueType) {
				// pdf spec -> do not use encoding when rendering smybolic
				// truetype
				CharMap charMap = face.getCharMap();
				if (charMap == null || charMap.getPlatformID() == 1) {
					loadShapeFromCode(face, glyphs.getCodepoint());
				} else {
					if (highByte == -1) {
						highByte = face.getFirstChar() & 0xff00;
					}
					int code = highByte | glyphs.getCodepoint();
					loadShapeFromCode(face, code);
				}
			} else if (font instanceof PDFontType0) {
				int code = font.getEncoding().getDecoded(glyphs.getCodepoint());
				int index = ((PDFontType0) font).getGlyphIndex(code);
				loadShapeFromIndex(face, index);
			} else {
				loadShapeFromIndex(face, glyphs.getCodepoint());
			}
		} else {
			if (font instanceof PDFontType1) {
				String glyphName = font.getEncoding().getGlyphName(
						glyphs.getCodepoint());
				if (glyphName == GlyphNameMap.GLYPH_NOTDEF) {
					int code = font.getEncoding().getDecoded(
							glyphs.getCodepoint());
					loadShapeFromCode(face, code);
				} else {
					loadShapeFromName(face, glyphName);
				}
			} else if (font instanceof PDFontTrueType) {
				CharMap charMap = face.getCharMap();
				if (charMap != null && charMap.isMicrosoftUnicode()) {
					int code = font.getEncoding().getDecoded(
							glyphs.getCodepoint());
					loadShapeFromCode(face, code);
				} else {
					int code = font
							.getMacintoshRomanCode(glyphs.getCodepoint());
					loadShapeFromCode(face, code);
				}
			} else if (font instanceof PDFontType0) {
				int code = font.getEncoding().getDecoded(glyphs.getCodepoint());
				int index = ((PDFontType0) font).getGlyphIndex(code);
				loadShapeFromIndex(face, index);
			} else {
				loadShapeFromIndex(face, glyphs.getCodepoint());
			}
		}
		return face.getGlyphSlot();
	}

	protected void loadShapeFromCode(Face face, int code)
			throws PlatformFontException {
		try {
			face.loadChar(code, Freetype.LOAD_DEFAULT | Freetype.LOAD_NO_SCALE
					| Freetype.LOAD_LINEAR_DESIGN
					| Freetype.LOAD_IGNORE_TRANSFORM);
		} catch (FreetypeException e) {
			throw new PlatformFontException(e);
		}
	}

	/**
	 * Load a glyph using a direct index into the glyph collection of the font.
	 * 
	 * @param index
	 * @throws PlatformFontException
	 */
	protected void loadShapeFromIndex(Face face, int index)
			throws PlatformFontException {
		try {
			face.loadGlyph(index, Freetype.LOAD_DEFAULT
					| Freetype.LOAD_NO_SCALE | Freetype.LOAD_LINEAR_DESIGN
					| Freetype.LOAD_IGNORE_TRANSFORM);
		} catch (FreetypeException e) {
			throw new PlatformFontException(e);
		}
	}

	/**
	 * Load a glyph using a logical name of the glyph.
	 * <p>
	 * This is normally used for selecting glyphs in a symbolic font.
	 * 
	 * @param glyphName
	 */
	protected void loadShapeFromName(Face face, String glyphName) {
		int index = face.getNameIndex(glyphName);
		try {
			face.loadGlyph(index, Freetype.LOAD_DEFAULT
					| Freetype.LOAD_NO_SCALE | Freetype.LOAD_LINEAR_DESIGN
					| Freetype.LOAD_IGNORE_TRANSFORM);
		} catch (FreetypeException ex) {
			// don't care if this worked
		}
	}

	public void saveState(ICSPlatformDevice device, GraphicsState oldState,
			GraphicsState newState) {
		// 
	}

	public void textStateChanged(ICSPlatformDevice device) {
		GraphicsState graphicsState = device.getGraphicsState();
		float fontScale = (1f / getFace().getUnitsPerEM())
				* graphicsState.textState.fontSize;
		AffineTransform tempTransform = AffineTransform
				.getScaleInstance(
						fontScale
								* graphicsState.textState.derivedHorizontalScalingFactor,
						fontScale);
		graphicsState.setAttribute(ATTR_TRANSFORM, tempTransform);
	}

}
