/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.color.awt;

import java.awt.Color;

import de.intarsys.pdf.pd.PDAnnotationTools;
import de.intarsys.pdf.pd.PDColorSpace;
import de.intarsys.pdf.pd.PDWidgetAnnotation;

public class AwtColorTools {

	/**
	 * The border color of the annotation.
	 * 
	 * @param annotation
	 * @return The border color of the annotation.
	 */
	public static Color getBorderColor(PDWidgetAnnotation annotation) {
		return toColor(PDAnnotationTools.getBorderColor(annotation));
	}

	/**
	 * The border color RGB values for the annotation.
	 * 
	 * @param annotation
	 * @return The border color RGB values for the annotation.
	 */
	public static float[] getBorderColorRGB(PDWidgetAnnotation annotation) {
		return toRGB(PDAnnotationTools.getBorderColor(annotation));
	}

	/**
	 * The fill color for the annotation.
	 * 
	 * @param annotation
	 * @return The fill color for the annotation.
	 */
	public static Color getFillColor(PDWidgetAnnotation annotation) {
		return toColor(PDAnnotationTools.getFillColor(annotation));
	}

	/**
	 * The fill color RGB values for the annotation.
	 * 
	 * @param annotation
	 * @return The fill color RGB values for the annotation.
	 */
	public static float[] getFillColorRGB(PDWidgetAnnotation annotation) {
		return toRGB(PDAnnotationTools.getFillColor(annotation));
	}

	/**
	 * Transform the color values to an AWT color in RGB space.
	 * 
	 * @param color
	 * @return The transformed color
	 */
	public static Color toColor(float[] color) {
		if (color == null) {
			return null;
		}
		if (color.length == 1) {
			PDColorSpace pdColorSpace = PDColorSpace
					.getSingleton(PDColorSpace.CN_CS_DeviceGray);
			AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
					.createColorSpace(pdColorSpace);
			float[] rgb = awtColorSpace.getRGB(color);
			return new Color(rgb[0], rgb[1], rgb[2]);
		}
		if (color.length == 4) {
			PDColorSpace pdColorSpace = PDColorSpace
					.getSingleton(PDColorSpace.CN_CS_DeviceCMYK);
			AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
					.createColorSpace(pdColorSpace);
			float[] rgb = awtColorSpace.getRGB(color);
			return new Color(rgb[0], rgb[1], rgb[2]);
		}
		if (color.length == 3) {
			return new Color(color[0], color[1], color[2]);
		}
		return null;
	}

	/**
	 * Transform the color values to an AWT color in RGB space.
	 * 
	 * @param color
	 * @return The transformed color
	 */
	public static Color toColor(int[] color) {
		if (color == null) {
			return null;
		}
		if (color.length == 1) {
			PDColorSpace pdColorSpace = PDColorSpace
					.getSingleton(PDColorSpace.CN_CS_DeviceGray);
			AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
					.createColorSpace(pdColorSpace);
			float[] rgb = awtColorSpace.getRGB(new float[] { color[0] / 255f,
					color[1] / 255f, color[2] / 255f });
			return new Color(rgb[0], rgb[1], rgb[2]);
		}
		if (color.length == 4) {
			PDColorSpace pdColorSpace = PDColorSpace
					.getSingleton(PDColorSpace.CN_CS_DeviceCMYK);
			AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
					.createColorSpace(pdColorSpace);
			float[] rgb = awtColorSpace.getRGB(new float[] { color[0] / 255f,
					color[1] / 255f, color[2] / 255f });
			return new Color(rgb[0], rgb[1], rgb[2]);
		}
		if (color.length == 3) {
			return new Color(color[0], color[1], color[2]);
		}
		return null;
	}

	/**
	 * The color as an array of floats.
	 * 
	 * @param color
	 * @return The color as an array of floats.
	 */
	public static float[] toFloatArray(Color color) {
		if (color == null) {
			return null;
		}
		return color.getRGBColorComponents(null);
	}

	public static int[] toIntArray(Color color) {
		if (color == null) {
			return null;
		}
		return new int[] { color.getRed(), color.getGreen(), color.getBlue() };
	}

	/**
	 * Transform the color values to RGB values.
	 * 
	 * @param color
	 * @return The color values as RGB float values.
	 */
	public static float[] toRGB(float[] color) {
		if (color == null) {
			return null;
		}
		if (color.length == 1) {
			PDColorSpace pdColorSpace = PDColorSpace
					.getSingleton(PDColorSpace.CN_CS_DeviceGray);
			AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
					.createColorSpace(pdColorSpace);
			return awtColorSpace.getRGB(color);
		}
		if (color.length == 4) {
			PDColorSpace pdColorSpace = PDColorSpace
					.getSingleton(PDColorSpace.CN_CS_DeviceCMYK);
			AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
					.createColorSpace(pdColorSpace);
			return awtColorSpace.getRGB(color);
		}
		if (color.length == 3) {
			return color;
		}
		return null;
	}

	/**
	 * Transform the color values to RGB values.
	 * 
	 * @param color
	 * @return The color values as RGB values.
	 */
	public static int[] toRGB(int[] color) {
		if (color == null) {
			return null;
		}
		if (color.length == 1) {
			PDColorSpace pdColorSpace = PDColorSpace
					.getSingleton(PDColorSpace.CN_CS_DeviceGray);
			AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
					.createColorSpace(pdColorSpace);
			float[] rgb = awtColorSpace.getRGB(new float[] { color[0] / 255f,
					color[1] / 255f, color[2] / 255f });
			return new int[] { (int) (rgb[0] * 255 + 0.5),
					(int) (rgb[1] * 255 + 0.5), (int) (rgb[2] * 255 + 0.5) };
		}
		if (color.length == 4) {
			PDColorSpace pdColorSpace = PDColorSpace
					.getSingleton(PDColorSpace.CN_CS_DeviceCMYK);
			AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
					.createColorSpace(pdColorSpace);
			float[] rgb = awtColorSpace.getRGB(new float[] { color[0] / 255f,
					color[1] / 255f, color[2] / 255f });
			return new int[] { (int) (rgb[0] * 255 + 0.5),
					(int) (rgb[1] * 255 + 0.5), (int) (rgb[2] * 255 + 0.5) };
		}
		if (color.length == 3) {
			return color;
		}
		return null;
	}

}
